import os
import socket
import logging
import re

import pyroute2

from infra.netconfig.lib import status
from infra.ya_salt.lib import constants


class NetworkInfo(object):
    def __init__(self,
                 interface_name=None,
                 bandwidth=None,
                 bb_fqdn=None, fb_fqdn=None,
                 bb_prefix=None, fb_prefix=None,
                 bb_ipv6_addr=None, fb_ipv6_addr=None):
        self.interface_name = interface_name
        self.bandwidth = bandwidth
        self.fb_fqdn = fb_fqdn
        self.bb_fqdn = bb_fqdn
        self.bb_prefix = bb_prefix
        self.fb_prefix = fb_prefix
        self.bb_ipv6_addr = bb_ipv6_addr
        self.fb_ipv6_addr = fb_ipv6_addr


def _interfaces():
    names_re = re.compile(constants.NET_IFACE_RE)
    with pyroute2.IPRoute() as ipr:
        links = ipr.get_links()
    for l in links:
        if names_re.match(l.get_attr('IFLA_IFNAME')):
            yield l


def get_ya_netconfig_state():
    result = {}
    # shut up netconfig own logging
    logging.disable(logging.CRITICAL)
    for iface in _interfaces():
        iface_pb = status.make_iface_proto(iface) or None
        if iface_pb:
            if iface_pb.kind == 'physical' and iface_pb.group == 'backbone':
                result['bb_iface'] = iface_pb.name
                result['bb_ipv6_addr'] = iface_pb.ipv6_address
                continue
            if iface_pb.kind == 'vlan' and iface_pb.vlan_id == 688:
                result['bb_net'] = iface_pb.network
                continue
            if iface_pb.kind == 'vlan' and iface_pb.vlan_id == 788:
                result['fb_net'] = iface_pb.network
                continue
            if (iface_pb.kind == 'vlan'
                    and iface_pb.group == 'fastbone'
                    and iface_pb.vlan_id not in status.MTN_VLANS):
                result['fb_ipv6_addr'] = iface_pb.ipv6_address
                continue
    # re-enable logging after netconfig call
    logging.disable(logging.NOTSET)
    return result


def get_ya_netconfig_state_azure():
    return {
        'bb_iface': 'eth0',
        'bb_ipv6_addr': 'override me via /var/lib/netconfig/override.json',
        'bb_net': 'override me via /var/lib/netconfig/override.json',
        'fb_net': 'override me via /var/lib/netconfig/override.json',
        'fb_ipv6_addr': 'override me via /var/lib/netconfig/override.json',
    }


def get_bandwidth(interface, open_func=open):
    with open_func(os.path.join('/sys/class/net', interface, 'speed'), 'r') as f:
        return int(f.read().strip())


def get_network_info(azure=False):
    if azure:
        netconfig_state = get_ya_netconfig_state_azure()
    else:
        netconfig_state = get_ya_netconfig_state()

    bb_fqdn = socket.gethostname()
    fb_fqdn = 'fb-{}'.format(bb_fqdn)

    # get connected interface name
    interface = netconfig_state['bb_iface']

    bandwidth = get_bandwidth(interface)

    bb_prefix = netconfig_state['bb_net']
    fb_prefix = netconfig_state['fb_net']

    bb_ipv6_addr = netconfig_state['bb_ipv6_addr']
    fb_ipv6_addr = netconfig_state['fb_ipv6_addr']

    return NetworkInfo(
        interface_name=interface,
        bandwidth=bandwidth,
        bb_fqdn=bb_fqdn,
        fb_fqdn=fb_fqdn,
        bb_prefix=bb_prefix,
        fb_prefix=fb_prefix,
        bb_ipv6_addr=bb_ipv6_addr,
        fb_ipv6_addr=fb_ipv6_addr
    )
