import os


class NUMANodeInfo(object):
    def __init__(self, node, cpus, mem_total):
        self.node = node
        self.cpus = cpus
        self.mem_total = mem_total


class NUMAInfo(object):
    def __init__(self, nodes):
        self.nodes = nodes


def parse_range_list(lstr):
    rv = []
    for part in lstr.split(','):
        if '-' in part:
            start, stop = part.split('-')
            for i in range(int(start), int(stop) + 1):
                rv.append(str(i))
        else:
            rv.append(part)
    return rv


def has_numa():
    return os.path.exists('/sys/devices/system/node/online')


def get_online_nodes(open_func=open):
    with open_func('/sys/devices/system/node/online', 'r') as f:
        return parse_range_list(f.read(4096).strip())


def node_path(node):
    return os.path.join('/sys/devices/system/node', 'node{}'.format(node))


def get_node_cpu_list(node, open_func=open):
    with open_func(os.path.join(node_path(node), 'cpulist'), 'r') as f:
        return parse_range_list(f.read(4096).strip())


def get_node_mem_total(node, open_func=open):
    with open_func(os.path.join(node_path(node), 'meminfo'), 'r') as f:
        for line in f:
            # according to https://github.com/torvalds/linux/blob/79160a603bdb51916226caf4a6616cc4e1c58a58/drivers/base/node.c#L382
            # and https://github.com/numactl/numactl/blob/93867c59b0bb29470873a427dc7f06ebaf305221/libnuma.c#L764
            # format "Node N MemTotal: <val> kB" is an interface
            if 'MemTotal:' in line:
                _, val_unit = line.split(':')
                val_unit = val_unit.strip()
                val, _ = val_unit.split()
                return int(val) * 1024
    return None


def get_node_info(node, open_func=open):
    # assume that cpu list is always valid and has at least one value
    cpus = get_node_cpu_list(node, open_func)
    # assume that node always has memory value
    mem_total = get_node_mem_total(node, open_func)
    cpus = [int(c) for c in cpus]
    return NUMANodeInfo(int(node), cpus, mem_total)


def get_numa_info(open_func=open):
    if not has_numa():
        return NUMAInfo([])
    nodes = get_online_nodes(open_func)
    nodes_info = []
    for node in nodes:
        nodes_info.append(get_node_info(node, open_func))
    return NUMAInfo(nodes_info)
