"""
Reporter status YP
"""
from __future__ import absolute_import

import logging
import grpc
import random

from library.python import resource

from infra.ya_salt.proto import ya_salt_pb2
from infra.rtc.nodeinfo.proto_yp import node_tracker_pb2, node_tracker_pb2_grpc
from infra.rtc.nodeinfo.proto_yp import discovery_pb2, discovery_pb2_grpc
from infra.rtc.nodeinfo.lib import yp_util

log = logging.getLogger('reporter-yp')

# Set small timeout. We do not send large bodies in our request.
# Assume that 10sec will be ok.
# It takes a long time to execute nodeinfo,
# cause we retry all code on fail (also retries yp request multiple times).
DEFAULT_REQ_TIMEOUT = 10
CAPI_PEM_PATH = '/etc/certs/capi.pem'


def send_nodeinfo(hostname, walle_tags, dc, status):
    h = yp_util.infer_yp_for_host(walle_tags, dc)
    if not h:
        return 'cannot discover yp master'
    log.info('Reporting to YP {}'.format(h))
    y, err = Yp.with_discovery(hostname, h)
    if err is not None:
        return 'failed to initialize YP reporter: {}'.format(err)
    err = y.notify(status)
    if err is not None:
        return 'failed to report to YP: {}'.format(err)
    else:
        log.info('Successfully reported to YP')


class ChannelFactory(object):
    def get_channel(self, cred):
        """
        :type cred: grpc.ChannelCredentials
        """
        raise NotImplementedError


class StaticFactory(ChannelFactory):
    """
    Creates channel to provided host port.
    """

    def __init__(self, host, port):
        self.host = host
        self.port = port

    def get_channel(self, cred):
        try:
            return grpc.secure_channel('{}:{}'.format(self.host, self.port), cred), None
        except Exception as e:
            return None, str(e)


class DiscoveryFactory(ChannelFactory):
    """
    Uses service discovery to get channel to random master.
    """

    def __init__(self, host, port):
        self.host = host
        self.port = port

    def get_channel(self, cred):
        # First, create channel through provided endpoint. Typically L3-4 balancer.
        t = '{}:{}'.format(self.host, self.port)
        try:
            ch = grpc.secure_channel(t, cred)
        except Exception as e:
            return None, str(e)
        stub = discovery_pb2_grpc.DiscoveryServiceStub(ch)
        try:
            resp = stub.GetMasters(discovery_pb2.TReqGetMasters(),
                                   timeout=DEFAULT_REQ_TIMEOUT)
        except Exception as e:
            return None, 'failed to discover master via {}: {}'.format(t, e)
        masters = []
        for master in resp.master_infos:
            if master.alive:
                masters.append(str(master.fqdn))
            else:
                log.debug('Dead master found "{}"'.format(master.fqdn))
        if not masters:
            return None, 'failed to discover alive masters via {}'.format(t)
        master = random.Random().choice(masters)
        log.debug('Randomly selected {} as master'.format(master))
        t = '{}:{}'.format(master, self.port)
        try:
            return grpc.secure_channel(t, cred), None
        except Exception as e:
            return None, str(e)


class Yp(object):
    @classmethod
    def with_discovery(cls, node_id, host, port=8084):
        cred, err = cls.load_production_credentials()
        if err is not None:
            return None, err
        return cls(node_id, cred, DiscoveryFactory(host, port)), None

    @classmethod
    def with_static(cls, node_id, host, port=8084):
        cred, err = cls.load_production_credentials()
        if err is not None:
            return None, err
        return cls(node_id, cred, StaticFactory(host, port)), None

    def __init__(self, node_id, cred, ch_factory):
        self.node_id = node_id
        self.cred = cred
        self.ch_factory = ch_factory

    @staticmethod
    def load_production_credentials():
        root_ca = resource.find('/builtin/cacert')
        if not root_ca:
            return None, 'failed to load root CA from resources'
        try:
            host_pem = open(CAPI_PEM_PATH, 'rb').read()
        except Exception as e:
            return None, 'failed to read {}: {}'.format(CAPI_PEM_PATH, e)
        try:
            return Yp.make_credentials(root_ca, host_pem, host_pem), None
        except Exception as e:
            return None, str(e)

    @staticmethod
    def make_credentials(root_ca, certificate_chain, private_key):
        return grpc.ssl_channel_credentials(
            root_certificates=root_ca,
            private_key=private_key,
            certificate_chain=certificate_chain,
        )

    def notify(self, status):
        """
        Reports provided status to YP.
        """
        ch, err = self.ch_factory.get_channel(self.cred)
        if err:
            return err
        stub = node_tracker_pb2_grpc.NodeTrackerServiceStub(ch)
        # To lower burden and change rate - do not send all status
        # to YP as we do not use all fields.
        st = ya_salt_pb2.HostmanStatus()
        st.node_info.MergeFrom(status.node_info)
        req = node_tracker_pb2.TReqSetHostStatus()
        req.node_id = self.node_id
        req.status.Pack(st)
        try:
            stub.SetHostStatus(req, timeout=DEFAULT_REQ_TIMEOUT)
        except Exception as e:
            return str(e)
        return None
