import collections
import logging
import os
import random
import jinja2
import yp.client
import api.hq
import api.gencfg
import api.walle
from walle_api import client as walle_client
from infra.rtc.nodeinfo.lib import yp_util
from infra.ya_salt.proto import ya_salt_pb2

log = logging.getLogger(__name__)


HTML_TEMPLATE='''
<html>
<head>
<title>{{lookup_group}}</title>
</head>
<body>
{%- for r in results %}
    <p>
        <h3>{{r.fqdn}}</h3>
        <div>
            <h4>lookup_type: {{r.lookup_type}}</h4>
            <h4>lookup_group: {{r.lookup_group}}</h4>
            <div>
                <h5>root user keys</h5>
                {%- for k in r.status.node_info.sox_info.root_access %}
                    <div>{{k}}</div>
                {% endfor %}
            </div>
            <div>
                <h5>/etc/passwd</h5>
                {%- for p in r.status.node_info.sox_info.passwd %}
                    <div>{{p}}</div>
                {% endfor %}
            </div>
            <div>
                <h5>/etc/group</h5>
                {%- for g in r.status.node_info.sox_info.group %}
                    <div>{{g}}</div>
                {% endfor %}
            </div>
            <div>
                <h5>/etc/security/yandex-access.conf</h5>
                {%- for i in r.status.node_info.sox_info.yandex_access %}
                    <div>{{i}}</div>
                {% endfor %}
            </div>
            <div>
                <h5>/etc/security/yandex-access-custom.conf</h5>
                {%- for i in r.status.node_info.sox_info.yandex_access_custom %}
                    <div>{{i}}</div>
                {% endfor %}
            </div>
            <div>
                <h4>sudoers</h4>
                {%- for e in r.status.node_info.sox_info.nopasswd_sudoers %}
                    <h5>{{e.file}}</h5>
                    {% for m in e.matches %}
                        <div>{{m}}</div>
                    {% endfor %}
                {% endfor %}
            </div>
            <div><b>Last update:</b> {{r.status.node_info.sox_info.last_update.ToDatetime()}}</div>
        </div>
    </p>
{% endfor %}
</body>
</html>
'''


MULTI_HTML_TEMPLATE='''
<html>
<head>
<title>Results</title>
</head>
<body>
{%- for g, group_results in results.items() %}
    <h2>{{g}}</h2>
    {%- for r in group_results %}
        <p>
            <h3>{{r.fqdn}}</h3>
            <div>
                <h4>lookup_type: {{r.lookup_type}}</h4>
                <h4>lookup_group: {{r.lookup_group}}</h4>
                <div>
                    <h5>root user keys</h5>
                    {%- for k in r.status.node_info.sox_info.root_access %}
                        <div>{{k}}</div>
                    {% endfor %}
                </div>
                <div>
                    <h5>/etc/passwd</h5>
                    {%- for p in r.status.node_info.sox_info.passwd %}
                        <div>{{p}}</div>
                    {% endfor %}
                </div>
                <div>
                    <h5>/etc/group</h5>
                    {%- for g in r.status.node_info.sox_info.group %}
                        <div>{{g}}</div>
                    {% endfor %}
                </div>
                <div>
                    <h5>/etc/security/yandex-access.conf</h5>
                    {%- for i in r.status.node_info.sox_info.yandex_access %}
                        <div>{{i}}</div>
                    {% endfor %}
                </div>
                <div>
                    <h5>/etc/security/yandex-access-custom.conf</h5>
                    {%- for i in r.status.node_info.sox_info.yandex_access_custom %}
                        <div>{{i}}</div>
                    {% endfor %}
                </div>
                <div>
                    <h4>sudoers</h4>
                    {%- for e in r.status.node_info.sox_info.nopasswd_sudoers %}
                        <h5>{{e.file}}</h5>
                        {% for m in e.matches %}
                            <div>{{m}}</div>
                        {% endfor %}
                    {% endfor %}
                </div>
                <div><b>Last update:</b> {{r.status.node_info.sox_info.last_update.ToDatetime()}}</div>
            </div>
        </p>
    {% endfor %}
{% endfor %}
</body>
</html>
'''


HostResult = collections.namedtuple("HostResult", "fqdn,lookup_type,lookup_group,status")
LOOKUP_WALLE = 'walle_project'
LOOKUP_GENCFG = 'gencfg'
LOOKUP_NANNY = 'hq'
LOOKUP_NODE = 'node'


def host_tags_dc(fqdn):
    c = walle_client.WalleClient()
    h = c.get_host(fqdn, fields=['tags', 'location'], resolve_tags=True)
    return h['tags'], h['location']['short_datacenter_name']


def cluster_for_host(fqdn):
    tags, location = host_tags_dc(fqdn)
    master = yp_util.infer_yp_for_host(tags, location)
    return master.split('.')[0]


def group_results_by_lookup_group(results):
    rv = collections.defaultdict(list)
    for r in results:
        rv[r.lookup_group].append(r)
    return rv


def format_html_to_dir(output_dir, results):
    results_by_group = group_results_by_lookup_group(results)
    for g, results in results_by_group.items():
        output_path = os.path.join(output_dir, '{}.html'.format(g))
        if os.path.exists(output_path):
            log.fatal('File {} already exists!'.format(output_path))
            exit(1)
        with open(output_path, 'w+') as f:
            tmpl = jinja2.Template(HTML_TEMPLATE)
            f.write(tmpl.render(results=results, lookup_group=g))


def format_html_to_writer(writer, results):
    results_by_group = group_results_by_lookup_group(results)
    tmpl = jinja2.Template(MULTI_HTML_TEMPLATE)
    writer.write(tmpl.render(results=results_by_group))


def format_console(_, results):
    for r in results:
        print("============")
        print("fqdn: {}".format(r.fqdn))

        if r.lookup_type == LOOKUP_WALLE:
            print("WALL-E Project: {}".format(r.lookup_group))
        elif r.lookup_type == LOOKUP_NANNY:
            print("Nanny Service: {}".format(r.lookup_group))
        elif r.lookup_type == LOOKUP_GENCFG:
            print("GENCFG Group: {}".format(r.lookup_group))
        else:
            print("YP Cluster: {}".format(r.lookup_group))

        print("\nroot user keys:")
        for k in r.status.node_info.sox_info.root_access:
            print("{}".format(k))

        print("\n/etc/passwd:")
        for p in r.status.node_info.sox_info.passwd:
            print("{}".format(p))

        print("\n/etc/group:")
        for g in r.status.node_info.sox_info.group:
            print("{}".format(g))

        print("\n/etc/security/yandex-access.conf")
        for i in r.status.node_info.sox_info.yandex_access:
            print("{}".format(i))

        print("\n/etc/security/yandex-access-custom.conf")
        for i in r.status.node_info.sox_info.yandex_access_custom:
            print("{}".format(i))

        print("\n==sudoers==")
        for e in r.status.node_info.sox_info.nopasswd_sudoers:
            print("{}".format(e.file))
            for m in e.matches:
                print("{}".format(m))
            print("")

        print("last_update: {}\n".format(r.status.node_info.sox_info.last_update.ToDatetime()))

        print("============")


class YPTokenMissingError(Exception):
    pass


def yp_token(argv):
    token = argv.token if argv.token else os.getenv('YP_TOKEN')
    if not token:
        raise YPTokenMissingError('neither YP_TOKEN set nor passed via cmdline')
    return token


def lookup_node(token, cluster, node):
    try:
        log.info('Starting node {} lookup on cluster {}'.format(node, cluster))
        client = yp.client.YpClient(cluster, config={"token": token})
        result = client.select_objects("node", filter='[/meta/id]="{}"'.format(node), selectors=['/status/host_manager'])
        rv = []
        log.info('Got {} results from yp'.format(len(result)))
        for i in result:
            for k in i:
                status = ya_salt_pb2.HostmanStatus()
                status.MergeFromString(k['value'])
                rv.append(HostResult(node, LOOKUP_NODE, cluster, status))
        return rv, None
    except Exception as e:
        log.exception('Failed to do node {} lookup on cluster {}'.format(node, cluster))
        return None, 'failed to do node {} lookup on cluster {}: {}'.format(node, cluster, e)


def group_hosts_by_yp_cluster(hosts):
    rv = collections.defaultdict(list)
    for h in hosts:
        cluster = cluster_for_host(h)
        rv[cluster].append(h)
    return rv


def lookup_hosts_by_cluster(token, lookup_type, lookup_group, hosts_by_cluster):
    rv = []
    for cluster, hosts in hosts_by_cluster.items():
        log.debug("Looking cluster {} for hosts: {}".format(cluster, hosts))
        try:
            client = yp.client.YpClient(cluster, config={"token": token})
            result = client.select_objects("node",
                                           filter='[/meta/id] IN ({})'.format(
                                               ', '.join('"{}"'.format(h) for h in hosts)),
                                           selectors=['/meta/id', '/status/host_manager'])
            for i in result:
                fqdn, k = i
                status = ya_salt_pb2.HostmanStatus()
                status.MergeFromString(k['value'])
                rv.append(HostResult(fqdn, lookup_type, lookup_group, status))
        except Exception:
            log.exception('Failed to lookup cluster {} for hosts: {}'.format(cluster, hosts))
    return rv


def lookup_gencfg(token, gencfg_group, all_hosts=False):
    try:
        log.info('Starting gencfg group {} lookup'.format(gencfg_group))
        gencfg = api.gencfg.GenCfg()
        hosts = gencfg.get_hosts(gencfg_group)
        log.info('Got {} hosts in group {}'.format(len(hosts), gencfg_group))
        if not all_hosts:
            hosts = [random.choice(hosts)]
        log.debug("Hosts in group {} after filtering: {}".format(gencfg_group, hosts))
        hosts_by_cluster = group_hosts_by_yp_cluster(hosts)
        log.debug('Hosts in group {} by yp cluster: {}'.format(gencfg_group, hosts_by_cluster))
        return lookup_hosts_by_cluster(token, LOOKUP_GENCFG, gencfg_group, hosts_by_cluster), None
    except Exception as e:
        log.exception('Failed to do gencfg {} lookup'.format(gencfg_group))
        return None, 'failed to do gencfg {} lookup: {}'.format(gencfg_group, e)


def lookup_walle(token, walle_project, all_hosts=False):
    try:
        walle = api.walle.Walle()
        hosts = walle.get_hosts(walle_project)
        log.info('Got {} hosts in project {}'.format(len(hosts), walle_project))
        if not all_hosts:
            hosts = [random.choice(hosts)]
        log.debug("Hosts in project {} after filtering: {}".format(walle_project, hosts))
        hosts_by_cluster = group_hosts_by_yp_cluster(hosts)
        log.debug('Hosts in project {} by yp cluster: {}'.format(walle_project, hosts_by_cluster))
        return lookup_hosts_by_cluster(token, LOOKUP_WALLE, walle_project, hosts_by_cluster), None
    except Exception as e:
        log.exception('Failed to do walle project {} lookup'.format(walle_project))
        return None, 'failed to do walle project {} lookup: {}'.format(walle_project, e)


def lookup_nanny(token, nanny_service, all_hosts=False):
    try:
        hq = api.hq.HQResolver(use_service_resolve_policy=True)
        hosts = hq.get_hosts(nanny_service)
        log.info('Got {} hosts in nanny service {}'.format(len(hosts), nanny_service))
        if not all_hosts:
            hosts = [random.choice(hosts)]
        log.debug("Hosts in nanny service {} after filtering: {}".format(nanny_service, hosts))
        hosts_by_cluster = group_hosts_by_yp_cluster(hosts)
        log.debug('Hosts in nanny service {} by yp cluster: {}'.format(nanny_service, hosts_by_cluster))
        return lookup_hosts_by_cluster(token, LOOKUP_NANNY, nanny_service, hosts_by_cluster), None
    except Exception as e:
        log.exception('Failed to do nanny service {} lookup'.format(nanny_service))
        return None, 'failed to do nanny service {} lookup: {}'.format(nanny_service, e)
