import tornado.web
import tornado.gen
import tornado.httpserver
import tornado.ioloop
import tornado.netutil
import tornado.web
import tornado.log

from infra.rtc.nodeinfo.lib.sox import report


class RootHandler(tornado.web.RequestHandler):
    @tornado.gen.coroutine
    def get(self):
        self.add_header('Content-type', 'text/html')
        self.set_status(200)
        self.write('''\
<html>
<head>
<title>SOX Viewer</title>
</head>
<body>
    <h4>Lookup by host and cluster</h4>
    <div>
        <form action="/host" method="post">
            <label for="cluster">Cluster:</label>
            <input id="cluster" name="cluster" type="text/">
            <label for="host">Host:</label>
            <input type="text" id="host" name="host"/>
            <button type="submit">Lookup host</button>
        </form>
    </div>
    <h4>Lookup by gencfg group</h4>
    <div>
        <form action="/gencfg" method="post">
            <label for="group">Gencfg:</label>
            <input id="group" name="group" type="text"/>
            <button type="submit">Lookup group</button>
        </form>
    </div>
    <h4>Lookup by wall-e project</h4>
    <div>
        <form action="/walle" method="post">
            <label for="project">Wall-E project:</label>
            <input id="project" name="project" type="text"/>
            <button type="submit">Lookup project</button>
        </form>
    </div>
    <h4>Lookup by nanny service</h4>
    <div>
        <form action="/nanny" method="post">
            <label for="service">Nanny service:</label>
            <input id="service" name="service" type="text"/>
            <button type="submit">Lookup service</button>
        </form>
    </div>
</body>
</html>
''')
        raise tornado.gen.Return()


class LookupHandler(tornado.web.RequestHandler):
    def initialize(self, token):
        self.token = token

    def _render_results(self, results, err):
        if err is not None:
            self.set_status(500)
            self.write(err)
            raise tornado.gen.Return()
        if len(results) == 0:
            self.set_status(404)
            self.write("Nothing found for your request")
            raise tornado.gen.Return()
        report.format_html_to_writer(self, results)
        raise tornado.gen.Return()


class HostHandler(LookupHandler):
    @tornado.gen.coroutine
    def post(self):
        self.add_header('Content-type', 'text/html')
        host = self.get_body_argument("host").strip()
        cluster = self.get_body_argument("cluster").strip()
        if not cluster:
            cluster = report.cluster_for_host(host)
        results, err = report.lookup_node(self.token, cluster, host)
        self._render_results(results, err)


class GencfgHandler(LookupHandler):
    @tornado.gen.coroutine
    def post(self):
        self.add_header('Content-type', 'text/html')
        group = self.get_body_argument("group").strip()
        results, err = report.lookup_gencfg(self.token, group)
        self._render_results(results, err)


class WalleHandler(LookupHandler):
    @tornado.gen.coroutine
    def post(self):
        self.add_header('Content-type', 'text/html')
        project = self.get_body_argument("project").strip()
        results, err = report.lookup_walle(self.token, project)
        self._render_results(results, err)


class NannyHandler(LookupHandler):
    @tornado.gen.coroutine
    def post(self):
        self.add_header('Content-type', 'text/html')
        service = self.get_body_argument("service").strip()
        results, err = report.lookup_nanny(self.token, service)
        self._render_results(results, err)


def make_app(token):
    app = tornado.web.Application([
        (r'/', RootHandler),
        (r'/host', HostHandler, {'token': token}),
        (r'/gencfg', GencfgHandler, {'token': token}),
        (r'/walle', WalleHandler, {'token': token}),
        (r'/nanny', NannyHandler, {'token': token}),
    ])
    return app


def run(token, port):
    tornado.log.app_log.info("Starting sox_report on port {}".format(port))
    app = make_app(token)
    server = tornado.httpserver.HTTPServer(app)
    sockets = tornado.netutil.bind_sockets(port)
    server.add_sockets(sockets)
    try:
        tornado.ioloop.IOLoop.current().start()
    except KeyboardInterrupt:
        tornado.ioloop.IOLoop.current().stop()
        tornado.log.app_log.warning('keyboard interrupt. exiting')
        return 1
