from __future__ import unicode_literals

# Cannot use io.{String|Bytes}IO, traceback.print_stack(file=io) does not work =(
import cStringIO
import sys
import time
import traceback


def _now_ms(clock=time):
    return int(clock.time() * 1000)


class TraceStep(object):
    __slots__ = ['msg', 'step_time']

    def __init__(self, msg, step_time):
        self.msg = msg
        self.step_time = step_time

    def __str__(self):
        return "TraceStep(t={},msg='{}')".format(self.step_time, self.msg)


class Trace(object):
    """
    Simple trace object which can log if some steps took too long.
    Example usage:
    >>> t = Trace('nodeinfo')
    >>> t.step('numa')
    >>> t.step('sox')
    >>> t.log_if_long(20000)
    """

    def __init__(self, name, now_ms=_now_ms):
        self.name = name
        self.start_time = now_ms()
        self._now_ms = now_ms
        self.steps = None

    def step(self, msg):
        if self.steps is None:
            self.steps = []
        self.steps.append(TraceStep(msg, self._now_ms()))

    def log(self, with_traceback=False):
        end_time = self._now_ms()
        b = cStringIO.StringIO()
        b.write(b"Trace '{}' (started: {}ms):\n".format(self.name, self.start_time))
        last_step_time = self.start_time
        for step in self.steps or []:
            b.write(b"[{}] [{}] {}\n".format(
                step.step_time - self.start_time,  # Diff from start
                step.step_time - last_step_time,  # Diff from previous step
                step.msg,
            ))
            last_step_time = step.step_time
        b.write(b"[{}] [{}] END\n".format(
            end_time - self.start_time,
            end_time - last_step_time,
        ))
        if with_traceback:
            b.write(b'Traceback:\n')
            traceback.print_stack(file=b)
        sys.stderr.write(b.getvalue())

    def log_if_long(self, ms, with_traceback=False):
        if self._now_ms() - self.start_time > ms:
            self.log(with_traceback)

    def total_time(self):
        return self._now_ms() - self.start_time
