package main

import (
	"errors"
	"fmt"
	"log"
	"os"
	"strings"
	"time"

	"a.yandex-team.ru/infra/rtc/nodeinfo/soxaudit/internal/pbutil"
	"github.com/golang/protobuf/ptypes"
	"google.golang.org/protobuf/encoding/protojson"

	"a.yandex-team.ru/infra/rtc/nodeinfo/soxaudit/internal/keyutil"
	"a.yandex-team.ru/infra/rtc/nodeinfo/soxaudit/internal/shellutil"
	ya_salt_pb2 "a.yandex-team.ru/infra/ya_salt/proto"
)

const RunInterval = 1 * time.Hour
const SoxInfoPath = "/var/lib/ya-salt/soxinfo.pb"

func calcNextUpdate(lastUpdate time.Time) time.Time {
	if lastUpdate.IsZero() {
		return time.Now()
	}
	return lastUpdate.Add(RunInterval)
}

func gatherSoxInfo() (*ya_salt_pb2.SOXSecurityInfo, error) {
	soxInfo := &ya_salt_pb2.SOXSecurityInfo{}
	errs := make([]string, 0)
	rootUsers := keyutil.RootUsers()
	soxInfo.RootAccess = rootUsers
	nopasswd := shellutil.EGrepGlob("/etc/sudoers*", "NOPASSWD|ALL|ALL$")
	for k, v := range nopasswd {
		soxInfo.NopasswdSudoers = append(soxInfo.NopasswdSudoers, &ya_salt_pb2.SudoersEntry{
			File:    k,
			Matches: v,
		})
	}
	groups, err := shellutil.Cat("/etc/group")
	if err != nil {
		errs = append(errs, fmt.Sprintf("Failed to read /etc/group: %s", err))
	} else {
		soxInfo.Group = groups
	}

	passwd, err := shellutil.Cat("/etc/passwd")
	if err != nil {
		errs = append(errs, fmt.Sprintf("Failed to read /etc/passwd: %s", err))
	} else {
		soxInfo.Passwd = passwd
	}

	yandexAccess, err := shellutil.Cat("/etc/security/yandex-access.conf")
	if err != nil {
		errs = append(errs, fmt.Sprintf("Failed to read /etc/security/yandex-access.conf: %s", err))
	} else {
		soxInfo.YandexAccess = yandexAccess
	}

	yandexAccessCustom, err := shellutil.Cat("/etc/security/yandex-access-custom.conf")
	if err != nil {
		errs = append(errs, fmt.Sprintf("Failed to read /etc/security/yandex-access-custom.conf: %s", err))
	} else {
		soxInfo.YandexAccessCustom = yandexAccessCustom
	}

	if len(errs) > 0 {
		return nil, fmt.Errorf("failed to gather sox info: %s", strings.Join(errs, "; "))
	}
	return soxInfo, nil
}

func getSoxInfo() (*ya_salt_pb2.SOXSecurityInfo, error) {
	now := time.Now()
	soxInfo, err := pbutil.Load(SoxInfoPath)
	if errors.Is(err, os.ErrNotExist) || err == nil && soxInfo.NextUpdate.AsTime().Before(now) {
		soxInfo, err = gatherSoxInfo()
	}
	if err != nil {
		return nil, err
	}
	if soxInfo.NextUpdate.AsTime().After(now) {
		return soxInfo, nil
	}
	nextUpdate := calcNextUpdate(now)
	soxInfo.LastUpdate, err = ptypes.TimestampProto(now)
	if err != nil {
		return nil, err
	}
	soxInfo.NextUpdate, err = ptypes.TimestampProto(nextUpdate)
	if err != nil {
		return nil, err
	}
	err = pbutil.Save(SoxInfoPath, soxInfo)
	if err != nil {
		return nil, err
	}
	return soxInfo, nil
}

func main() {
	soxInfo, err := getSoxInfo()
	if err != nil {
		log.Fatalf("Failed to get sox info: %s", err)
	}
	content, err := protojson.Marshal(soxInfo)
	if err != nil {
		log.Fatalf("Failed to marshall SOXSecurityInfo object as JSON: %v", err)
	}
	content = append(content, '\n')
	_, err = os.Stdout.Write(content)
	if err != nil {
		log.Fatalf("Failed to write json to stdout: %v", err)
	}
}
