package keyutil

import (
	"bufio"
	"fmt"
	"log"
	"os"
	"path/filepath"
	"strings"
)

type Key string
type KeyUser []string
type KeyRegistry map[Key]KeyUser

const (
	cauthServerAdminsPath = "/var/cache/yandex-cauth-userd/current/serveradmins"
	userdKeysPath         = "/var/cache/yandex-cauth-userd/current/keys"
)

func NewWithUsers(users ...string) KeyRegistry {
	r := KeyRegistry{}
	for _, user := range users {
		err := r.FromUser(user)
		if err != nil {
			log.Printf("cannot load user %s to registry: %v", user, err)
		}
	}

	return r
}

func FromUserdRegistry() KeyRegistry {
	r := KeyRegistry{}
	err := r.keysFromUserd(userdKeysPath)
	if err != nil {
		log.Printf("cannot read keys from userd: %v", err)
	}

	return r
}

func RootRegistry() KeyRegistry {
	return NewWithUsers("root")
}

func UsersRegistry() (KeyRegistry, error) {
	matches, err := filepath.Glob("/home/*")
	if err != nil {
		return nil, fmt.Errorf("cannot load user list: %v", err)
	}
	users := KeyUser{}
	for _, match := range matches {
		users = append(users, match[6:])
	}
	return NewWithUsers(users...), nil
}

func GetServerAdmins(saFilePath string) (map[string]bool, error) {
	var allServerAdmins = make(map[string]bool)

	f, err := os.Open(saFilePath)
	if err != nil {
		return nil, fmt.Errorf("cannot read users from %s: %v", saFilePath, err)
	}
	defer f.Close()
	scanner := bufio.NewScanner(f)

	for scanner.Scan() {
		splitted := strings.Split(scanner.Text(), ":")
		name := splitted[0]
		if strings.HasPrefix(name, "#") {
			continue
		}
		if !allServerAdmins[name] {
			allServerAdmins[name] = true
		}
	}
	return allServerAdmins, nil
}

func (r *KeyRegistry) keysFromUserd(UdKeysPath string) error {
	f, err := os.Open(UdKeysPath)
	if err != nil {
		return fmt.Errorf("cannot read ssh keys from %s: %v", UdKeysPath, err)
	}
	defer f.Close()
	scanner := bufio.NewScanner(f)
	for scanner.Scan() {
		var name = ""
		var line = ""

		splitted := strings.Split(scanner.Text(), ":")
		if len(splitted) == 2 {
			name = splitted[0]
			line = splitted[1]
		}
		// assume key has form "ssh-TYP <key contents> [key comment]\n"
		if strings.HasPrefix(line, "ssh-") {
			parts := strings.SplitN(line, " ", 3)
			if len(parts) != 2 && len(parts) != 3 {
				continue
			}
			key := (Key)(parts[1])
			if _, ok := (*r)[key]; !ok {
				(*r)[key] = make(KeyUser, 0)
			}
			currentUsers := (*r)[key]
			(*r)[key] = append(currentUsers, name)
		}
	}
	err = scanner.Err()
	if err != nil {
		return err
	} else {
		return nil
	}
}

func (r *KeyRegistry) keysFromFile(name, path string) error {
	f, err := os.Open(path)
	if err != nil {
		return fmt.Errorf("cannot read authorized_keys from %s: %v", path, err)
	}
	defer f.Close()
	scanner := bufio.NewScanner(f)
	for scanner.Scan() {
		line := scanner.Text()
		// assume key has form "ssh-TYP <key contents> [key comment]\n"
		if strings.HasPrefix(line, "ssh-") {
			parts := strings.SplitN(line, " ", 3)
			if len(parts) != 2 && len(parts) != 3 {
				continue
			}
			key := (Key)(parts[1])
			if _, ok := (*r)[key]; !ok {
				(*r)[key] = make(KeyUser, 0)
			}
			currentUsers := (*r)[key]
			(*r)[key] = append(currentUsers, name)
		}
	}
	err = scanner.Err()
	if err != nil {
		return err
	} else {
		return nil
	}
}

func (r *KeyRegistry) FromUser(name string) error {
	var keysGlob string
	if name != "root" {
		keysGlob = fmt.Sprintf("/home/%s/.ssh/authorized*", name)
	} else {
		keysGlob = "/root/.ssh/authorized*"
	}
	matches, err := filepath.Glob(keysGlob)
	if err != nil {
		return fmt.Errorf("cannot find authorized_keys for user %s: %v", name, err)
	}
	if matches == nil {
		return nil
	}
	for _, path := range matches {
		err := r.keysFromFile(name, path)
		if err != nil {
			log.Printf("cannot read keys from file %s: %v", path, err)
		}
	}

	return nil
}

func RootUsers() []string {
	// keys from /root/authorized_keys2
	rootKeys := RootRegistry()
	// keys from /home/*/authorized_keys
	userKeys, err := UsersRegistry()
	if err != nil {
		log.Fatalf("Failed to load user keys: %v", err)
	}
	// keys from userd
	userKeysFromUserd := FromUserdRegistry()
	// all users from /var/cache/yandex-cauth-userd/current/serveradmins
	serverAdmins, err := GetServerAdmins(cauthServerAdminsPath)
	if err != nil {
		log.Fatalf("Failed to load server admins: %v", err)
	}

	var rootUsersSet = make(map[string]bool)

	// Check if authorized_keys2 contains keys from user's home directories
	for key := range rootKeys {
		if users, ok := userKeys[key]; ok {
			for _, user := range users {
				rootUsersSet[user] = true
			}
		}
	}

	// Check if authorized_keys2 contains keys from /var/cache/yandex-cauth-userd/current/keys
	for key := range rootKeys {
		if users, ok := userKeysFromUserd[key]; ok {
			for _, user := range users {
				rootUsersSet[user] = true
			}
		}
	}

	// Add all users from /var/cache/yandex-cauth-userd/current/serveradmins to rootUsers
	for user := range serverAdmins {
		rootUsersSet[user] = true
	}

	rootUsers := make([]string, 0, len(rootUsersSet))
	for ru := range rootUsersSet {
		rootUsers = append(rootUsers, ru)
	}

	return rootUsers
}
