package shellutil

import (
	"bufio"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"path/filepath"
	"regexp"
)

func EGrepGlob(glob, pattern string) map[string][]string {
	matches, err := filepath.Glob(glob)
	if err != nil {
		log.Fatalf("Cannot expand glob '%s': %v", glob, err)
	}
	rv := map[string][]string{}
	for _, match := range matches {
		files := expandDir(match)
		for _, file := range files {
			fileMatches, err := EGrepFile(file, pattern)
			if err != nil {
				log.Printf("Cannot egrep file %s: %v", match, err)
				continue
			}
			if len(fileMatches) > 0 {
				rv[file] = fileMatches
			}
		}
	}
	return rv
}

func EGrepFile(path, pattern string) ([]string, error) {
	re, err := regexp.Compile(pattern)
	if err != nil {
		return nil, fmt.Errorf("failed to compile re: '%s': %w", pattern, err)
	}
	f, err := os.Open(path)
	if err != nil {
		return nil, err
	}
	defer f.Close()
	info, _ := f.Stat()
	if info != nil && info.IsDir() {
		return nil, err
	}
	rv := make([]string, 0)
	scanner := bufio.NewScanner(f)
	for scanner.Scan() {
		line := scanner.Bytes()
		if re.Match(line) {
			rv = append(rv, string(line))
		}
	}
	return rv, nil
}

func expandDir(path string) []string {
	f, err := os.Open(path)
	if err != nil {
		log.Printf("%v", err)
		return nil
	}
	defer f.Close()
	info, err := f.Stat()
	if err != nil {
		log.Printf("%v", err)
		return nil
	}
	if info.IsDir() {
		rv := make([]string, 0)
		entries, err := ioutil.ReadDir(path)
		if err != nil {
			log.Printf("%v", err)
		}
		for _, ent := range entries {
			if !ent.IsDir() {
				rv = append(rv, fmt.Sprintf("%s/%s", path, ent.Name()))
			} else {
				rv = append(rv, expandDir(fmt.Sprintf("%s/%s", path, ent.Name()))...)
			}
		}
		return rv
	} else {
		return []string{path}
	}
}
