import logging

from infra.rtc.notifyctl.lib import PROVIDERS_MAP
from infra.rtc.notifyctl.lib.renderer import get_template
from infra.rtc.notifyctl.proto import notify_pb2

logger = logging.getLogger()


filtered_names = []
filtered_users = []


def factory(obj):
    stock = {}

    def inner(*args, **kwargs):
        new_obj = obj(*args, **kwargs)
        if new_obj not in stock:
            stock[new_obj] = new_obj
        return stock[new_obj]
    return inner


class FilteredNameError(Exception):
    def __init__(self, name):
        self.name = name


class Notification:
    proto_class = notify_pb2.Notification

    def __init__(self, notification_type, message, meta):
        self.notification_type = notification_type
        self.message = message
        self.meta = meta

    def serialize(self):
        proto_instance = self.proto_class()
        proto_instance.notification_type = self.notification_type
        proto_instance.message = self.message
        proto_instance.meta.update(self.meta)
        return proto_instance


@factory
class Login(object):
    proto_class = notify_pb2.Login

    def __init__(self, name):
        self.name = name
        self.instances = set()
        self.notifications = []

    def __eq__(self, other):
        if isinstance(other, self.__class__):
            return self.name == other.name
        return False

    def __hash__(self):
        return hash(self.name)

    def __str__(self):
        return '{login}, instances: {instances}'.format(login=self.name, instances=[str(i) for i in self.instances])

    def serialize(self):
        proto_instance = self.proto_class()
        proto_instance.name = self.name
        proto_instance.instances.extend([i.serialize() for i in self.instances])
        proto_instance.notifications.extend([i.serialize() for i in self.notifications])
        return proto_instance

    def add_instances(self, instances):
        self.instances.update(instances)


@factory
class Instance(object):
    proto_class = notify_pb2.Instance

    def __init__(self, name, provider):
        if any(filtered_name in name for filtered_name in filtered_names):
            raise FilteredNameError(name=name)
        self.name = name
        self.provider = provider
        self.hosts = set()
        self.meta = {}

    def __eq__(self, other):
        if isinstance(other, self.__class__):
            return self.name == other.name and self.provider == other.provider
        return False

    def __hash__(self):
        return hash((self.name, self.provider))

    def __str__(self):
        return '{instance} ({provider}) on {hosts}'.format(instance=self.name,
                                                           provider=self.provider,
                                                           hosts=[str(h) for h in self.hosts])

    def serialize(self):
        proto_instance = self.proto_class()
        proto_instance.name = self.name
        proto_instance.provider = self.provider
        proto_instance.hosts.extend(self.hosts)
        proto_instance.meta.update(self.meta)
        return proto_instance

    def add_host(self, host):
        self.hosts.add(host)

    def add_meta(self, meta):
        self.meta.update(meta)


class Logins:
    proto_class = notify_pb2.Logins

    def __init__(self, batch_id):
        self.logins = {}
        self.batch_id = batch_id

    def __iter__(self):
        return self.logins.values().__iter__()

    def add_login(self, login):
        self.logins.update({login: login})

    def render(self, notification_type, template, notification_meta):
        for login in self.logins.values():
            login.notifications.append(Notification(notification_type,
                                                    template.render(dataset=login, meta=notification_meta),
                                                    notification_meta))

    def serialize(self):
        self.proto_instance = self.proto_class()
        self.proto_instance.logins.extend([i.serialize() for i in self.logins.values()])
        self.proto_instance.batch_id = self.batch_id
        return self.proto_instance


def create_instance(data, instance_type):
    key_id = PROVIDERS_MAP[instance_type]['instance_key']
    for i in getattr(data, 'Instances', []):
        logger.debug('%s -> %s', i, key_id)
        if getattr(i, key_id, None):
            try:
                instance = Instance(getattr(i, key_id), instance_type)
                instance.add_host(i.Host)
                logger.debug('Create instance %s on %s', getattr(i, key_id), instance.hosts)
            except FilteredNameError as e:
                pass
                # logger.debug("Name filtered {name}".format(name=e.name))


def find_group_instances(group):
    group_instances = set()
    for provider, provider_properties in PROVIDERS_MAP.items():
        provider_key = provider_properties['responsible_key']
        if getattr(group, provider_key, None):
            try:
                instance = Instance(getattr(group, provider_key), provider)
                logger.debug('Responsible group %s %s (%s)', group, getattr(group, provider_key), provider)
                group_instances.add(instance)
            except FilteredNameError as e:
                pass
                # logger.debug("Name filtered {name}".format(name=e.name))
    return group_instances


def get_logins(data, logins):
    # create all instances for this dataset
    # we have to do it for provide some extra infromation f.i. about host
    for provider in PROVIDERS_MAP:
        create_instance(data, provider)

    # create or reuse Login and map instances
    for group in getattr(data, 'ResponsibleGroups', []):
        group_instances = find_group_instances(group)

        if group_instances:
            for l in getattr(group, 'Logins', []):
                if l not in filtered_users:
                    login = Login(l)
                    login.add_instances(group_instances)
                    logins.add_login(login)
    return logins


def gather_logins(data, batch_id, notification_providers, filtered_instances, filtered_logins):
    filtered_names.extend(filtered_instances)
    filtered_users.extend(filtered_logins)
    logins = Logins(batch_id)
    for d in data:
        logger.debug('-> %s', d)
        get_logins(d, logins)
    logger.debug('Logins are gathered')
    for notification_provider in notification_providers:
        logins.render(notification_provider.name,
                      get_template(notification_provider.template),
                      notification_provider.meta)
    logger.debug('Notifications are rendered')
    return logins
