import logging
import os
import time
from google.protobuf import json_format

from infra.rtc.notifyctl.lib import INSTANCE_RESOLVER_URL, PATH_FOR_PREPARED, PATH_FOR_BATCH, PROVIDERS_FILE_NAME, PATH_FOR_SENDED, \
    NOTIFICATION_PROVIDERS_MAP
from infra.rtc.notifyctl.lib.collector import gather_logins
from infra.rtc.notifyctl.lib.tvm import get_service_ticket
from infra.rtc.notifyctl.lib.utils import iter_struct, expand_hostnames, get_hosts_info, clear_path, store, split_batch
from infra.rtc.notifyctl.proto import notify_pb2

logger = logging.getLogger()


def prepare_app(spec_path, user=None):

    tmv_ticket = get_service_ticket(user=user)
    clear_path(PATH_FOR_BATCH.format(''))

    for script in iter_struct(spec_path, notify_pb2.Script):
        hostnames = expand_hostnames(script.spec)
        logger.debug('Hostlist expanded')
        urls = [INSTANCE_RESOLVER_URL.format(h) for h in hostnames]
        logins_message = gather_logins(data=get_hosts_info(urls, tmv_ticket),
                                       batch_id=str(int(time.time())),
                                       notification_providers=script.spec.notification_providers,
                                       filtered_instances=script.spec.filtered_instance_names,
                                       filtered_logins=script.spec.filtered_logins)
        notification_providers = [json_format.MessageToDict(provider)
                                  for provider in script.spec.notification_providers]
        store(notification_providers,
              batch_id=logins_message.batch_id,
              file_name_template=os.path.join(PATH_FOR_BATCH, PROVIDERS_FILE_NAME))

        for login_message in logins_message:
            login = json_format.MessageToDict(login_message.serialize())
            store(login,
                  batch_id=logins_message.batch_id,
                  file_name_template=os.path.join(PATH_FOR_PREPARED, login_message.name))

        logger.info("Batch id # %s prepared", logins_message.batch_id)


def send_app(batch_id):
    import threading

    thread_count = 5

    new_message_path = PATH_FOR_PREPARED.format(batch_id)
    stored_message_path = PATH_FOR_SENDED.format(batch_id)
    providers_file_name = PROVIDERS_FILE_NAME.format(batch_id)
    flag = True

    os.makedirs(stored_message_path, exist_ok=True)
    splitted = split_batch(data=os.listdir(new_message_path), splits=thread_count)

    for notification_provider in iter_struct(providers_file_name, notify_pb2.NotificationProvider):
        threads = []
        target = NOTIFICATION_PROVIDERS_MAP.get(notification_provider.name)
        try:
            if not target:
                raise Exception('Notification provider %s not found', notification_provider.name)
            for i in splitted.values():
                t = threading.Thread(target=target,
                                     args=[i, new_message_path,
                                           stored_message_path,
                                           notification_provider.name,
                                           lambda: flag],
                                     kwargs=notification_provider.meta)
                t.start()
                threads.append(t)
            for t in threads:
                t.join()
        except KeyboardInterrupt:
            flag = False
            logger.warning('User interruption')
            exit(1)

    logger.info('Batch id# %s successfully sent', batch_id)
