import asyncio
import logging
import os
import smtplib
from email.mime.text import MIMEText

import aiohttp
import yaml
from aiohttp import ClientSession, ClientResponseError, ServerDisconnectedError
from google.protobuf import json_format

from library.sky.hostresolver.resolver import Resolver
from infra.rtc.notifyctl.lib import NOTIFICATION_PROVIDERS_MAP, REQUESTS_LIMIT
from infra.rtc.notifyctl.proto import notify_pb2
from infra.rtc.instance_resolver.api import resolver_pb2


logger = logging.getLogger()


def register_notification_provider(alias):

    def middle(func):
        NOTIFICATION_PROVIDERS_MAP[alias] = func

        def inner(*args, **kwargs):
            return func(*args, **kwargs)
        return inner
    return middle


def iter_struct(path, pb_class):
    with open(path) as f:
        for d in yaml.load_all(f, Loader=yaml.SafeLoader):
            m = pb_class()

            json_format.ParseDict(d, m)
            yield m


def hostnames_sky(spec):
    data = list(Resolver().resolveHosts(spec.select.sky))
    return data


def hostnames_list(spec):
    return spec.select.hosts.list


def expand_hostnames(spec):
    res = set()
    if getattr(spec.select, 'sky', None):
        res.update(hostnames_sky(spec))
    if getattr(spec.select, 'hosts', None):
        res.update(hostnames_list(spec))
    return res


def get_hosts_info(urls, tvm_ticket):
    return (json_format.ParseDict(i, resolver_pb2.TInstancesReply(),) for i in get_host_info(urls, tvm_ticket))


def clear_path(path):
    import shutil

    try:
        shutil.rmtree(path, ignore_errors=True)
    except OSError:
        pass


def store(data, batch_id, file_name_template):
    file_path = file_name_template.format(batch_id)
    os.makedirs(os.path.dirname(file_path), exist_ok=True)
    with open(file_path, 'w') as _file:
        if isinstance(data, list):
            yaml.dump_all(data, _file, allow_unicode=True, explicit_start=True)
        else:
            yaml.dump(data, _file, allow_unicode=True, explicit_start=True)
    return file_path


def split_batch(data, splits):
    res = {}
    for i in range(splits):
        res[i] = [j for j in data if hash(j) % splits == i]
    return res


def merge_hostnames(host_list, useGrouping=True, addDomain=None, yrFormat=True):
    from .format import formatHosts
    return formatHosts(host_list, useGrouping=useGrouping, addDomain=addDomain, yrFormat=yrFormat, separator='|').split('|')


async def fetch(url, session):

    try:
        async with session.get(url, raise_for_status=True) as response:
            return await response.json()
    except ClientResponseError as e:
        logger.warning('Unable to fetch url {url}, '
                       'status: {status}, '
                       'message: {message}'.format(url=url,
                                                   status=e.status,
                                                   message=e.message))
    except ServerDisconnectedError as e:
        logger.warning('Unable to fetch url {url}, '
                       'message: {message}'.format(url=url,
                                                   message=e.message))


async def limited_fetch(sem, url, session):
    async with sem:
        return await fetch(url, session)


async def gather(portion, tvm_ticket):
    sem = asyncio.Semaphore(REQUESTS_LIMIT)
    tasks = []
    async with ClientSession(headers={"X-Ya-Service-Ticket": tvm_ticket},
                             connector=aiohttp.TCPConnector(verify_ssl=False)) as session:
        for url in portion:
            task = asyncio.ensure_future(limited_fetch(sem, url, session))
            tasks.append(task)
        return await asyncio.gather(*tasks)


def get_host_info(portion, tvm_ticket):
    loop = asyncio.get_event_loop()
    return list(filter(None, loop.run_until_complete(gather(portion, tvm_ticket))))


def send_smtp_sync(login, session, notification, **kwargs):
    message = MIMEText(notification.message, 'html')
    message['From'] = kwargs.get('from', 'yakernel@yandex-team.ru')
    message['To'] = '{}@yandex-team.ru'.format(login.name)
    message['Subject'] = kwargs.get('subj', 'Kernel update on your instances')

    logger.debug("Message from: {_from} "
                 "to: {_to} "
                 "subject: {_subj} sent".format(_from=kwargs.get('from', 'yakernel@yandex-team.ru'),
                                                  _to='{}@yandex-team.ru'.format(login.name),
                                                  _subj=kwargs.get('subj', 'Kernel update on your instances')))
    return session.send_message(msg=message)


@register_notification_provider('email_generic')
def email_generic(portion, new_message_path, stored_message_path, notification_provider, is_running, **kwargs):
    while portion and is_running():
        try:
            with smtplib.SMTP(host=kwargs['host'], port=kwargs['port']) as smtp_session:
                while portion:
                    if not is_running():
                        return
                    try:
                        file = portion.pop()
                        process_path = os.path.join(new_message_path, file)
                        stored_path = os.path.join(stored_message_path, file)
                        for login in iter_struct(process_path, notify_pb2.Login):
                            notifications = list(filter(
                                lambda x: getattr(x, 'notification_type', '') == notification_provider,
                                login.notifications))
                            if notifications:
                                send_smtp_sync(login=login,
                                               session=smtp_session,
                                               notification=notifications.pop(),
                                               **kwargs)
                        os.rename(process_path, stored_path)
                    except Exception as e:
                        portion.append(file)
                        logger.warning(e)
                        raise e
        except Exception as e:
            logger.warning(e)
