#! /usr/bin/env python2
# -*- coding: utf-8 -*-

# Provides: hbf_drops

import argparse
import time

from . import lib


CHECK_NAME = "hbf_drops"

DEFAULT_HISTORY_FILE = "/var/tmp/hbf-monitoring-drops-history.tmp"

DEFAULT_LIMITS_CONFIG_FILE = "/etc/yandex-hbf-agent/drops-monitoring-config.yaml"

DEFAULT_MAX_HISTORY_RECORDS = 12

DEFAULT_NEW_PROBES_NUM = 3


current_state = lib.make_state()

history_states = []

states_stats_parameters = {
    "has_stats": False,
    "has_newest_stats": False,
    "drops_per_second": 0.0,
    "summ_packets_total": 0,
    "summ_packets_dropped": 0,
    "max_packets_dropped": 0,
    "dropped_total_ratio": 0.0,
    "ratio_average_increase": 0.0
}

states_stats = {
    "period_secs": None,
    "v4": {
        "input": dict(states_stats_parameters),
        "output": dict(states_stats_parameters),
    },
    "v6": {
        "input": dict(states_stats_parameters),
        "output": dict(states_stats_parameters)
    }
}

limits_parameters = {
    "drops_per_second": {
        "in_use": False,
        "WARN": 0.0,
        "CRIT": 0.0
    },
    "summ_packets_dropped": {
        "in_use": False,
        "WARN": 0.0,
        "CRIT": 0.0
    },
    "max_packets_dropped": {
        "in_use": False,
        "WARN": 0.0,
        "CRIT": 0.0
    },
    "dropped_total_ratio": {
        "in_use": False,
        "WARN": 0.0,
        "CRIT": 0.0
    },
    "ratio_average_increase": {
        "in_use": False,
        "WARN": 0.0,
        "CRIT": 0.0
    }
}

limits = {
    "v4": {
        "input": dict(limits_parameters),
        "output": dict(limits_parameters)
    },
    "v6": {
        "input": dict(limits_parameters),
        "output": dict(limits_parameters)
    }
}


def parse_arguments():
    parser = argparse.ArgumentParser(description='HBF drops monitoring tool.')
    parser.add_argument('-c', '--limits-config-file',
                        help="Path to config file with limits",
                        dest='LIMITS_CONFIG_FILE',
                        default=DEFAULT_LIMITS_CONFIG_FILE)
    parser.add_argument('-p', '--history-probes-file',
                        help="Path to previous probes file",
                        dest='HISTORY_FILE',
                        default=DEFAULT_HISTORY_FILE)
    parser.add_argument('-l', '--history-len',
                        help="Number of items stored in history",
                        type=int,
                        dest='MAX_HISTORY_RECORDS',
                        default=DEFAULT_MAX_HISTORY_RECORDS)
    parser.add_argument('-r', '--ratio-burst-new-num',
                        help="Number of items which are treated as 'new'",
                        type=int,
                        dest='NEW_PROBES_NUM',
                        default=DEFAULT_NEW_PROBES_NUM)
    parser.add_argument('-d', '--debug',
                        help="Print probes analysis info",
                        dest='DEBUG',
                        action='store_true')
    parser.add_argument('-m', '--manual-counters',
                        help="Provide counters from command line, see below",
                        dest='MANUAL',
                        action='store_true')
    parser.add_argument('-o', '--off-no-data-alarm',
                        help="No data is OK. CRIT by default.",
                        dest='NO_DATA_IS_FINE',
                        action='store_true')
    parser.add_argument('-z', '--prelog-hook-zero-drop',
                        help="deprecated. No action",
                        dest='PRELOG_HOOK_ZERO_DROP',
                        action='store_true')
    parser.add_argument('--v4-input', metavar=('TOTAL', 'DROPPED'), nargs=2, type=int, default=[None, None])
    parser.add_argument('--v4-output', metavar=('TOTAL', 'DROPPED'), nargs=2, type=int, default=[None, None])
    parser.add_argument('--v6-input', metavar=('TOTAL', 'DROPPED'), nargs=2, type=int, default=[None, None])
    parser.add_argument('--v6-output', metavar=('TOTAL', 'DROPPED'), nargs=2, type=int, default=[None, None])
    args = parser.parse_args()
    return args


def load_current_state(state):
    """
        Updates current state dictionary with counters values from CLI args
        Input:  state dictionary
        Output: Updated state dictionary
    """
    state["timestamp"] = int(time.time())
    for ip_v in ("v4", "v6"):
        for io in lib.INOUT:
            arg_name = ip_v + '_' + io
            (total, drop) = getattr(args, arg_name)
            state[ip_v][io]["packets_total"] = total
            state[ip_v][io]["packets_dropped"] = drop
    return state


def update_history_states_with_current(history_states, current_state):
    """
        Updates history state dictionary with new current state.
        Records are sorted, oldest is removed. Respects MAX_HISTORY_RECORDS.
        Input:  history_states and current_state dictionaries
        Output: Updated history_states dictionary
    """
    # Sort history by timestamp
    history_states = sorted(history_states, key=lambda k: k['timestamp'])

    # Remove oldest probes if there are too many of them
    history_len = len(history_states)
    if history_len > args.MAX_HISTORY_RECORDS:
        for _ in range(0, history_len - args.MAX_HISTORY_RECORDS):
            history_states.pop(0)

    history_len = len(history_states)

    # Optionally pop the oldest probe, then add currently collected probe to the list
    if history_len == args.MAX_HISTORY_RECORDS:
        history_states.pop(0)
    history_states.append(current_state)

    return history_states


def count_average(data):
    """
        Calculates average value of series.
    """
    return sum(data) / float(len(data))


def count_ratio_average_increase(data):
    """
        Calculates average value of "newest" N probes / average value of "old" M probes
        M must be >= N.
    """
    if len(data) < args.NEW_PROBES_NUM * 2:
        return 0.0
    history_average = count_average(data[0:-args.NEW_PROBES_NUM])
    current_average = count_average(data[-args.NEW_PROBES_NUM:])
    if history_average == 0:
        return 0.0
    elif current_average == 0:
        return 0.0
    else:
        return current_average / float(history_average)


def analyze_history(history, stats):
    """
        Fills in states_stats dictionary based on history states.
    """
    # History data is sorted by timestamp
    history_len = len(history)
    stats["period_secs"] = history[history_len - 1]["timestamp"] - history[0]["timestamp"]

    # Cycle through all types of probes
    for ip_v in ("v4", "v6"):
        for io in lib.INOUT:

            # Cycle through all probes and
            # define if they have at least one meaningful number of drops,
            # if not - we are not interested in them
            stats[ip_v][io]["has_stats"] = False
            for i in range(0, history_len):
                if history[i][ip_v][io]["packets_dropped"] is not None:
                    stats[ip_v][io]["has_stats"] = True
            if not stats[ip_v][io]["has_stats"]:
                # print "SKIPPING", ip_v, io, "BECAUSE OF NO DROP VALUES IN ALL PROBES"
                continue

            # If there are all needed items in history files, cycle through newest probes and
            # define if they have at least one meaningful number of drops
            stats[ip_v][io]["has_newest_stats"] = False
            if history_len == args.MAX_HISTORY_RECORDS:
                for i in range(history_len - args.NEW_PROBES_NUM, history_len):
                    if history[i][ip_v][io]["packets_dropped"] is not None:
                        stats[ip_v][io]["has_newest_stats"] = True

            drops_ratios = []
            previous_total = None
            previous_dropped = None

            # Cycle through all probes and fill the stats
            for i in range(0, history_len):
                packets_total = history[i][ip_v][io]["packets_total"]
                packets_dropped = history[i][ip_v][io]["packets_dropped"]

                # Skip probes that do not have meaningful numbers
                if packets_total is None or packets_dropped is None:
                    continue

                # Fill initial values and continue
                if previous_total is None and previous_dropped is None:
                    previous_total = packets_total
                    previous_dropped = packets_dropped
                    continue

                diff_total = 0
                # Increment total summ
                if previous_total <= packets_total:
                    # Counter was not reset between probes
                    diff_total = packets_total - previous_total
                    previous_total = packets_total
                else:
                    # Counter was reset between probes
                    diff_total = packets_total
                    previous_total = packets_total
                stats[ip_v][io]["summ_packets_total"] += diff_total

                # Increment dropped summ
                diff_dropped = 0
                if previous_dropped <= packets_dropped:
                    # Counter was not reset between probes
                    diff_dropped = packets_dropped - previous_dropped
                    previous_dropped = packets_dropped
                else:
                    # Counter was reset between probes
                    diff_dropped = packets_dropped
                    previous_dropped = packets_dropped
                stats[ip_v][io]["summ_packets_dropped"] += diff_dropped

                # Save max drops value of all probes
                if diff_dropped > stats[ip_v][io]["max_packets_dropped"]:
                    stats[ip_v][io]["max_packets_dropped"] = diff_dropped

                # Fill the array of drops ratios
                if diff_total > 0:
                    drops_ratios.append(float(diff_dropped) / float(diff_total))

            # Calculate ratio between sums of drops and sums of total packages
            if stats[ip_v][io]["summ_packets_total"] > 0:
                stats[ip_v][io]["dropped_total_ratio"] = stats[ip_v][io]["summ_packets_dropped"] / float(stats[ip_v][io]["summ_packets_total"])
                # print "dropped_total_ratio_percent = %0.8f" % stats[ip_v][io]["dropped_total_ratio_percent"]

            # Calculate burst of drops: current ratio - average ratio on previous ratios
            if len(drops_ratios) > 1:
                stats[ip_v][io]["ratio_average_increase"] = count_ratio_average_increase(drops_ratios)
                # print "ratio_burst = %0.2f" % stats[ip_v][io]["ratio_burst"]
            if stats["period_secs"] > 0:
                stats[ip_v][io]["drops_per_second"] = stats[ip_v][io]["summ_packets_dropped"] / float(stats["period_secs"])
    return stats


def load_config(limits, config):
    """
        Merge values from monitiring's config JSON into default 'all-disabled' JSON.\
    """
    for ip_v in ("v4", "v6"):
        for io in lib.INOUT:
            for limit_parameter in limits_parameters.keys():
                try:
                    limits[ip_v][io][limit_parameter] = dict(config[ip_v][io][limit_parameter])
                except KeyError:
                    pass
    return limits


def test_limits(states_stats, limits, juggler_result):
    """
        Tests gathered statistics against limits.
    """
    for ip_v in ("v4", "v6"):
        for io in lib.INOUT:
            for limit_parameter in limits_parameters.keys():
                if limits[ip_v][io][limit_parameter]["in_use"]:
                    if states_stats[ip_v][io]["has_stats"]:
                        # There is statistics data for this ip_v and io
                        current = states_stats[ip_v][io][limit_parameter]
                        warn_limit = limits[ip_v][io][limit_parameter]["WARN"]
                        crit_limit = limits[ip_v][io][limit_parameter]["CRIT"]
                        if current > crit_limit:
                            juggler_result.update(2, "{}-{}-{} > {}".format(ip_v, io, limit_parameter, crit_limit))
                        elif current > warn_limit:
                            juggler_result.update(1, "{}-{}-{} > {}".format(ip_v, io, limit_parameter, warn_limit))
                    else:
                        if args.NO_DATA_IS_FINE:
                            juggler_result.update(0, "{}-{} no data at all".format(ip_v, io))
                            continue
                        else:
                            # There is no any statistics data for this ip_v and io -
                            # raise CRIT and do not check other limits
                            juggler_result.update(2, "{}-{} no data at all".format(ip_v, io))
                            break


def print_debug():
    """
        Prints information about statistics on history_stats.
    """
    print "=== Analysis results"
    print "History probes file: {}".format(args.HISTORY_FILE)
    print "Limits config file: {}".format(args.LIMITS_CONFIG_FILE)
    print "Max history probes number: {}".format(args.MAX_HISTORY_RECORDS)
    print "Number of 'new' probes: {}".format(args.NEW_PROBES_NUM)
    print "Period: {} secs".format(states_stats["period_secs"])
    for ip_v in ("v4", "v6"):
        print "IP version:", ip_v
        for io in lib.INOUT:
            print "    Traffic direction:", io
            print "        Data:"
            for p in states_stats_parameters.keys():
                warn_crit_limits = ""
                if p in limits[ip_v][io] and limits[ip_v][io][p]["in_use"] is True:
                    warn_crit_limits = " WARN " + str(limits[ip_v][io][p]["WARN"]) + " CRIT " + str(limits[ip_v][io][p]["CRIT"])
                print "            {:<25}{:>20}{}".format(
                    p,
                    states_stats[ip_v][io][p],
                    warn_crit_limits
                )


class JugglerStatus(object):
    """
        Class for yiending check results to juggler-client
    """
    def __init__(self, name):
        self.name = name
        self.status = 0
        self.description = ""

    def update(self, status, description=""):
        description = str(description).replace("\n", r"\n")
        if status > self.status:
            self.status = status
        if description:
            if self.description:
                self.description = self.description + "; " + description
            else:
                self.description = description

    def __str__(self):
        out = "PASSIVE-CHECK:{};{};{}"
        out = out.format(self.name, self.status,
                         self.description if self.description else "OK")
        return out


# Main stuff
def main():
    global args
    global limits
    global state
    global states_stats

    juggler_result = JugglerStatus(CHECK_NAME)

    args = parse_arguments()

    read_config_success, config = lib.read_yaml_file(args.LIMITS_CONFIG_FILE)
    if not read_config_success:
        juggler_result.update(2, "Limits config file {} could not be read or parsed as YAML".format(args.LIMITS_CONFIG_FILE))
    else:
        limits = load_config(limits, config)

    read_history_states_success, history_states = lib.read_json_file(args.HISTORY_FILE)
    if not read_history_states_success:
        juggler_result.update(2, "History file {} could not be read or parsed as JSON, continue anyway".format(args.HISTORY_FILE))

    if args.MANUAL:
        state = load_current_state(current_state)
    else:
        state = lib.update_current_state(current_state)

    if not args.DEBUG:
        history_states = update_history_states_with_current(history_states, current_state)
    else:
        if not read_history_states_success:
            print "DEBUG FAIILED: History file {} could not be read or parsed as JSON, so nothing to analyze.".format(args.HISTORY_FILE)
            print juggler_result
            exit(0)

    states_stats = analyze_history(history_states, states_stats)

    if not args.DEBUG:
        write_history_states_success = lib.write_json_file(history_states, args.HISTORY_FILE)
        if not write_history_states_success:
            juggler_result.update(2, "History file {} could not be written, exiting".format(args.HISTORY_FILE))
            print juggler_result
            exit(0)

    test_limits(states_stats, limits, juggler_result)

    if args.DEBUG:
        print_debug()

    print juggler_result
    exit(0)
