#! /usr/bin/env python2
# -*- coding: utf-8 -*-

import argparse
import sys
import functools

from socket import getfqdn

from . import lib


DEFAULT_GRAPHITE_HISTORY_FILE = "/var/tmp/hbf-monitoring-drops-graphite.tmp"
DEFAULT_GRAPHITE_TIME_PREFIX = "one_min"
DEFAULT_GRAPHITE_FQDN = getfqdn().replace('.', '_')
DEFAULT_GRAPHITE_POSTFIX = "hbf"
DEFAULT_GRAPHITE_OUTPUT_STYLE = "default"


state = lib.make_state()


def parse_arguments():
    parser = argparse.ArgumentParser(description="HBF drops Graphite collector.")
    parser.add_argument('-d', '--previous-data',
                        help="Path to file with previous data",
                        dest='HISTORY_FILE',
                        default=DEFAULT_GRAPHITE_HISTORY_FILE)
    parser.add_argument('-t', '--time-prefix',
                        help="Graphite time period prefix",
                        dest='TIME_PREFIX',
                        default=DEFAULT_GRAPHITE_TIME_PREFIX)
    parser.add_argument('-f', '--fqdn',
                        help="This server's FQDN in Graphite format",
                        dest='FQDN',
                        default=DEFAULT_GRAPHITE_FQDN)
    parser.add_argument('-p', '--postfix',
                        help="Postfix, under which metrics data will be stored",
                        dest='POSTFIX',
                        default=DEFAULT_GRAPHITE_POSTFIX)
    parser.add_argument('-o', '--output-style',
                        help="Output style, can be \"default\" or \"simple\".\nFor \"simple\" style output POSTFIX used as PREFIX and has no TS in output.",
                        dest='OUTPUT_STYLE',
                        default=DEFAULT_GRAPHITE_OUTPUT_STYLE)
    args = parser.parse_args()
    return args


def make_output_for_graphite(current_state, previous_state, prefix, timestamp=None):
    # hbf.v4.input.packets_dropped 42
    names_values = []
    for ip_v in ("v4", "v6"):
        for io in lib.INOUT:
            for counter_type in ["packets_total", "packets_dropped"]:
                c = current_state[ip_v][io][counter_type]
                p = previous_state[ip_v][io][counter_type]
                if c is not None and p is not None:
                    if c >= p:
                        diff = c - p
                    else:
                        diff = c
                    names_values.append("{p}.{ip_v}.{io}.{c} {v}{ts}".format(
                        p=prefix,
                        ip_v=ip_v,
                        io=io,
                        c=counter_type,
                        v=diff,
                        ts=timestamp and (" %s" % timestamp) or "",
                    ))

    result = '\n'.join(names_values)
    return result


# Main logic
def main():
    global state

    args = parse_arguments()

    state = lib.update_current_state(state)

    read_previous_state_success, previous_state = lib.read_json_file(args.HISTORY_FILE)

    if not read_previous_state_success:
        print >> sys.stderr, "Cannot read JSON from file with previous data"
        write_file_success = lib.write_json_file(state, args.HISTORY_FILE)
        if not write_file_success:
            print >> sys.stderr, "Cannot write JSON to file with previous data"
            exit(1)
    else:
        make_output = functools.partial(make_output_for_graphite, state, previous_state)
        if args.OUTPUT_STYLE == "simple":
            print make_output(prefix=args.POSTFIX)
        else:
            print make_output(
                prefix="%s.%s.%s" % (args.TIME_PREFIX, args.FQDN, args.POSTFIX),
                timestamp=state["timestamp"]
            )
        write_file_success = lib.write_json_file(state, args.HISTORY_FILE)
        if not write_file_success:
            print >> sys.stderr, "Cannot write JSON to file with previous data"
            exit(1)

    exit(0)
