import time
import os
import subprocess
import json
# python-yaml package required
import yaml
import shlex

INOUT = ("input", "output")

_IPTABLES_COMMAND = {
    "v4": "/sbin/iptables",
    "v6": "/sbin/ip6tables"
}

_ROOT_CHAINS = {
    "input": "Y_FW",
    "output": "Y_FW_OUT",
}

_DROP_CHAINS = {
    "input": "Y_END_IN",
    "output": "Y_END_OUT",
}


def make_state():
    state_values = {
        "packets_total": None,
        "packets_dropped": None
    }

    return {
        "timestamp": None,
        "v4": {
            "input": dict(state_values),
            "output": dict(state_values)
        },
        "v6": {
            "input": dict(state_values),
            "output": dict(state_values)
        }
    }


def update_current_state(state):
    """
        Updates current state dictionary with counters values from ip[6]tables.
        Input:  state dictionary
        Output: Updated state dictionary
    """
    state["timestamp"] = int(time.time())
    for ip_v in ("v4", "v6"):
        for io in INOUT:
            total = get_counters(ip_v, _ROOT_CHAINS[io])
            if total is not None:
                state[ip_v][io]["packets_total"] = sum(total.values())

            drops = get_counters(ip_v, _DROP_CHAINS[io])
            if drops is not None:
                state[ip_v][io]["packets_dropped"] = max(
                    drops.get("DROP", 0) + drops.get("REJECT", 0),
                    drops.get("LOG", 0)
                )
    return state


def get_opt(opts, opt_name):
    try:
        return opts[opts.index(opt_name) + 1]
    except (ValueError, IndexError):
        return None


def get_counters(ip_version, chain_name):
    """
        Runs /sbin/ip[6]tables -L chain_name.
        Input:  IP version, 4 or 6
                Traffic direction, input or output
                Chain name
        Output: None or dict(str(rule_target): int(sum_of_matched_pkts))
    """
    cmd = ["sudo", "-n", _IPTABLES_COMMAND[ip_version], "-vS", chain_name]
    with open(os.devnull, 'w') as devnull:
        p = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=devnull)
    devnull.close()
    stdout = p.communicate()[0]
    if p.returncode != 0:
        return None

    ret = {}
    for line in stdout.splitlines():
        opts = shlex.split(line)
        pkts = get_opt(opts, "-c")
        if pkts is None:
            continue
        action = get_opt(opts, "-j")
        if action not in ret:
            ret[action] = 0
        ret[action] += int(pkts)
    return ret


def read_json_file(file_path):
    """
        Input:  File path
        Output: Boolean read_file_success, dictionary
    """
    data = []
    read_file_success = False
    try:
        f = open(file_path, "r")
        data = json.load(f)
        f.close()
        read_file_success = True
    except (IOError, ValueError):
        read_file_success = False
        data = []
    return read_file_success, data


def write_json_file(data, file_path):
    """
        Input:  Dictionary and file path
        Output: Boolean write_file_success
    """
    write_file_success = False
    try:
        f = open(file_path, "w")
        json.dump(data, f)
        f.close()
        write_file_success = True
    except IOError:
        write_file_success = False
    try:
        os.chmod(file_path, 0o666)
    except IOError:
        pass
    return write_file_success


def read_yaml_file(file_path):
    """
        Input:  File path
        Output: Boolean read_file_success, data object
    """
    data = []
    read_file_success = False
    try:
        f = open(file_path, "r")
        data = yaml.load(f, Loader=yaml.SafeLoader)
        f.close()
        read_file_success = True
    except (IOError, ValueError):
        read_file_success = False
        data = []
    return read_file_success, data
