package cmd

import (
	"context"
	"fmt"
	"log"
	"net"
	"os"
	"os/signal"
	"syscall"
	"text/tabwriter"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/infra/rtc/packages/yandex-hbf-agent/hbf-metrics-pusher/internal/iptables"
	"a.yandex-team.ru/infra/rtc/packages/yandex-hbf-agent/hbf-metrics-pusher/internal/netwrap"
	"a.yandex-team.ru/infra/rtc/packages/yandex-hbf-agent/hbf-metrics-pusher/internal/porto"
)

var dropsCmd = &cobra.Command{
	Use:   "drops",
	Short: "show current drops",
	Run:   dropsShow,
}

var showAll bool

func dropsShow(cmd *cobra.Command, args []string) {

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	ipt6, err := iptables.New(ctx, iptables.IPv6, vanila)
	if err != nil {
		log.Fatal(err)
	}

	// signal handler
	sig := make(chan os.Signal)
	signal.Notify(sig, syscall.SIGTERM, syscall.SIGINT, syscall.SIGQUIT)
	go func() {
		defer signal.Stop(sig)
		s := <-sig
		log.Printf("Got signal: %d", s)
		cancel()
	}()

	portoClient, err := porto.CreatePortoClient()
	if err != nil {
		log.Fatal(err)
	}
	defer portoClient.Close()

	IPLabelMap, err := porto.IPLabelMap(portoClient)
	if err != nil {
		cancel()
		log.Fatal(err)
	}

	// iptables output reader
	iptOut := make(chan []iptables.Answer, 1)
	go func(ipt *iptables.IPTables, c chan []iptables.Answer) {
		defer close(iptOut)
		ipt.ListPreDropRules(c)
	}(ipt6, iptOut)

	var addr string
	resolver := &net.Resolver{}

	w := tabwriter.NewWriter(os.Stdout, 0, 0, 1, ' ', 0)
	_, err = fmt.Fprintf(w, "pkts\taddr\tdirection\tengine\tservice-id\tfqdn\n")
	// Explicit is better than implicit.
	if err != nil {
		log.Printf("%s", err)
	}
	for _, answ := range <-iptOut {
		for _, rule := range answ.Rules {
			if string(rule["pkts"]) == "0" && !showAll {
				continue
			}
			addr = iptables.SelectAddrField(answ.Direction, rule)
			addr, err = netwrap.TryParseIP(addr)
			if err != nil {
				log.Printf("could not parse %s to net.IP", addr)
				continue
			}
			label := IPLabelMap[addr]
			engine := label.Engine
			if engine == "" {
				engine = "host"
			}
			serviceID := label.ServiceID
			if serviceID == "" {
				serviceID = "None"
			}
			names, _ := resolver.LookupAddr(ctx, addr)
			_, err = fmt.Fprintf(w, "%s\t%s\t%s\t%s\t%s\t%q\n", rule["pkts"], addr, answ.Direction.String(), engine, serviceID, names)
			// Explicit is better than implicit.
			if err != nil {
				log.Printf("%s", err)
			}
		}
	}
	_ = w.Flush()
}

func init() {
	rootCmd.AddCommand(dropsCmd)
	dropsCmd.Flags().BoolVar(&showAll, "show-all", false, "show services with zero dropped packets too")
}
