// IPTables provides limited functionality to work with iptables.
// Since functional is limited, it belongs to 'internal'.
package iptables

import (
	"bytes"
	"context"
	"fmt"
	"os/exec"
	"path"
	"regexp"
	"syscall"

	"go.uber.org/zap"
)

const (
	yndxIptablesBasePath = "/opt/yandex-iptables"

	IPv4 = syscall.AF_INET
	IPv6 = syscall.AF_INET6

	InChain  = "Y_END_IN_PRE"
	OutChain = "Y_END_OUT_PRE"
)

const (
	DirectionIn = Direction(iota)
	DirectionOut
)

type Answer struct {
	Rules     ParsedRules
	Direction Direction
}

type Direction uint8

type ChainDirection map[string]Direction

func (d Direction) String() string {
	return [...]string{"IN", "OUT"}[d]
}

func (d Direction) EnumIndex() uint8 {
	return uint8(d)
}

var DefChainDirection ChainDirection

type IPTables struct {
	proto   uint8
	vanila  bool
	binPath string
	ctx     context.Context
	re      *regexp.Regexp
}

type ParsedRules []map[string][]byte

func New(ctx context.Context, protocol uint8, vanila bool) (*IPTables, error) {
	bp, err := determineBinPath(protocol, vanila)
	if err != nil {
		return nil, err
	}

	re := regexp.MustCompile(`(?m)\s+?(?P<pkts>\d+)\s+(?P<bytes>\d+)\s+(?P<target>\w+)\s+(?P<prot>\w+)\s+(?P<opt>\w+)?\s+(?P<in>[\w\*\-\_\@]+)\s+(?P<out>[\w\*\-\_\@]+)\s+(?P<source>[\w:/]+)\s+(?P<destination>[\w:/]+)\s+\/\*\s+(?P<comment>.*?)\s+\*\/`)
	return &IPTables{
		ctx:     ctx,
		binPath: bp,
		proto:   protocol,
		vanila:  vanila,
		re:      re,
	}, nil
}

func (ipt *IPTables) ListRules(table, chain string) (ParsedRules, error) {
	var out bytes.Buffer
	var res ParsedRules

	c := exec.CommandContext(ipt.ctx, ipt.binPath, "-t", table, "-v", "-w", "5", "-x", "-n", "-L", chain)
	c.Stdout = &out
	err := c.Run()
	if err != nil {
		return res, err
	}
	res = ipt.parseChain(out.Bytes())
	return res, nil
}

func (ipt *IPTables) parseChain(b []byte) ParsedRules {
	match := ipt.re.FindAllSubmatch(b, -1)
	// res := []map[string][]byte{}
	res := make(ParsedRules, 0)
	for i := 0; i < len(match); i++ {
		m := make(map[string][]byte)
		for j, name := range ipt.re.SubexpNames() {
			if j != 0 && name != "" {
				m[name] = match[i][j]
			}
		}
		if len(m) != 0 {
			res = append(res, m)
		}

	}
	return res
}

func determineBinPath(proto uint8, vanila bool) (string, error) {
	var baseName string

	switch proto {
	case syscall.AF_INET:
		baseName = "iptables"
	case syscall.AF_INET6:
		baseName = "ip6tables"
	default:
		return "", fmt.Errorf("unsupported protocol, should be AF_INET (%d) or AF_INET6 (%d), got: %d", syscall.AF_INET, syscall.AF_INET6, proto)
	}

	if !vanila {
		return path.Join(yndxIptablesBasePath, baseName), nil
	} else {
		ap, err := exec.LookPath(baseName)
		if err != nil {
			return "", err
		}
		return ap, nil
	}
}

func (ipt *IPTables) ListPreDropRules(c chan []Answer) {
	a := []Answer{}
	for ch := range DefChainDirection {
		if r, err := ipt.ListRules("filter", ch); err != nil {
			zap.L().Warn("iptables", zap.Error(err))
		} else {
			a = append(a, Answer{Rules: r, Direction: DefChainDirection[ch]})
		}
	}
	c <- a
}

func SelectAddrField(d Direction, rule map[string][]byte) string {
	if d == DirectionIn {
		return string(rule["destination"])
	}
	return string(rule["source"])
}

func init() {
	DefChainDirection = make(ChainDirection, 2)
	DefChainDirection[OutChain] = DirectionOut
	DefChainDirection[InChain] = DirectionIn
}
