package porto

import (
	"fmt"
	"strings"

	"go.uber.org/zap"

	porto_api "a.yandex-team.ru/infra/porto/api_go"
	"a.yandex-team.ru/infra/rtc/packages/yandex-hbf-agent/hbf-metrics-pusher/internal/netwrap"
)

type Labels struct {
	Engine    string
	ServiceID string
}

func CreatePortoClient() (porto_api.PortoAPI, error) {
	conn, err := porto_api.Dial()
	if err != nil {
		zap.L().Error("failed to dial Porto", zap.Error(err))
		return nil, err
	}
	return conn, nil
}

func IPLabelMap(p porto_api.PortoAPI) (map[string]Labels, error) {
	cntnrs, err := p.ListContainers("*")
	if err != nil {
		return nil, err
	}
	props, err := p.GetProperties(cntnrs, []string{"ip", "labels"})
	if err != nil {
		return nil, err
	}
	iplm, warns := formIPLabelMap(props)
	if len(warns) > 0 {
		zap.L().Warn("", zap.Strings("Skipped containers", warns))
	}
	return iplm, nil
}

func formIPLabelMap(ctsProps map[string]map[string]string) (map[string]Labels, []string) {
	res := make(map[string]Labels)
	warns := []string{}

	for ctName, ctProps := range ctsProps {
		ipProps, ok := ctProps["ip"]
		if !ok {
			warns = append(warns, fmt.Sprintf("%s: no 'ip' property", ctName))
			continue
		}
		labelsProps, ok := ctProps["labels"]
		if !ok {
			warns = append(warns, fmt.Sprintf("%s: no 'labels' property", ctName))
			continue
		}
		ips := parseIPProps(ipProps)
		label := parseLabels(labelsProps)
		for _, ip := range ips {
			res[ip] = label
		}
	}
	return res, warns
}

func parseIPProps(ipProps string) []string {
	res := []string{}
	split := strings.Split(ipProps, ";")
	for _, s := range split {
		ip := strings.Split(s, " ")
		if len(ip) != 2 {
			continue
		}
		nip, err := netwrap.TryParseIP(ip[1])
		if err != nil {
			zap.L().Error("failed to parse IP", zap.String("addr", ip[1]), zap.Error(err))
			continue
		}
		res = append(res, nip)
	}
	return res
}

func parseLabels(labelsProp string) Labels {
	l := Labels{}
	split := strings.Split(labelsProp, "; ")
	for _, s := range split {
		lv := strings.Split(s, ":")
		if len(lv) != 2 {
			continue
		}
		key := strings.Trim(lv[0], " ")
		value := strings.Trim(lv[1], " ")
		switch key {
		case "INFRA.engine":
			l.Engine = value
		case "INFRA.service-id":
			l.ServiceID = value
		default:
			continue
		}
	}
	return l
}
