package porto

import (
	"reflect"
	"testing"
)

func TestParseLabels(t *testing.T) {
	prop := "CORE.total: 7; HOOK.iss_hook_install: done; HOOK.iss_hook_notify: done; HOOK.iss_hook_status: done; HOOK.iss_hook_stop: done; INFRA.engine: XXX; INFRA.service-id: YYY"
	expect := Labels{
		Engine:    "XXX",
		ServiceID: "YYY",
	}
	res := parseLabels(prop)
	if res != expect {
		t.Errorf("parseLabels result unexpected,\nwant: %#v\ngot: %#v", expect, res)
	}
}

func TestParseIPProps(t *testing.T) {
	inp := "veth 2001:db8:fc23:1d83:0:696:35b5:0;veth 2001:db8:c23:1d83:0:696:df2d:0"
	expect := []string{"2001:db8:fc23:1d83:0:696:35b5:0", "2001:db8:c23:1d83:0:696:df2d:0"}
	res := parseIPProps(inp)

	if !reflect.DeepEqual(res, expect) {
		t.Errorf("parseIPProps result unexpected,\nwant: %#v\ngot: %#v", expect, res)
	}

}

func TestFormIPLabelMap(t *testing.T) {
	inp := make(map[string]map[string]string)

	ctInternalM1 := make(map[string]string)
	ctInternalM1["ip"] = "veth 2001:db8:fc23:1d83:0:696:35b5:0;veth 2001:db8:c23:1d83:0:696:df2d:0"
	ctInternalM1["labels"] = "HOOK.iss_hook_stop: done; INFRA.engine: XXX; INFRA.service-id: YYY; HOOK.iss_hook_status: done"

	ctInternalM2 := make(map[string]string)
	ctInternalM2["ip"] = "veth 2001:db8::1;veth 2001:db8::2"
	ctInternalM2["labels"] = "HOOK.iss_hook_stop: done; INFRA.engine: AAA; INFRA.service-id: ZZZ; HOOK.iss_hook_status: done"

	ctInternalM3 := make(map[string]string)
	ctInternalM3["some"] = "test"

	ctInternalM4 := make(map[string]string)
	ctInternalM4["ip"] = "test"

	inp["ct1"] = ctInternalM1
	inp["ct2"] = ctInternalM2
	inp["ct3"] = ctInternalM3
	inp["ct4"] = ctInternalM4

	res, warns := formIPLabelMap(inp)
	expRes := make(map[string]Labels)
	expWarns := []string{"ct3: no 'ip' property", "ct4: no 'labels' property"}

	ctLabels1 := Labels{Engine: "XXX", ServiceID: "YYY"}
	ctLabels2 := Labels{Engine: "AAA", ServiceID: "ZZZ"}
	expRes["2001:db8:fc23:1d83:0:696:35b5:0"] = ctLabels1
	expRes["2001:db8:c23:1d83:0:696:df2d:0"] = ctLabels1
	expRes["2001:db8::1"] = ctLabels2
	expRes["2001:db8::2"] = ctLabels2

	if !reflect.DeepEqual(expRes, res) {
		t.Errorf("formIPLabelMap unexpected result\nwant: %#v\ngot: %#v", expRes, res)
	}
	if !reflect.DeepEqual(expRes, res) {
		t.Errorf("unexpected  warns\nwant: %#v\ngot: %#v", expWarns, warns)
	}
}
