package solomon

import (
	"context"
	"fmt"
	"io"
	"net/http"

	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/security/gideon/nuvault/pkg/nuvault"
)

type SolomonPusher struct {
	tvmClient  *tvmauth.Client
	httpClient *http.Client
	cfg        *Config
}

func New(ctx context.Context, cfg *Config) (*SolomonPusher, error) {
	nuc, err := nuvault.NewClient(nuvault.WithMutualAuth(cfg.Secrets.HostCert))
	if err != nil {
		return nil, fmt.Errorf("failed to create nuvault client: %w", err)
	}
	defer nuc.Close()

	secret, err := nuc.GetSecret(ctx, cfg.Secrets.VaultID)
	if err != nil {
		return nil, fmt.Errorf("failed to get secret: %w", err)
	}

	tvmSecret, ok := secret.Values[cfg.Secrets.StorageKey]
	if !ok {
		return nil, &NotFoundKey{Key: cfg.Secrets.StorageKey}
	}

	tvms := tvmauth.TvmAPISettings{
		SelfID: cfg.Tvm.MyselfID,
		ServiceTicketOptions: tvmauth.NewIDsOptions(
			tvmSecret,
			[]tvm.ClientID{
				cfg.Tvm.SolomonID,
			}),
		DiskCacheDir: cfg.Tvm.CacheDir,
	}

	tvmC, err := tvmauth.NewAPIClient(tvms, &nop.Logger{})
	if err != nil {
		return nil, err
	}

	hc := &http.Client{
		Timeout: cfg.Solomon.Timeout,
	}
	return &SolomonPusher{
		tvmClient:  tvmC,
		httpClient: hc,
		cfg:        cfg,
	}, nil
}

func (s *SolomonPusher) Push(ctx context.Context, body io.Reader) (int, error) {
	serviceTicket, err := s.tvmClient.GetServiceTicketForID(ctx, s.cfg.Tvm.SolomonID)
	if err != nil {
		return 0, err
	}

	r, err := http.NewRequestWithContext(ctx, "POST", s.cfg.Solomon.URL, body)
	if err != nil {
		return 0, err
	}
	r.Header.Add("Content-Type", "application/json")
	r.Header.Add("X-Ya-Service-Ticket", serviceTicket)

	q := r.URL.Query()
	q.Add("project", s.cfg.Solomon.Project)
	q.Add("cluster", s.cfg.Solomon.Cluster)
	q.Add("service", s.cfg.Solomon.Service)
	r.URL.RawQuery = q.Encode()

	resp, err := s.httpClient.Do(r)
	if err != nil {
		return 0, err
	}
	defer resp.Body.Close() // just in case
	return resp.StatusCode, nil
}
