from __future__ import print_function

import logging
import sys

import configobj
from util import Singleton
from validate import Validator

log = logging.getLogger(__name__.split(".")[-1])
log.addHandler(logging.NullHandler())


class ConfigError(Exception):
    pass


def overrides_from_args(args):
    """
    Parses overrides from args:
    ['http_api.port=8080', 'log_level=DEBUG'] -> {('main', 'log_level'): 'DEBUG', ('http_api', 'port'): '8080'}
    """
    rv = {}
    if not args:
        return rv
    for arg in args:
        if '=' not in arg:
            raise ConfigError("Invalid override: no '=' in '{}'".format(arg))
        k, v = arg.split('=', 1)
        # Now try to split k into section and key
        if '.' in k:
            section, key = k.split('.', 1)
        else:
            section, key = 'main', k
        if not section:
            raise ConfigError("Invalid override: no section in '{}'".format(arg))
        if not key:
            raise ConfigError("Invalid override: no key in '{}'".format(arg))
        rv[(section, key)] = v
    return rv


class Config(object):
    __metaclass__ = Singleton

    @classmethod
    def delete(cls):
        cls.__metaclass__.delete(cls)

    def __init__(self, config=None, configspec=None,
                 overrides=None,
                 unrepr=False, check_extra=True):
        self.config = config
        self.configspec = configspec
        self.unrepr = unrepr
        self.overrides = overrides
        self.check_extra = check_extra
        self.obj, errors = self._load_config()
        if errors:
            for msg in errors:
                print(msg, file=sys.stderr)
            raise ConfigError("Invalid configuration.")

    def _load_config(self):
        spec = configobj.ConfigObj(self.configspec,
                                   list_values=False,
                                   write_empty_values=True,
                                   interpolation=False,
                                   _inspec=True)
        obj = configobj.ConfigObj(self.config,
                                  configspec=spec,
                                  unrepr=self.unrepr,
                                  list_values=False,
                                  write_empty_values=True,
                                  interpolation=False)
        if self.overrides:
            for (section, key), val in self.overrides.items():
                try:
                    obj[section][key] = val
                except KeyError:
                    obj[section] = {key: val}
        result = obj.validate(Validator(), preserve_errors=True)
        errors = self._validation_errors(obj, result)
        if self.check_extra:
            errors += self._extra_values(obj)
        return obj, errors

    @staticmethod
    def _validation_errors(obj, result):
        errors = configobj.flatten_errors(obj, result)
        messages = []
        for sections, key, error in errors:
            if not sections:
                sections = ("top level",)
            msg = "Invalid configuration. Sections: {}; key: {}; error: {}"
            messages.append(msg.format(", ".join(sections), key, error))
        return messages

    @staticmethod
    def _extra_values(obj):
        extras = configobj.get_extra_values(obj)
        messages = []
        for sections, key in extras:
            if not sections:
                sections = ("top level",)
            msg = ("Extra section/key in configuration (probably a typo)."
                   " Section: {}; key: {}.")
            messages.append(msg.format(", ".join(sections), key))
        return messages

    def reload(self):
        try:
            obj, errors = self._load_config()
        except Exception:
            log.exception("Unable to reload config:")
            # FIXME: Reraise here?
        if not errors:
            self.obj = obj
        else:
            log.error("Invalid configuration:\n" + "\n".join(errors))
        return errors

    @classmethod
    def remove_comments(cls, obj):
        def clear_dict(d):
            for k in d:
                d[k] = []

        if isinstance(obj, configobj.ConfigObj):
            obj.initial_comment = []
            obj.final_comment = []
            clear_dict(obj.comments)
            clear_dict(obj.inline_comments)

            # Restore one newline before every section starting from second.
            for n, k in enumerate(obj):
                if isinstance(obj[k], configobj.Section):
                    if n >= 1:
                        obj.comments[k] = ['\n']

        for v in obj.values():
            if isinstance(v, configobj.Section):
                clear_dict(v.comments)
                clear_dict(v.inline_comments)
                cls.remove_comments(v)

    @classmethod
    def dump_default(cls, configspec):
        obj = configobj.ConfigObj(configspec=configspec,
                                  list_values=False,
                                  write_empty_values=True)
        result = obj.validate(Validator(), copy=True, preserve_errors=True)
        errors = cls._validation_errors(obj, result)
        for msg in errors:
            print(msg, file=sys.stderr)
        cls.remove_comments(obj)
        obj.write(sys.stdout)
        return errors

    def __str__(self):
        try:
            self.obj.filename = None
            s = "\n".join(self.obj.write()) + "\n"
        finally:
            self.obj.filename = self.config
        return s

    def __getitem__(self, name):
        return self.obj[name]

    def use_yandex_iptables(self):
        try:
            use_yandex_iptables = self.obj['main']['use_yandex_iptables']
            return use_yandex_iptables
        except Exception:
            return False


class LoggingConfig(Config):
    pass
