from collections import namedtuple
import logging

from hbfagent.iptables import IPTables
from hbfagent import util
from hbfagent.config import Config


log = logging.getLogger(__name__.split(".")[-1])
log.addHandler(logging.NullHandler())


PROTOCOLS = ("tcp", "udp")
RUNTIME_PROJECT_ID_START = 0x1000000
RUNTIME_PROJECT_ID_END = 0x1FFFFFF


def runtime_project_id(project_id):
    return RUNTIME_PROJECT_ID_START <= project_id <= RUNTIME_PROJECT_ID_END


PortRange = namedtuple("PortRange", "start, end")


def run(ip_version, host_ips, guest_ips):
    config = Config()
    use_yandex_iptables = config.use_yandex_iptables()
    if ip_version == "v4":
        return IPTables(ip_version, use_yandex_iptables=use_yandex_iptables)

    log.info('Looking for IP addresses with "Runtime Cloud" project IDs.')
    project_id_ips = []
    local_port_start, _ = get_local_port_range()
    reserved_ports = get_local_reserved_ports()
    for ip in guest_ips:
        if (isinstance(ip, util.ISSIPv6Address) and
                ip.iss_hbf_nat != "disabled" and
                ip in util.BACKBONE_MTN_NET):
            project_id = util.get_project_id(ip)
            if runtime_project_id(project_id):
                port = util.get_port(ip)
                port_range = PortRange(port, port + 7)
                if ok_for_nat(port_range, local_port_start, reserved_ports):
                    msg = ("Generating NAT rules for IP '{}', "
                           "project ID {:x}, port range {}.")
                    log.info(msg.format(ip, project_id, tuple(port_range)))
                    project_id_ips.append((ip, port_range))

    if project_id_ips:
        return gen_mtn_nat_rules(host_ips, project_id_ips)
    else:
        return IPTables(ip_version, use_yandex_iptables=use_yandex_iptables)


def get_local_port_range():
    path = "/proc/sys/net/ipv4/ip_local_port_range"
    try:
        with open(path) as f:
            start, end = f.readline().split()
        return (int(start), int(end))
    except Exception:
        log.exception("Unable get local port range from '{}':".format(path))
        return (32768, 65535)


def get_local_reserved_ports():
    path = "/proc/sys/net/ipv4/ip_local_reserved_ports"
    reserved_ports = set()
    try:
        with open(path) as f:
            line = f.readline().rstrip()
            if line:
                for elem in line.split(","):
                    if "-" in elem:
                        start, end = map(int, elem.split("-"))
                        reserved_ports.update(range(start, end + 1))
                    else:
                        reserved_ports.add(int(elem))
    except Exception:
        msg = "Unable get local reserved ports from '{}':"
        log.exception(msg.format(path))
    return reserved_ports


def ok_for_nat(port_range, local_port_start, reserved_ports):
    if port_range.start < 1024:
        msg = "Port range {} has intersection with privileged ports."
        log.error(msg.format(tuple(port_range)))
        return False
    elif port_range.end >= local_port_start:
        msg = "Port range {} has intersection with local ports."
        log.error(msg.format(tuple(port_range)))
        return False
    elif set(range(port_range.start, port_range.end + 1)) & reserved_ports:
        msg = "Port range {} has intersection with reserved ports."
        log.error(msg.format(tuple(port_range)))
        return False

    return True


def gen_mtn_nat_rules(host_ips, project_id_ips):
    config = Config()
    use_yandex_iptables = config.use_yandex_iptables()

    tables = IPTables("v6", use_yandex_iptables=use_yandex_iptables)

    tables.append_rule(
        "nat", "PREROUTING", "-A PREROUTING -j PREROUTING_MTN_NAT"
    )
    tables.append_rule(
        "nat", "OUTPUT", "-A OUTPUT -j OUTPUT_MTN_NAT"
    )
    tables.append_rule(
        "raw", "PREROUTING", "-A PREROUTING -j PREROUTING_MTN_NAT"
    )
    tables.append_rule(
        "raw", "OUTPUT", "-A OUTPUT -j OUTPUT_MTN_NAT"
    )

    for guest_ip, port_range in project_id_ips:
        for host_ip in host_ips:
            # PREROUTING
            for proto in PROTOCOLS:
                tables.append_rule(
                    "nat", "PREROUTING_MTN_NAT",
                    "-A PREROUTING_MTN_NAT -p {} -d {} --dport {}:{}"
                    " -j DNAT --to-destination {}".format(
                        proto, host_ip, port_range.start, port_range.end,
                        guest_ip
                    )
                )
        for proto in PROTOCOLS:
            # OUTPUT
            tables.append_rule(
                "nat", "OUTPUT_MTN_NAT",
                "-A OUTPUT_MTN_NAT -p {} -o lo --dport {}:{}"
                " -j DNAT --to-destination {}".format(
                    proto, port_range.start, port_range.end, guest_ip
                )
            )

    for guest_ip, port_range in project_id_ips:
        # PREROUTING
        for host_ip in host_ips:
            tables.append_rule(
                "raw", "PREROUTING_MTN_NAT",
                "-I PREROUTING_MTN_NAT -p tcp -d {} --dport {}:{}"
                " -j ACCEPT".format(
                    host_ip, port_range.start, port_range.end
                )
            )
        tables.append_rule(
            "raw", "PREROUTING_MTN_NAT",
            "-I PREROUTING_MTN_NAT -p tcp -s {} --sport {}:{}"
            " -j ACCEPT".format(
                guest_ip, port_range.start, port_range.end
            )
        )
        # OUTPUT
        tables.append_rule(
            "raw", "OUTPUT_MTN_NAT",
            "-I OUTPUT_MTN_NAT -p tcp -o lo --dport {}:{}"
            " -j ACCEPT".format(port_range.start, port_range.end)
        )
        tables.append_rule(
            "raw", "OUTPUT_MTN_NAT",
            "-I OUTPUT_MTN_NAT -p tcp -o L3+ --dport {}:{}"
            " -j ACCEPT".format(port_range.start, port_range.end)
        )
        tables.append_rule(
            "raw", "PREROUTING_MTN_NAT",
            "-I PREROUTING_MTN_NAT -p tcp -i lo --sport {}:{}"
            " -j ACCEPT".format(port_range.start, port_range.end)
        )

    return tables
