import argparse
import json
import os
from os import path
import shutil
import socket
import struct
import subprocess
import time
import urllib2
from functools import partial

from ipaddr import IPv6Address
from ipaddr import IPv6Network

BACKBONE_MTN_NET = IPv6Network("2a02:6b8:c00::/40")
FASTBONE_MTN_NET = IPv6Network("2a02:6b8:fc00::/40")


class Singleton(type):
    """https://stackoverflow.com/a/6798042"""

    _instances = {}

    def __call__(cls, *args, **kwargs):
        if cls not in cls._instances:
            cls._instances[cls] = super(Singleton, cls).__call__(
                *args, **kwargs
            )
        return cls._instances[cls]

    @classmethod
    def delete(meta, cls):
        """Delete the only instance of a class so that the next instantiation
        returns a new one. Useful in tests.
        """
        del meta._instances[cls]


class ISSIPv6Address(IPv6Address):

    def __init__(self, *args, **kwargs):
        self.iss_hbf_nat = kwargs.pop("iss_hbf_nat", None)
        super(ISSIPv6Address, self).__init__(*args, **kwargs)

    def __eq__(self, other):
        super_eq = super(ISSIPv6Address, self).__eq__(other)
        if super_eq is NotImplemented:
            return NotImplemented

        try:
            self_eq = self.iss_hbf_nat == other.iss_hbf_nat
        except AttributeError:
            return NotImplemented

        return super_eq and self_eq


class Timer(object):

    def __init__(self):
        self.start = time.time()

    @property
    def interval(self):
        return time.time() - self.start

    def __str__(self):
        return str(self.interval())


def get_project_id(address):
    project_id = None
    if (address in BACKBONE_MTN_NET) or (address in FASTBONE_MTN_NET):
        project_id = struct.unpack(">L", address.packed[8:12])[0]

    return project_id


def get_port(address):
    return struct.unpack(">H", address.packed[14:16])[0]


def enumerate_lines(text, start=1):
    lines = text.splitlines()
    nw = len(str(len(lines)))
    return "\n".join(["{:<{nw}}: {}".format(n, line, nw=nw) for n, line
                      in enumerate(lines, start)]) + "\n"


def join_lines(*args):
    return "\n".join(args) + "\n"


def update_ssh_exception():
    parser = argparse.ArgumentParser(description="SSH exception rules updater")
    parser.add_argument(
        "--sleep",
        type=int, default=hash(socket.gethostname()) % 3600, metavar="S",
        help=("sleep before updating, seconds (default varies depending on"
              " hostname, current: %(default)s)")
    )
    args = parser.parse_args()

    time.sleep(args.sleep)
    r = urllib2.urlopen("https://sandbox.yandex-team.ru:443/api/v1.0/resource?limit=1&order=-id&type=SSH_EXCEPTION_RULES&state=READY")  # noqa: E501
    data = json.loads(r.read())
    skynet_id = data["items"][0]["skynet_id"]
    tmp_dir = "/var/tmp/hbf-update-ssh-exception"
    cmd = ["sky", "get", "-t", "300", "-d", tmp_dir, skynet_id]
    subprocess.check_call(cmd)
    new_rules_dir = path.join(tmp_dir, "ssh-exception-rules")
    for name in os.listdir(new_rules_dir):
        new_rules_path = path.join(new_rules_dir, name)
        etc_rules_path = path.join("/etc/yandex-hbf-agent/rules.d", name)
        tmp_rules_path = etc_rules_path + ".tmp"
        shutil.copy2(new_rules_path, tmp_rules_path)
        os.chmod(tmp_rules_path, 0o644)
        os.rename(tmp_rules_path, etc_rules_path)
    return 0


def list_from_csv(s, t=str):
    return [t(x.strip()) for x in s.split(",")]


class LazyOptionParser(object):
    """
    Makes it easy to create classes that store textual representation of a parameter
    and parses it only when needed (call self.ensure_parsed() in access methods)
    """
    def __init__(self, text):
        self._text = text
        self._parsed = False

    def ensure_parsed(self):
        if not self._parsed:
            self.parse(self._text)
            self._parsed = True

    def parse(self, text):
        raise NotImplementedError

    def __hash__(self):
        return hash(self._text)

    def __eq__(self, other):
        return type(self) is type(other) and self._text == other._text


class ArgParserFast(object):
    """
    This is an ad-hoc reimplementation of argparse.ArgumentParser
    Does not support many features, but enough to process iptables rules' options.
    But it's faster (10x) than the ArgumentParser. We need to parse huge amounts of
    iptables rules' parameters (10k+ lines).

    Known limitations:
    - no keyword arg "default=" in add_argument()
    - naive implementation of quoted parameters
      (does not support backslash escaping and converts multiple spaces into one)
    """

    def __init__(self):
        self._map = {}
        self._actions = {}

    def _store_opt(self, items, result, dest, conv_type, action):
        if action == "store_true":
            value = True
        else:
            param = next(items)

            # handle quoted strings
            if param.startswith(('"', "'")):
                quote_char = param[0]
                while not param.endswith(quote_char):
                    param += " " + next(items)
                param = param.strip(quote_char)

            value = conv_type(param)

        if action == "append":
            getattr(result, dest).append(value)
        else:
            setattr(result, dest, value)

    def add_argument(self, *opts, **kwargs):
        dest = kwargs.get("dest")
        if dest is None:
            dest = [opt[2:] for opt in opts if opt[:2] == "--"][0]
        conv_type = kwargs.get("type", str)
        action = kwargs.get("action", "store")
        self._actions[dest] = action
        for opt in opts:
            self._map[opt] = partial(
                self._store_opt,
                dest=dest,
                conv_type=conv_type,
                action=action,
            )

    def parse_known_args(self, args):
        result = ParamsNamespace()
        unknown_args = []
        iargs = iter(args)
        for dest, action in self._actions.items():
            setattr(result, dest, [] if action == "append" else None)
        try:
            while True:
                arg = next(iargs)
                if arg in self._map:
                    self._map[arg](iargs, result)
                else:
                    unknown_args.append(arg)
        except StopIteration:
            pass
        return result, unknown_args


class ParamsNamespace(object):
    pass
