from __future__ import print_function
import requests
import argparse
import sys
import time
import subprocess32
import random

from api.cqueue import Client


CLIENT_URL = 'http://juggler-api.search.yandex.net/v2'
LIMIT = 20
LOGDROP_SCRIPT = '/usr/sbin/ya-hbf-logdrop'


class JugglerClient(object):
    def __init__(self, url=CLIENT_URL):
        self.url = url
        self.headers = {
            'Content-Type': 'application/json',
        }

    def get_raw_events(self, service, tags, limit):
        raw = '/'.join((self.url, 'events/get_raw_events'))
        payload = {
            "filters": [
                {
                    "service": service,
                    "tags": tags,
                }
            ],
            "limit": limit,
        }
        r = requests.post(raw, headers=self.headers, json=payload)
        if r.status_code == requests.codes.ok:
            return r.json()
        r.raise_for_status()


class Logdrop(object):
    def __init__(self, path_to_script=LOGDROP_SCRIPT):
        self.logdrop = path_to_script

    def enable(self):
        self.__run('enable')

    def disable(self):
        self.__run('disable')

    def __run(self, mode):
        res = subprocess32.run(['sudo', self.logdrop, mode], timeout=5, stdout=subprocess32.PIPE, stderr=subprocess32.PIPE)
        if res.returncode != 0:
            raise Exception(res.stderr)


def sky_run_cmd(use_paste=False):
    logs = Logdrop()
    logs.enable()
    time.sleep(10)
    # dmesg | egrep 'IN|OUT' | ya paste
    try:
        out, err = do_dmesg_grep(use_paste)
    except Exception as e:
        out = ''
        err = "{}".format(e)
    logs.disable()
    return out, err


def randomize_hosts(base_hostlist, limit):
    return random.sample(base_hostlist, k=min(len(base_hostlist), limit))


def do_dmesg_grep(use_paste=False):
    dmesg = subprocess32.Popen(['dmesg'], stdout=subprocess32.PIPE, stderr=subprocess32.PIPE)
    grep = subprocess32.Popen(['egrep', 'IN|OUT'], stdin=dmesg.stdout, stdout=subprocess32.PIPE, stderr=subprocess32.PIPE)
    dmesg.stdout.close()
    if use_paste:
        ya_paste = subprocess32.Popen(['ya', 'paste'], stdin=grep.stdout, stdout=subprocess32.PIPE, stderr=subprocess32.PIPE)
        grep.stdout.close()
        app = ya_paste
    else:
        app = grep
    try:
        out, err = app.communicate(timeout=30)
    except subprocess32.TimeoutExpired:
        app.kill()
        out, err = app.communicate()
    return out, err


def get_some_drops(host_list, rand_limit=5, use_paste=False):
    rand_hosts = randomize_hosts(host_list, rand_limit)
    print('Working on hosts: {}\n'.format(rand_hosts))
    with Client() as cqueue:
        with cqueue.run(rand_hosts, sky_run_cmd, params=(use_paste,)) as session:
            for host, result, err in session.wait():
                result_err = None
                if result and result[1]:
                    result_err = result[1].strip()
                if err or result_err:
                    print('Error on {0}: `{1}, {2}`'.format(host, err, result_err), file=sys.stderr)
                else:
                    print('{0:<10} {1}'.format(host, result[0].strip()))


def parse_args():
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument('--limit', type=int, default=LIMIT, help='Limit of events')
    parser.add_argument('--service', type=str, default='hbf-drop-buckets', help='service name')
    parser.add_argument('--tags', type=str, default='output_reject_60', help='tags, comma separated')
    parser.add_argument('--rand-limit', type=int, default=5, help='work on <n> random hosts')
    parser.add_argument('--paste', default=False, help='export result to paste.y-t.ru', action='store_true')
    return parser.parse_args()


def main():
    args = parse_args()
    tags = args.tags.split(',')
    jc = JugglerClient()
    resp = jc.get_raw_events(args.service, tags, args.limit)
    if not resp:
        sys.exit('No response data')

    hosts = set()
    for i in resp['items']:
        hosts.add(i['host'])

    print('Total hosts with this service/tags combination: {}'.format(resp['total']))
    if not hosts:
        sys.exit()
    get_some_drops(hosts, rand_limit=args.rand_limit, use_paste=args.paste)
