import time

import datetime
import hashlib
import os
import debian.debfile


def get(d, k):
    try:
        return d[k]
    except KeyError:
        return None
    except Exception:
        raise


CHANGES_TEMPLATE = '''\
Format: 1.7
Date: {date}
Source: {source}
Binary: {binaries}
Architecture: {archs}
Version: {version}
Maintainer: {mnt}
Description:
{descr}
Checksums-Sha256:
{sha256}
Checksums-Sha1:
{sha1}
Files:
{files}
Changed-By: {changed_by}
Changes:
 {source} ({version}) unstable; urgency=medium
 .
   * added upstream version\n'''
CHANGES_DISTRO_TEMPLATE = '''\
Format: 1.7
Date: {date}
Source: {source}
Binary: {binaries}
Architecture: {archs}
Version: {version}
Maintainer: {mnt}
Description:
{descr}
Checksums-Sha256:
{sha256}
Checksums-Sha1:
{sha1}
Files:
{files}
Changed-By: {changed_by}
Changes:
 {source} ({version}) unstable; urgency=medium
 .
   * added upstream version
Distribution: {distro}\n'''


class DebFile:
    def __init__(self, path):
        with open(path, 'rb') as f:
            md5 = hashlib.md5()
            sha1 = hashlib.sha1()
            sha256 = hashlib.sha256()
            data = f.read(4096)
            while data:
                md5.update(data)
                sha1.update(data)
                sha256.update(data)
                data = f.read(4096)
            self.size = f.tell()
            self.md5 = md5.hexdigest()
            self.sha1 = sha1.hexdigest()
            self.sha256 = sha256.hexdigest()
        deb = debian.debfile.DebFile(path)
        control = deb.debcontrol()
        self.binary = control['Package']
        self.arch = control['Architecture']
        self.version = control['Version']
        self.mnt = control['Maintainer']
        self.desc = control['Description'].split('\n')[0]
        self.section = get(control, 'Section') or '-'
        self.priority = get(control, 'Priority') or '-'
        self.name = os.path.basename(path)

    def fmt_sha256(self):
        return ' {} {} {}'.format(
            self.sha256,
            self.size,
            self.name
        )

    def fmt_sha1(self):
        return ' {} {} {}'.format(
            self.sha1,
            self.size,
            self.name
        )

    def fmt_file(self):
        return ' {} {} {} {} {}'.format(
            self.md5,
            self.size,
            self.section,
            self.priority,
            self.name
        )

    def fmt_desc(self):
        return ' {} - {}'.format(self.binary, self.desc)


class Changes:
    def __init__(self, source, version, changed_by, distro):
        self.source = source
        self.version = version
        self.changed_by = changed_by
        self.distro = distro
        self.files = []

    def add_deb(self, path):
        df = DebFile(path)
        self.files.append(df)

    def _get_archs(self):
        tmp = set()
        for f in self.files:
            tmp.add(f.arch)
        if 'all' in tmp and 'any' in tmp:
            return 'any', 'all'
        if 'any' in tmp:
            return 'any',
        if 'all' in tmp:
            tmp.remove('all')
            return sorted(tmp) + ['all']
        return sorted(tmp)

    def dump(self, path):
        self.files.sort(key=lambda x: x.binary)
        tdict = {
            'date': datetime.datetime.now().strftime("%a, %d %B %Y %X {}").format(time.tzname[0]),
            'source': self.source,
            'version': self.version,
            'changed_by': self.changed_by,
            'mnt': self.files[0].mnt,

            'binaries': ' '.join(sorted(d.binary for d in self.files)),
            'archs': ' '.join(self._get_archs()),
            'descr': '\n'.join(f.fmt_desc() for f in self.files),
            'sha256': '\n'.join(f.fmt_sha256() for f in self.files),
            'sha1': '\n'.join(f.fmt_sha1() for f in self.files),
            'files': '\n'.join(f.fmt_file() for f in self.files),
        }
        tmpl = CHANGES_TEMPLATE
        if self.distro:
            tdict['distro'] = self.distro
            tmpl = CHANGES_DISTRO_TEMPLATE
        with open(path, 'w+') as f:
            f.write(tmpl.format(**tdict))
