import re


class Package:
    def __init__(self, name):
        self.name = name
        self.versions = dict({})

    def addVersion(self, dpkg):
        if dpkg['Version'] not in self.versions:
            self.versions[dpkg['Version']] = Version(dpkg['Version'])
        self.versions[dpkg['Version']].fileName = dpkg['Filename']
        self.versions[dpkg['Version']].md5 = dpkg['MD5sum']
        if 'Source' in dpkg:
            self.versions[dpkg['Version']].parent = dpkg['Source']
            self.versions[dpkg['Version']].addDepends(dpkg['Source'])
        if 'Depends' in dpkg:
            self.versions[dpkg['Version']].addDepends(dpkg['Depends'])
        if 'Pre-Depends' in dpkg:
            self.versions[dpkg['Version']].addDepends(dpkg['Pre-Depends'])

    def addChanges(self, ver, child):
        if ver not in self.versions:
            self.versions[ver] = Version(ver)
        self.versions[ver].addChanges(child)

    def list(self):
        return self.versions.keys()

    def getDependsByVersion(self, v):
        if v in self.versions:
            return self.versions[v].depends
        return ''

    def getChangesByVersion(self, v):
        if v in self.versions:
            return self.versions[v].changes
        return ''

    def getFileName(self, v):
        if v in self.versions:
            return self.versions[v].fileName
        return ''

    def getMD5Sum(self, v):
        if v in self.versions:
            return self.versions[v].md5
        return ''

    def getParent(self, v):
        if v in self.versions:
            return self.versions[v].parent
        return ''


class Version:
    def __init__(self, ver):
        self.version = ver
        self.parent = None
        self.changes = set([])
        self.depends = set([])
        self.fileName = None
        self.md5 = None

    def addChanges(self, c):
        self.changes.add(c)

    def addDepends(self, dp):
        if isinstance(dp, str):
            self.depends.add(dp)
            return
        for d in dp:
            self.depends.add(d)
        return


class DPKG:
    metaKeys = {'Package', 'Source', 'Version', 'Depends', 'Filename', 'MD5sum', 'Pre-Depends'}

    def __init__(self):
        self.packages = dict({})

    def parseInfoDepends(self, d):
        dpkgs = set([])
        for dpkg1 in d.split(','):
            for dpkg2 in dpkg1.split('|'):
                m = re.match(r'\s{,1}([^\s]*).*', format(dpkg2))
                p = m.group(1)
                dpkgs.add(p)
        return dpkgs

    def parseInfo(self, info):
        dpkg = dict()
        for s in info.split('\n'):
            for k in self.metaKeys:
                m = re.match(r'^{}:\s(.*)$'.format(k), s)
                try:
                    v = m.group(1)
                except:
                    continue
                if k == 'Source':
                    v = v.split(' ')[0]
                if k in ['Depends', 'Pre-Depends']:
                    dpkg[k] = self.parseInfoDepends(v)
                else:
                    dpkg[k] = v
        return dpkg

    def add(self, info):
        dpkg = self.parseInfo(info)
        if dpkg['Package'] not in self.packages:
            self.packages[dpkg['Package']] = Package(dpkg['Package'])
        self.packages[dpkg['Package']].addVersion(dpkg)
        if 'Source' not in dpkg:
            return self.packages[dpkg['Package']]
        if dpkg['Source'] not in self.packages:
            self.packages[dpkg['Source']] = Package(dpkg['Source'])
        self.packages[dpkg['Source']].addChanges(dpkg['Version'], dpkg['Package'])
        return self.packages[dpkg['Package']]

    def isParentByVersion(self, name, version):
        return self.getParentByVersion(name, version) is None

    def existByVersion(self, name, version):
        return self.getFileName(name, version) != ''

    def list(self):
        return self.packages.keys()

    def getName(self, name):
        if name in self.packages:
            return self.packages[name].name
        return ''

    def getParentByVersion(self, name, version):
        if name in self.packages:
            return self.packages[name].getParent(version)
        return ''

    def listVersions(self, name):
        if name in self.packages:
            return self.packages[name].list()
        return ''

    def listDepends(self, name, version):
        if name in self.packages:
            return self.packages[name].getDependsByVersion(version)
        return ''

    def listChanges(self, name, version):
        if name in self.packages:
            return self.packages[name].getChangesByVersion(version)
        return ''

    def getFileName(self, name, version):
        if name in self.packages:
            return self.packages[name].getFileName(version)
        return ''

    def getMD5Sum(self, name, version):
        if name in self.packages:
            return self.packages[name].getMD5Sum(version)
        return ''
