#! /usr/bin/env python3
import sys

import argparse
import bz2
import gzip
import hashlib
import logging
import os
import os.path
import shutil
import subprocess
import urllib3
from functools import partial

from libs import changes
from libs import juggler
from libs.config import Config
from libs.dpkg import DPKG
from libs.packages import Packages

PACKAGES_PATH = './TMP'
SRC_PACKAGES_FILE = '{}/_UPSTREAM_RELEASE_Packages.gz'.format(PACKAGES_PATH)
DST_PACKAGES_FILE = '{}/_MIRROR_RELEASE_Packages.gz'.format(PACKAGES_PATH)

log = logging.getLogger('yandex-mirroring-upstream')

dpkgSRC = None
dpkgDST = None
dpkgForDownloads = None


def initLoggers(level=logging.DEBUG):
    log_format = "%(asctime)s - %(filename)s:%(lineno)d - %(funcName)s() - %(levelname)s - %(message)s"
    date_fmt = "%Y-%m-%d %H:%M:%S"
    logger_handler = logging.StreamHandler()
    formatter = logging.Formatter(log_format, date_fmt)
    logger_handler.setFormatter(formatter)
    log.setLevel(level)
    log.addHandler(logger_handler)


def md5sum(fileName):
    with open(fileName, mode='rb') as f:
        d = hashlib.md5()
        for buf in iter(partial(f.read, 128), b''):
            d.update(buf)
    return d.hexdigest()


def addForDownloadWithDepends(packet):
    if packet not in dpkgForDownloads:
        dpkgForDownloads[packet] = set([])
    versions = dpkgSRC.listVersions(packet)
    for version in versions:
        if version in dpkgForDownloads[packet]:
            continue
        dpkgForDownloads[packet].add(version)
        dependPackets = dpkgSRC.listDepends(packet, version)
        for dependPacket in dependPackets:
            addForDownloadWithDepends(dependPacket)


def downloadDEB(url, dirPath, md5):
    try:
        os.mkdir(dirPath)
    except FileExistsError:
        pass
    if url.split('/')[-1] == 'None':
        return True
    if not downloadFile(url, '{}/{}'.format(dirPath, url.split('/')[-1])):
        return False
    if md5sum('{}/{}'.format(dirPath, url.split('/')[-1])) != md5:
        log.error('md5sum isn\'t correct for %s by %s.', url.split('/')[-1], url)
        return False
    log.debug('%s/%s downloaded from %s.', dirPath, url.split('/')[-1], url)
    return True


def downloadFile(url, filePath):
    http = urllib3.PoolManager()
    r = http.request('GET', url, preload_content=False)
    if r.status != 200:
        log.error('%s return %s.', url, r.status)
        return False
    with open(filePath, "wb") as f:
        f.write(r.read())
    log.debug('downloaded: %s', url)
    return True


def generateChangesfile(path, packetName, version, email, distribution=None):
    ch_file = changes.Changes(packetName, version, email, distribution)
    for i in os.listdir(path):
        if i.endswith('.deb'):
            pp = os.path.join(path, i)
            if os.path.isfile(pp):
                ch_file.add_deb(pp)
    ch_path = '{}/{}_{}.changes'.format(path, packetName, version)
    ch_path = ch_path.replace(':', '_colon_')
    ch_file.dump(ch_path)
    return True


def signChangesfile(path, fileName, email):
    cmd = 'debsign -e "{}" "{}.changes"'.format(email, fileName)
    cmd = cmd.replace(':', '_colon_')
    if subprocess.call(cmd, cwd=path, shell=True, stdout=subprocess.DEVNULL) == 0:
        return True
    return False


def uploadChangesfile(path, fileName, uploadTo):
    cmd = 'dupload --to {} --nomail "{}.changes"'.format(uploadTo, fileName)
    cmd = cmd.replace(':', '_colon_')
    if subprocess.call(cmd, cwd=path, shell=True) == 0:
        return True
    return False


def cleanPath(pathName):
    try:
        shutil.rmtree(pathName)
    except FileNotFoundError:
        pass


def createDir(pathName):
    try:
        os.mkdir(pathName)
    except FileExistsError:
        pass


def get_codec(f):
    if f.endswith('.gz'):
        return gzip.open
    elif f.endswith('.bz2'):
        return bz2.open
    else:
        return open


def run(config):
    global dpkgSRC
    global dpkgDST
    global dpkgForDownloads
    if 'tmp-dir' in config:
        PACKAGES_PATH = config['tmp-dir']
        SRC_PACKAGES_FILE = '{}/_UPSTREAM_RELEASE_Packages.gz'.format(PACKAGES_PATH)
        DST_PACKAGES_FILE = '{}/_MIRROR_RELEASE_Packages.gz'.format(PACKAGES_PATH)

    try:
        createDir(PACKAGES_PATH)
        for mirror in config['mirrors']:
            dpkgSRC = DPKG()
            dpkgDST = DPKG()
            dpkgForDownloads = {}
            log.debug('scanning packages for mirror: %s', mirror)

            # Parse a SRC Packages.gz
            rel = 0
            for release in mirror['upstream-release']:
                rel += 1
                if not downloadFile('{}/{}'.format(mirror['upstream-url'], release),
                                    "{}{}.gz".format(SRC_PACKAGES_FILE, rel)):
                    return 1
                dpkgScan = Packages("{}{}.gz".format(SRC_PACKAGES_FILE, rel), get_codec(release))
                while True:
                    dpkgInfo = dpkgScan.Next()
                    if dpkgInfo == '':
                        break
                    dpkgSRC.add(dpkgInfo)
                del dpkgScan

            # Parse a DST Packages.gz
            for url in mirror['mirror-release']:
                if not downloadFile(url, '{}{}.gz'.format(DST_PACKAGES_FILE, url.split('/')[-2])):
                    return 1
                dpkgScan = Packages('{}{}.gz'.format(DST_PACKAGES_FILE, url.split('/')[-2]), get_codec(url))
                while True:
                    dpkgInfo = dpkgScan.Next()
                    if dpkgInfo == '':
                        break
                    dpkgDST.add(dpkgInfo)
                del dpkgScan

            # Create a list with packages for downloading.
            for package in config['download_packages']:
                if dpkgSRC.getName(package) == '':
                    log.warning('Skip "%s" because dosn\'t exist in upstream.', package)
                    continue
                addForDownloadWithDepends(package)

            # Search a parent package by verision for downloading ang generating changes for upload to Dist.
            for package in dpkgForDownloads:
                for version in dpkgSRC.listVersions(package):
                    if not dpkgSRC.isParentByVersion(package, version):
                        continue
                    if dpkgDST.existByVersion(package, version):
                        log.debug('Skip "%s" "%s" because it exists.', package, version)
                        continue
                    cleanPath('{}/{}'.format(PACKAGES_PATH, package))
                    # Download a parent package and create a dir for a particular version of bundle.
                    u = '{}/{}'.format(mirror['upstream-url'], dpkgSRC.getFileName(package, version))
                    d = '{}/{}'.format(PACKAGES_PATH, package)
                    if not downloadDEB(u, d, dpkgSRC.getMD5Sum(package, version)):
                        return 'failed to download "{}" to "{}"'.format(u, d)
                    # Download child packages by version as sources and validate a md5sum of packages.
                    for childPackage in dpkgSRC.listChanges(package, version):
                        u = '{}/{}'.format(mirror['upstream-url'], dpkgSRC.getFileName(childPackage, version))
                        d = '{}/{}'.format(PACKAGES_PATH, package)
                        if not downloadDEB(u, d, dpkgSRC.getMD5Sum(childPackage, version)):
                            return 'failed to download "{}" to "{}"'.format(u, d)
                    # Generate a changes file for upload.
                    if not generateChangesfile('{}/{}'.format(PACKAGES_PATH, package), package, version,
                                               config['sign-email'], mirror['name']):
                        msg = 'failed to generate a chages file for "{}" "{}".'.format(package, version)
                        log.error(msg)
                        return msg
                    # Sign a package.
                    if not signChangesfile('{}/{}'.format(PACKAGES_PATH, package), '{}_{}'.format(package, version),
                                           config['sign-email']):
                        msg = 'failed to sign a chages file for "{}" "{}".'.format(package, version)
                        log.error(msg)
                        return msg
                    # Upload package to dist.
                    if not uploadChangesfile('{}/{}'.format(PACKAGES_PATH, package), '{}_{}'.format(package, version),
                                             config['upload-to']):
                        msg = 'failed to upload a chages file for "{}" "{}".'.format(package, version)
                        log.error(msg)
                        return msg
                    cleanPath('{}/{}'.format(PACKAGES_PATH, package))
        return None
    except Exception as e:
        msg = 'unexpected exception: {}'.format(str(e))
        log.exception(msg)
        return msg


def main():
    initLoggers()
    parser = argparse.ArgumentParser(description='Script for mirroring upstream packages.')
    parser.add_argument('-c', '--conf', dest='config_file', required=True, type=str, help="Config file in JSON format.")
    args = parser.parse_args()
    config = Config(args.config_file)
    if config is None:
        log.fatal('failed to load config from {}'.format(args.config_file))
        sys.exit(1)
    err = run(config)
    checks = [juggler.Check('mirror-{}'.format(config['upload-to']), 'CRIT' if err else 'OK', err or 'ok')]
    jerr = juggler.push(checks)
    if jerr:
        log.error('failed to push status to juggler: {}'.format(jerr))
    if err:
        sys.exit(1)
    else:
        sys.exit(0)


if __name__ == "__main__":
    main()
