package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"os"
	"sort"
	"syscall"
	"time"
	"unsafe"
)

// This URL is suggested by efmv@
const jugglerURL = "http://juggler-api.search.yandex.net/v2/events/get_raw_events"
const ioctlReadTermios = syscall.TCGETS

type Filter struct {
	Host string `json:"host"`
}

type GetRawEvents struct {
	Filters  []Filter `json:"filters"`
	Statuses []string `json:"statuses"`
	Limit    int      `json:"limit"`
	Offset   int      `json:"offset"`
}

type CheckInfo struct {
	Service      string  `json:"service"`
	ReceivedTime float64 `json:"received_time"`
	Instance     string  `json:"instance"`
	Description  string  `json:"description"`
}

type RawEvents struct {
	Items []CheckInfo            `json:"items"`
	X     map[string]interface{} `json:"-"` // Rest of the fields should go here.
}

type ByName []CheckInfo

func (a ByName) Len() int           { return len(a) }
func (a ByName) Swap(i, j int)      { a[i], a[j] = a[j], a[i] }
func (a ByName) Less(i, j int) bool { return a[i].Service < a[j].Service }

func IsTerminal(fd uintptr) bool {
	var termios syscall.Termios
	_, _, err := syscall.Syscall6(syscall.SYS_IOCTL, fd, ioctlReadTermios,
		uintptr(unsafe.Pointer(&termios)), 0, 0, 0)
	return err == 0
}

func IsSSH() bool {
	_, ok := os.LookupEnv("SSH_TTY")
	return ok
}

func fetch(hostname string) ([]byte, error) {
	c := http.Client{
		Timeout: time.Second * 3,
	}
	j := GetRawEvents{
		Filters: []Filter{
			{Host: hostname},
		},
		Statuses: []string{
			"CRIT",
		},
	}
	buf, err := json.Marshal(j)
	if err != nil {
		return nil, err
	}
	req, err := http.NewRequest("POST", jugglerURL, bytes.NewBuffer(buf))
	if err != nil {
		return nil, err
	}
	req.Header.Set("Content-Type", "application/json")
	req.Header.Set("Accept", "application/json")
	resp, err := c.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("bad response code: %d", resp.StatusCode)
	}
	buf, err = ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, fmt.Errorf("failed to read response: %s", err)
	}
	return buf, nil
}

func main() {
	if !IsTerminal(os.Stdout.Fd()) {
		return
	}
	if !IsSSH() {
		return
	}
	hostname, err := os.Hostname()
	if err != nil {
		panic(err.Error())
	}
	// Allow overriding hostname for tests
	if len(os.Args) == 2 {
		hostname = os.Args[1]
	}
	buf, err := fetch(hostname)
	if err != nil {
		fmt.Println("WARNING: failed to fetch CRIT errors:", err)
		return
	}
	m := RawEvents{}
	err = json.Unmarshal(buf, &m)
	if err != nil {
		fmt.Println("Failed to parse juggler response:", err)
		fmt.Println(m)
		return
	}
	if len(m.Items) == 0 {
		return
	}
	sort.Sort(ByName(m.Items))
	fmt.Printf("\n%-35s%-20s%-30s%s\n", "Name:", "Status:", "Time:", "Description:")
	for _, check := range m.Items {
		t := time.Unix(int64(check.ReceivedTime), 0)
		fmt.Printf("\x1b[32m%-35s\x1b[0m\x1b[31m%-20s\x1b[0m%-30s%s\n\n",
			check.Service+"/"+check.Instance,
			"CRIT",
			t.Format("Mon Jan 2  15:04:05 2006"),
			check.Description,
		)
	}
}
