#!/bin/sh -e
#
# Script for custom backup of antivirus' MySQL database
#
# $Id$

PATH=/sbin:/bin:/usr/sbin:/usr/bin:/usr/local/sbin:/usr/local/bin
TMPDIR=/var/tmp
export PATH TMPDIR


#-- Subroutines --------------------------------------------------------

warn()
{       
	echo 1>&2 "$0: WARNING: $*"
}

err()
{       
	local _exitval

	_exitval=$1
	shift

	echo 1>&2 "$0: ERROR: $*"
	exit $_exitval
}

get_tbl_list()
{
	local _list _skip_list

	_list=$($mysql_run "SELECT table_name FROM information_schema.tables \
		WHERE table_schema = '${db}' AND engine = 'InnoDB'" | \
		grep -Ev "${exclude_re}" | tr '\n' ' ' || true)

	if [ -z "${_list}" ]; then
		err 1 "Can't get list of tables for backup !"
	fi

	_skip_list=$($mysql_run "SELECT table_name FROM information_schema.tables \
		WHERE table_schema = '${db}' AND engine != 'InnoDB'" | \
		grep -Ev "${exclude_re}" | tr '\n' ' ' || true)

	if [ -n "${_skip_list}" ]; then
		warn "Non InnoDB tables found in ${db} (${_skip_list}) !"
	fi

	echo $_list
}

create_backup()
{
	local _tbl_list _tmpfile

	_tbl_list=$(get_tbl_list)

	_tmpfile=`mktemp /tmp/leprosus.XXXXXX` || err 1 "Can't make temp file !"
	trap "rm -rf ${_tmpfile}" EXIT

	if ! ($mysqldump_run $db $_tbl_list || echo $? > ${_tmpfile}) | bzip2 -9 > $dump_file; then
		err 1 "Can't backup database $db to $dump_file (bzip2 failed)!"
	fi

	if [ -s ${_tmpfile} ]; then
		err 1 "Can't backup database $db to $dump_file (mysqldump failed)!"
	fi
}

save_backup()
{
	if ! $rsync_run -t $dump_file ${rsync_path}/${backup_name}; then
		err 1 "Can't save backup on remote server !"
	fi

	if ! mv ${dump_file} ${backup_local_dir}/${backup_name}; then
		err 1 "Can't mv \"${dump_file}\" to \"${backup_name}\" !"
	fi

	find ${backup_local_dir} -type f -ctime +${backup_count}d -delete > /dev/null 2>&1
}

clean_old()
{
	local _keep

	_keep=$($rsync_run --list-only \
		--include="/${backup_prefix}*" \
		--exclude="*" \
		${rsync_path} | \
		awk -v prefix="${backup_prefix}" -v count=${backup_count} '
			BEGIN{ n = 0 }
			$5 ~ prefix { files[n++] = $5 }
			END{
			for (i = --n; i > n - count && i >= 0; i--)
				printf "--exclude=/%s ", files[i]
			}
		'
	) || err 1 "Can't get keep-list !"

	if [ -z "$_keep" ]; then
		err 1 "Keep-list is empty !"
	fi

	if ! $rsync_run -r --delete $_keep \
		--include="/${backup_prefix}*" --exclude="*" \
		${empty_dir}/ ${rsync_path}/
	then
		err 1 "Can't clean old backups !"
	fi
}


#-- Variables ----------------------------------------------------------

db="leprosus"
exclude_re="queue_sent_.*|queue_urls_.*"

mysql_batch_cfg="/root/.my.cnf"
use_batch_cfg="--defaults-file=$mysql_batch_cfg"

mysql_opts="$use_batch_cfg -NBAe"
mysql_run="mysql $mysql_opts"

mysqldump_opts="$use_batch_cfg --single-transaction -q"
mysqldump_run="mysqldump $mysqldump_opts"

date=$(date +%Y%m%d) || err 1 "Can't get current date !"

dump_file=$(mktemp -t ${db}) || err 1 "Can't make temp file !"
chmod 0644 ${dump_file} || err 1 "Can't chmod temp file !"
arch_file="${dump_file}.bz2"

backup_local_dir="/place/mysql_backup"
backup_prefix="${db}-"
backup_name="${backup_prefix}${date}.sql.bz2"
backup_count=14

rsync_path="rsync://deposito.yandex.ru/vdb-master"
rsync_opts="--contimeout=300 --timeout=300"
rsync_run="rsync $rsync_opts"

empty_dir="/var/empty"


#-- Main ---------------------------------------------------------------

create_backup
save_backup
clean_old
