#!/bin/sh -e
#
# $Header$
#
# Script for creating backups of MySQL databases.

PATH=/bin:/sbin:/usr/bin:/usr/sbin:/usr/local/bin:/usr/local/sbin
export PATH


#-- Subroutines --------------------------------------------------------

err()
{
	local _exitval _d

	_exitval=$1
	shift

	_d=$(date "+%F %T")
	echo 1>&2 "[${_d}] ERROR: $*"
	exit $_exitval
}

verbose()
{
	local _d

	if [ -n "${verbose}" ]; then
		_d=$(date "+%F %T")
		echo "[${_d}] VERBOSE: $*"
	fi
}

usage()
{
	echo 1>&2 "Usage: ${thiscmd} -i <instance> [-v]"
	echo 1>&2 "Options are:"
	echo 1>&2 "  -i <instance>         backup instance"
	echo 1>&2 "  -v                    verbose output"

	exit 1
}

get_opts()
{
	local _opt

	while getopts "i:v" _opt; do
		case "$_opt" in
			i) instance="${OPTARG}" ;;
			v) verbose="yes" ;;
			*) usage ;;
		esac
	done

	shift $(($OPTIND - 1))

	if [ $# -ne 0 ]; then
		usage
	fi
}

check_opts()
{
	if [ -z "${instance}" ]; then
		usage
	fi
}

set_vars()
{
	eval mysql_def_file=\"\${${instance}_mysql_def_file}\"
	: ${mysql_def_file:=${default_mysql_def_file}}
	eval mysql_def_grp_sfx=\"\${${instance}_mysql_def_grp_sfx}\"
	: ${mysql_def_grp_sfx:=${default_mysql_def_grp_sfx}}
	eval mysql_db=\"\${${instance}_mysql_db}\"
	: ${mysql_db:=${default_mysql_db}}
	eval mysqldump_opts=\"\${${instance}_mysqldump_opts}\"
	: ${mysqldump_opts:=${default_mysqldump_opts}}
	eval bak_dir=\"\${${instance}_bak_dir}\"
	: ${bak_dir:=${default_bak_dir}}
	eval bak_prfx=\"\${${instance}_bak_prfx}\"
	: ${bak_prfx:=${default_bak_prfx}}
	eval bak_sfx=\"\${${instance}_bak_sfx}\"
	: ${bak_sfx:=${default_bak_sfx}}
	eval keep_num=\"\${${instance}_keep_num}\"
	: ${keep_num:=${default_keep_num}}

	bak_file="${bak_dir}/${bak_prfx}${date}${bak_sfx}"
	bak_mask="${bak_dir}/${bak_prfx}${date_mask}${bak_sfx}*"
}

get_dump()
{
	verbose "Getting dump of ${mysql_db} to ${bak_file} ..."
	if ! mysqldump --defaults-file=$mysql_def_file \
		--defaults-group-suffix=$mysql_def_grp_sfx $mysqldump_opts \
		$mysql_db > $bak_file
	then
		err 1 "Can't save dump of ${mysql_db} to ${bak_file} !"
	fi
	verbose "The dump is saved."
}

arch_dump()
{
	verbose "Compressing dump ${bak_file} ..."
	if ! bzip2 $bak_file; then
		err 1 "Can't compress ${back_file} !"
	fi
	verbose "Compressing is done."

}

clean_old_baks()
{
	local _old_baks

	_old_baks=$(ls -r1 $bak_mask | tail +$((${keep_num} + 1 )))

	if [ -n "${_old_baks}" ]; then
		verbose "Cleaning old backups by mask ${bak_mask} ..."
		if ! rm -f${verbose:+v} $_old_baks; then
			err 1 "Can't clean old backups !"
		fi
		verbose "Cleaning is done."
	else
		verbose "There are no too old backups to clean."
	fi
}

#-- Variables ----------------------------------------------------------

thiscmd=$(basename $0)
date=$(date +%Y%m%d_%H%M)
date_mask="[0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9]_[0-9][0-9][0-9][0-9]"

# Default params
default_mysql_def_file="/place/mysql/batch_config"
default_mysql_def_grp_sfx=""
default_mysql_db=""
default_mysqldump_opts="--single-transaction --master-data=2 --quick"
default_bak_dir="/var/tmp"
default_bak_prfx="${thiscmd}."
default_bak_sfx=".sql"
default_keep_num=7

# Sandbox base
sandbox_mysql_def_grp_sfx="_sandbox_master"
sandbox_mysql_db="sandbox"
sandbox_bak_dir="/backup/hosts/dbs-00"
sandbox_bak_prfx="${sandbox_mysql_db}."
sandbox_bak_sfx=".sql"

# SEPE Charts base
sepecharts_mysql_def_grp_sfx="_sepecharts_master"
sepecharts_mysql_db="sepecharts"
sepecharts_bak_dir="/backup/hosts/dbs-00"
sepecharts_bak_prfx="${sepecharts_mysql_db}."
sepecharts_bak_sfx=".sql"

# Subscription base
subscription_mysql_def_grp_sfx="_subscription_master"
subscription_mysql_db="subscription"
subscription_bak_dir="/backup/hosts/dbs-00"
subscription_bak_prfx="${subscription_mysql_db}."
subscription_bak_sfx=".sql"


#-- Main ---------------------------------------------------------------

get_opts $@
check_opts
set_vars

get_dump
arch_dump
clean_old_baks

