#!/bin/sh

# This is a simple wrapper around the jail_generator script, created to help in
# creation of several jails in one command.
# Author: Denis Korenevsky <denkoren@yandex.ru>
#
# You are free to change this script if you're think, it will help other SEARCH
# admins.
# Please, don't forget to notify searchadm@ for changes.
# Script description is available at wiki:
# http://wiki.yandex-team.ru/SEEK/Instruments/scripts/createjails

set -u

##########################
#### GLOBAL VARIABLES ####
##########################



#########################
#### LOCAL VARIABLES ####
#########################

script_name="$(basename $0)"
script_dir="$(dirname $0)"
# Long option trigger. Defines whether print treating host's FQDN or not.
long_output=false
# Protection boolean for "dry run" activation.
real_run=true
# Number of required parameters, specified after all options.
required_params=1

# Name of configuration to install
jails_config="jails-9.0"
# Path to install jails
jails_path="/place/jail/"
# Jail name template
name_template=""
# Active interface for attaching jail
active_interface=""

#########################
#### LOCAL FUNCTIONS ####
#########################

help () {
	cat <<EndOfHelp | ${PAGER:-less}

NAME
	$script_name - create several jails on host

SYNOPSIS
	$script_name [-p path] [-c config] name_template [name_template]

DESCRIPTION
	This sctipt is a wrapper to jail_generator.py, crated to help in
	creation of group of jails on host machine. It uses templates for naming
	jails.  Script extended manual is available on seek wiki cluster:
	wiki.yandex-team.ru/SEEK/Instruments/scripts/createjails
	Information about environment requirements is available at
	wiki.yandex-team.ru/SEEK/Instruments/nalivki/devel-9.0-jail

OPTIONS
	-c config
		specify the configuration name to install on jail.
		Default value: jails-9.0

	-d	debug mode. Enables verbose and debug messages.

	-h	show this help and exit

	-l	long output

	-n	dry run

	-p path
		specify the path to install jails (place to store it's root
		directories).
		Default value: /place/jail/

	-v	verbose mode. Enables verbose messages.

PARAMETERS
	name_template
		jail name template. Consists of alphanumeric symbols and
		shell-style substitution class (e.g. [0-9], [a-z], etc.)

EXAMPLES
	Create 4 jails on host machine leon3, named leon30, leon31, leon32,
	leon33, placed on /place/jail with jails-9.0 configuration
	configuration installed:
	    $script_name leon3[0-3]

	Create 4 jails on host machine zamora1, named zamora10, zamora11,
	carandiru1, carandiru2:
	    $script_name zamora1[0-1] carandiru[1-2]

EXIT CODES
	0	all ok.
	64	errors in program usage. Not enough parameters, bad flag, etc.
	70	errors in environment, that will cause execution errors on jail
		generation.
	71	can't alias jail's IP on active interface.


BSD				March 15, 2013				    BSD

EndOfHelp
}

# Print a short script usage description
reference ()
{
	cat <<EndOfReference

	$script_name requires ${required_params} parameters
	    $script_name name_template

	For verbose help use -h flag:
	    $script_name -h

EndOfReference
}

. "${script_dir}/messages.sh"
. "${script_dir}/functions.sh"
. "/etc/ya.subr"

parse_template() {
	# Local variables definition
	local _template

	# Local variables initialization
	_template=${1}

	if printf "${_template}\n" | grep -q -v "\["; then
		debug "${_template} is not a template." \
		    "Interpreting as jail hostname."
		printf "${_template}\n"
	else
		debug "Generating jail names from template ${_template}."
		for i in $(jot -w %c - A Z; jot -w %c - a z; jot - 0 9); do
			printf "${_template}\n" | sed "s/\[.*\]/$i/"
		done | grep "${_template}"
	fi
}

#################################################
#### TESTING OPTIONS, CONFIGURING CURREN RUN ####
#################################################

while getopts "c:dhvlnp:" opt; do
	case $opt in
	c)
		jails_config=${OPTARG}
		;;
	d)
		msg_verbose_flag=true
		msg_debug_flag=true
		;;
	h)
		help
		exit
		;;
	v)
		msg_verbose_flag=true
		;;
	l)
		long_output=true
		;;
	n)
		real_run=false
		;;
	p)
		jails_path=${OPTARG}
		;;
	\:)
		critical 64 "Option -${OPTARG} requires an argument"
		;;
	\?)
		reference
		exit 64
		;;
	esac
done

# Checking our privileges before any other actions.
if [ "$(id -u)" -ne "0" ]; then
	verbose "Need administrative privileges. Running sudo..."
	sudo -E ${0} ${@}
	exitcode=$?
	exit ${exitcode}
fi

shift $(($OPTIND - 1))

if [ "$#" -lt "$required_params" ]; then
	reference
	exit 64
fi

#######################
#### SCRIPT'S BODY ####
#######################

! $real_run \
    && printf "${_msg_LRed}Dry run. Jail installation will not be started.${_msg_Normal}\n"

verbose "Performing environment checks..."
checks_ok=true

debug "Checking python modules..."
if (python -c "import json";
    python -c "import ldap";
    python -c "import netaddr";
    ) 2>&1 | grep "ImportError"; then
	checks_ok=false
	error "One of required python modules was not installed or" \
	    "can not be found."
fi

debug "Checking folders..."
if [ ! -L "/etc/dhcpldap" ]; then
	checks_ok=false
	error "'/etc/dhcpldap' directory not exists." \
	    "'ln -s /place/nalivki/dhcpldap /etc/' possibly" \
	    "should be executed."
fi

if [ ! -L "/etc/rclib" ]; then
	checks_ok=false
	error "'/etc/rclib' directory not exists." \
	    "'ln -s /place/nalivki/rclib /etc/' possibly" \
	    "should be executed."
fi

if [ ! -d "${jails_path}" ]; then
	checks_ok=false
	error "${jails_path} directory not exists."
fi

if [ ! -d "/etc/rc.conf.d/jails/" ]; then
	checks_ok=false
	error "'/etc/rc.conf.d/jails' directory not exists."
fi

if ${checks_ok}; then
    debug "All checks has been performed successfully."
else
	if [ ! -d "/place/nalivki" ]; then
		warning "Directory '/place/nalivki' not exists." \
		    "You possibly should clone it from git repository to" \
		    " '/place' directory: 'sudo git clone" \
		    "ssh://${SUDO_USER:-'your_user'}@git.yandex.ru/search-admin/nalivki'"
	fi

    critical 70 "Some checks were failed. Look for messages above."
fi

# Preparing for ya.subr function usage.
set +u
active_interface="$(ya_interface_get_active)"
# Used ya.subr, return restrictions.
set -u

if [ -z "${active_interface}" ]; then
	critical 70 "Can't get active interface name"
else
	verbose "Active interface to attach jails is ${active_interface}"
fi

for name_template in "${@}"; do
	for jail_name in $(parse_template ${name_template}); do
		debug "Getting jail's ${jail_name} IP address."
		jail_ip="$( fn_get_host_ip4 ${jail_name} )"
		exitcode=$?

		if [ "${exitcode}" -ne "0" ] \
		    || ! fn_is_real_host "${jail_name}"; then
			critical 68 "No DNS record found for ${jail_name} jail."
		else
			debug "${jail_name}'s IP address is ${jail_ip}"
			debug "Configuring interface ${active_interface} for" \
			    "new jail ${jail_name}."
			${real_run} && ifconfig ${active_interface} \
			    ${jail_ip} \
			    netmask 255.255.255.255 \
			    alias \
			    && sleep 1
			exitcode=$?
			${real_run} && test "${exitcode}" -ne "0" \
			    && critical 71 "Can't alias ${jail_name}'s IP" \
			    "${jail_ip} to interface ${active_interface}."

			verbose "Creating /etc/fstab.${jail_name} file."
			${real_run} && printf "" > /etc/fstab.${jail_name}

			debug "Checking /fasthd availability."
			if [ -e "/fasthd" ]; then
				debug "/fasthd found. Adding jail mount option."
				${real_run} && \
				    printf "fasthd\t/place/jail/leon43/fasthd\tnullfs\trw\t\t0\t0\n" > /etc/fstab.${jail_name}
			fi

			verbose "Starting ${jail_name} creation."
			${real_run} && "${script_dir}/jail_generator.py" \
			    -p ${jails_path} \
			    -c ${jails_config} \
			    -n ${jail_name}

			verbose "Checking if ${jail_ip} is already in" \
			    "'/ect/rc.conf.local'."
			if grep "ipaddr" /etc/rc.conf.local \
			    | grep -q -v "${jail_ip}"; then
				debug "Adding ${jail_name}'s IP address to " \
				    "ipaddr list in /etc/rc.conf.local."
				sed -a -n "/ipaddr=/s/\"$/ ${jail_ip}\"/;w /tmp/rc.conf.local.tmp" /etc/rc.conf.local
				exitcode=$?
				if [ "${exitcode}" -eq "0" ]; then
					debug "Saving configuration (new IP" \
					    "${jail_ip} has been added) to" \
					    "'/etc/rc.conf.local'."
					${real_run} \
					    && mv /tmp/rc.conf.local.tmp \
					    /etc/rc.conf.local
				fi
			fi

		fi
	done
done
