#!/usr/bin/env python
#
# This script parses evlogdump output and calculates
# errors by search sources. This script works
# with upper eventlog, supposedly it supports mmeta|int
# eventlogs.
#
# Currently no scripts use this script's output, so you may change it.
#
# example usage:
#   tail -c 1000000 /hol/www/logs/eventlog.U | evlogdump | eventlog_statistics.py
#

import sys, signal, time, socket
import string, re

class Parser:
    def __init__(this, stream):
        this.requests = {}
        this.subsource_names = {}
        this.statistics_by_sources = {}
        this.statistics_by_hosts = {}
        this.input_stream = stream
        this.lines_parsed = 0
        this.source_requests = 0
        this.last_sigint_time = 0

    def parse_line(this, line):
        if not line:
            return
        fields = string.split(line)
        requests = this.requests

        request_id = int(fields[1])
        if not request_id in requests:
            requests[request_id] = {}

        line_timestamp = int(fields[0])
        requests[request_id]['last_timestamp'] = line_timestamp

        action = fields[2]
        if action == 'SubSourceInit':
            source_id = int(fields[3])
            requests[request_id][source_id] = {}
        elif action == 'SubSourceConnect':
            source_id = int(fields[3])
            if source_id in requests[request_id]:
                connect_attempt = int(fields[4])
                requests[request_id][source_id][connect_attempt] = {'ip' : 'none', 'port' : 'none', 'SubSourceOk' : None }
        elif action == 'SubSourceRequest':
            source_id = int(fields[3])
            if source_id in requests[request_id]:
                connect_attempt = int(fields[4])
                assert connect_attempt in requests[request_id][source_id], 'connect_attempt %d not in requests[%d][%d]' % (connect_attempt, equest_id, source_id)
                requests[request_id][source_id][connect_attempt]['ip'] = fields[5]
                requests[request_id][source_id][connect_attempt]['port'] = fields[6]
                this.source_requests += 1
        elif action == 'SubSourceError':
            source_id = int(fields[3])
            if source_id in requests[request_id]:
                connect_attempt = int(fields[4])
                assert connect_attempt in requests[request_id][source_id], 'connect_attempt %d not in requests[%d][%d]' % (connect_attempt, equest_id, source_id)

                full_error_message = ' '.join(fields[5:])
                short_error_message = ' '.join(fields[7:-4])
                #match = re.match
                requests[request_id][source_id][connect_attempt]['error_msg'] = short_error_message
                requests[request_id][source_id][connect_attempt]['SubSourceOk'] = False

                # in case of connection error there could be no SubSourceRequest
                if requests[request_id][source_id][connect_attempt]['ip'] == 'none':
                    match = re.search('http://(\S+).yandex.ru:(\d+)', full_error_message)
                    if not match:
                        return
                    try:
                        ip = socket.gethostbyname(match.group(1) + '.yandex.ru')
                    except:
                        ip = match.group(1)
                    requests[request_id][source_id][connect_attempt]['ip'] = ip
                    requests[request_id][source_id][connect_attempt]['port'] = match.group(2)


        elif action == 'SubSourceOk':
            source_id = int(fields[3])
            if source_id in requests[request_id]:
                connect_attempt = int(fields[4])
                assert connect_attempt in requests[request_id][source_id], 'connect_attempt %d not in requests[%d][%d]' % (connect_attempt, equest_id, source_id)
                requests[request_id][source_id][connect_attempt]['SubSourceOk'] = True
        elif action == 'BaseSearchTouched':
            source_id = int(fields[-2])
            source_name = fields[-1]
            if not source_id in this.subsource_names:
                this.subsource_names[source_id] = source_name
            else:
                assert this.subsource_names[source_id] == source_name, 'tried to redefine subsorce %d name from %s to %s' % (source_id, this.subsource_names[source_id], source_name)
        elif action == 'EndOfFrame':
            # got end of frame. now time to digest acquired data
            this.digest_request_log(requests[request_id])
            del requests[request_id]

    def digest_request_log(this, parsed_request):
        #print parsed_request
        for subsource in parsed_request:
            # whoops! not a subsource
            if subsource != 'last_timestamp':
                this.update_statistics_by_sources(subsource, parsed_request[subsource])
                this.update_statistics_by_hosts(subsource, parsed_request[subsource])


    def update_statistics_by_sources(this, subsource, subsource_request):
        if not subsource in this.statistics_by_sources:
            this.statistics_by_sources[subsource] = { 'requests' : 0, 'errors' : 0, 'errors_on_instances' : {}, 'error_messages' : {} }

        for attempt in subsource_request:
            this.statistics_by_sources[subsource]['requests'] += 1
            if not subsource_request[attempt]['SubSourceOk']:
                this.statistics_by_sources[subsource]['errors'] += 1
                bad_instance = subsource_request[attempt]['ip'] + ':' + subsource_request[attempt]['port']

                if not bad_instance in this.statistics_by_sources[subsource]['errors_on_instances']:
                    this.statistics_by_sources[subsource]['errors_on_instances'][bad_instance] = 0

                this.statistics_by_sources[subsource]['errors_on_instances'][bad_instance] += 1

                error_message = subsource_request[attempt]['error_msg']

                if not error_message in this.statistics_by_sources[subsource]['error_messages']:
                    this.statistics_by_sources[subsource]['error_messages'][error_message] = 0

                this.statistics_by_sources[subsource]['error_messages'][error_message] += 1

    def update_statistics_by_hosts(this, subsource, subsource_request):
        for attempt in subsource_request:
            host = subsource_request[attempt]['ip']

            if not host in this.statistics_by_hosts:
                this.statistics_by_hosts[host] = {}
            if not subsource in this.statistics_by_hosts[host]:
                this.statistics_by_hosts[host][subsource] = { 'requests' : 0, 'errors' : 0 }

            if not subsource_request[attempt]['SubSourceOk']:
                 this.statistics_by_hosts[host][subsource]['errors'] += 1

            this.statistics_by_hosts[host][subsource]['requests'] += 1


    def show_statistics(this):
        shift = 0
        colors = {}
        if sys.stdout.isatty():
            colors['red']    = '\033[00;31m'
            colors['green']  = '\033[00;32m'
            colors['cyan']   = '\033[00;36m'
            colors['normal'] = '\033[00m'
        else:
            colors['red']    = ''
            colors['green']  = ''
            colors['cyan']   = ''
            colors['normal'] = ''


        print >> sys.stdout, 'Statistics:'
        print >> sys.stdout, this.per_source_statistics(shift + 4, colors)
        print >> sys.stdout
        print >> sys.stdout, this.per_host_statistics(shift + 4, colors)

    def per_source_statistics(this, shift, colors):
        result = ' ' * shift + 'Source statistics:\n'
        for source in this.statistics_by_sources:
           result += this.str_source_statistics(source, this.statistics_by_sources[source], shift + 4, colors)
           result += '\n'
        return result

    def str_source_statistics(this, source, stats, shift, colors):
        result = ''
        if source in this.subsource_names:
            source_name = this.subsource_names[source]
        else:
            source_name = str(source)

        result += ' ' * shift + 'Subsource ' + colors['cyan'] + source_name + colors['normal'] + ':\n'
        shift += 4

        result += ' ' * shift + 'requests: ' + str(stats['requests']) + '\n'
        error_percent = ((100.0 * stats['errors']) / stats['requests'])
        result += ' ' * shift + 'errors  : ' + str(stats['errors']) + ' (%.3f' % (error_percent) +' %)\n'

        if len(stats['errors_on_instances']) != 0:
            result += ' ' * shift + 'Top 5 instances by errors:\n'
            result += this.str_topmost_host_errors(stats['errors_on_instances'], shift + 4, colors)
        if len(stats['error_messages']) != 0:
            result += ' ' * shift + 'Top 10 error messages:\n'
            result += this.str_topmost_error_messages(stats['error_messages'], shift + 4, colors)

        return result

    def str_topmost_host_errors(this, instance_errors, shift, colors):
        result = ''
        # convert to tuple of pairs (<errors>, <ip>:<port>)
        errors_list = map(lambda x: (instance_errors[x], x), instance_errors)
        errors_list.sort(reverse = True)
        print_results = min(5, len(errors_list))

        total_errors_amount = sum(map(lambda x: instance_errors[x], instance_errors))

        for instance in errors_list[0: print_results - 1]:
            try:
                ip_address = instance[1].split(':')[0]
                hostname = socket.getfqdn(ip_address) + ':' + instance[1].split(':')[1]
            except Parser:
                hostname = instance[1]
            errors_percent = ( (100.0 * instance[0]) / total_errors_amount )
            result += ' ' * shift + '%35s: %4d (%5.2f %%)\n' % (hostname, instance[0], errors_percent)

        return result

    def str_topmost_error_messages(this, instance_errors, shift, colors):
        result = ''
        # convert to tuple of pairs (<errors>, <ip>:<port>)
        errors_list = map(lambda x: (instance_errors[x], x), instance_errors)
        errors_list.sort(reverse = True)
        print_results = min(10, len(errors_list))

        total_errors_amount = sum(map(lambda x: instance_errors[x], instance_errors))

        for instance in errors_list[0: print_results - 1]:
            errors_percent = ( (100.0 * instance[0]) / total_errors_amount )
            result += ' ' * shift + 'occurances: %4d (%5.2f %%): %s\n' % (instance[0], errors_percent, instance[1])

        return result

    def per_host_statistics(this, shift, colors):
        result = ' ' * shift + 'Top 10 hosts by overall errors:\n'
#        result += repr(this.statistics_by_hosts)
        # generate tuple of pairs in form (<error_percent>, <ip_address>)
        percent_of_errors_on_hosts = []
        for host in this.statistics_by_hosts:
            requests_to_host, errors_on_host = this.all_requests_to_host(this.statistics_by_hosts[host])
            if requests_to_host < 100:
                continue
            percent_of_errors_on_hosts += [((100.0 * errors_on_host) / requests_to_host, requests_to_host, errors_on_host, host)]

        percent_of_errors_on_hosts.sort(reverse=True)

        return_results = min(10, len(percent_of_errors_on_hosts))

        for host_stats in percent_of_errors_on_hosts[0:return_results - 1]:
            try:
                hostname = socket.getfqdn(host_stats[3])
            except:
                hostname = host_stats[3]
            result += ' ' * shift + "%35s requests: %5d  errors: %5d  error_percent: %3.3f %%\n" % ( hostname, host_stats[1], host_stats[2], host_stats[0] )

        #this.statistics_by_hosts[host][subsource]['error_messages']
        #for source in this.statistics_by_sources:
        #   result += this.str_source_statistics(source, this.statistics_by_sources[source], shift + 4, colors)
        #   result += '\n'
        return result

    def all_requests_to_host(this, stats):
        requests = sum(map(lambda x: stats[x]['requests'], stats))
        errors   = sum(map(lambda x: stats[x]['errors'], stats))
        return (requests, errors)


#    @staticmethod
    def sigint_handler(this, signum, frame):
        time_now = time.time()
        if time_now - this.last_sigint_time < 1.0:
          sys.exit()
        else:
          this.show_statistics()

    def analyze(this):
        line = this.input_stream.readline()
        signal.signal(signal.SIGINT, this.sigint_handler)
        while line:
          try:
              this.parse_line(line.strip())
          except BaseException as e:
              print "Failed parsing line '%s': %r" % (line, e)
          this.lines_parsed += 1
          line = this.input_stream.readline()

        this.show_statistics()






if __name__ == "__main__":
    parser = Parser(sys.stdin)
    parser.analyze()

