#!/bin/sh -e
#
# Script for merging master.passwd's / group's records from local files
# and NIS.
#
# $Id$

PATH=/sbin:/bin:/usr/sbin:/usr/bin:/usr/local/sbin:/usr/local/bin
export PATH

#-- Subroutines --------------------------------------------------------

err()
{
	local _exitval _d

	_exitval=$1
	shift

	if [ -s "${err_file}" ]; then
		cat $err_file || true
	fi

	_d=$(date "+%F %T")
	echo 1>&2 "[${_d}] ERROR: $*"
	exit $_exitval
}

verbose()
{
	local _d

	if [ -n "$verbose" ]; then
		_d=$(date "+%F %T")
		echo "[${_d}] VERBOSE: $*"
	fi
}

usage()
{
	echo 1>&2 "Usage: ${thiscmd} [-H] [-s seconds] [-v]"
	echo 1>&2 "Options:"
	echo 1>&2 "  -H           create \$HOME directories for users"
	echo 1>&2 "  -s seconds   sleep random time up to \"seconds\"" \
		"before running"
	echo 1>&2 "  -v           verbose output"

	exit 1
}

get_os() 
{
	if uname -a | grep -q -i linux; then
		echo -n "linux"
	elif uname -a | grep -q -i bsd; then
		echo -n "bsd"
	else
		err 1 "failed to grep linux or bsd from \"$(uname -a)\""
	fi
}

is_linux()
{
	if echo $1 | grep -q -i linux; then
		return 0
	else
		return 1
	fi
}

is_bsd()
{
	if echo $1 | grep -q -i bsd; then
		return 0
	else
		return 1
	fi
}
get_options()
{
	local _opt

	while getopts "Hs:v" _opt; do
		case "$_opt" in
			H) mk_homes="yes" ;;
			s) sleep_seconds="${OPTARG}" ;;
			v) verbose="yes" ;;
			*) usage ;;
		esac
	done

	shift $(($OPTIND - 1))

	if [ $# -ne 0 ]; then
		usage
	fi
}

check_options()
{
	if [ -n "${sleep_seconds}" ]; then
		if ! echo "${sleep_seconds}" | grep -qE '^[0-9]+$'; then
			usage
		fi
	fi
}


check_depends()
{
	local _pgrep

	if [ $(id -u) -ne 0 ]; then
		err 1 "You must be the super-user (uid 0) to use this utility."
	fi

	if [ -z "${thiscmd}" ]; then
		# See 'cleanup' subroutine for details.
		err 1 "Empty \$thiscmd is potentially dangerous in cleanup."
	fi

	if is_linux $os; then
		_pgrep=pidof
	elif is_bsd $os; then
		_pgrep=pgrep
	else
		err 1 "  Unknown os: $os"
	fi

	if [ -z "$(${_pgrep} ypbind)" ]; then
		verbose "ypbind is not running"
		exit 0 # reduce spam of failed cron jobs
	fi

	local _probe_results _retval
	_retval=0
	_probe_results="$( ypmatch wheel group )" 2>/dev/null || _retval=$?
	if [ $_retval -ge 1 ] || [ -z "${_probe_results}" ]; then
		# got trash from ypmath. aborting. no error message to reduce logging
		verbose "ypmatch failed to return any data. Aborting."
		exit 0;
	fi

}

random_sleep()
{
	local _rnd_seconds

	if [ -z "${sleep_seconds}" ]; then
		return 
	fi

	if [ $sleep_seconds -gt 0 ]; then
		_rnd_seconds=$(jot -r 1 1 ${sleep_seconds})
		verbose "Sleep ${_rnd_seconds} seconds before running."
		sleep $_rnd_seconds
	fi
}

mk_tmp_file()
{
	if ! mktemp ${tmp_prefix}XXXXXXXX; then
		err 1 "Can't make temp file !"
	fi
}

line_comment()
{
	local _text _rest_sym_count

	_text="$*"

	echo -n "${line_comment_prefix}${_text}${line_comment_suffix}"

	_rest_sym_count=$(($line_comment_length - ${#line_comment_prefix} \
		- ${#_text} - ${#line_comment_suffix}))
	
	if [ $_rest_sym_count -gt 0 ]; then
		jot -b $line_comment_symbol -s "" $_rest_sym_count
	fi
}

get_local_records()
{
	local _map_name _file _upper_delim _lower_delim _local

	_map_name="$1"

	eval _file=\$${_map_name}_file
	eval _upper_delim=\$${_map_name}_upper_delim
	eval _lower_delim=\$${_map_name}_lower_delim
	eval _local=\$${_map_name}_local

	verbose "  Get local records from ${_file}."

	if ! awk -v upper_delimiter="^${_upper_delim}" \
		-v lower_delimiter="^${_lower_delim}" '
		BEGIN { local_records = "yes"; }
		! /^\+/ {
			if ($0 ~ upper_delimiter) local_records = "no";
			if (local_records == "yes") print $0;
			if ($0 ~ lower_delimiter) local_records = "yes";
		}
		' $_file > $_local 2>> $err_file
	then
		err 1 "Can't get local records from ${_file} !"
	fi

	verbose "  Local records are saved in ${_local}."
}

get_allowed_nis_users()
{
	local _names _users _awk_opt

	if is_linux $os; then
		_awk_opt=""
		if type update-alternatives >/dev/null 2>/dev/null; then
			if ! update-alternatives --query awk >/dev/null 2>/dev/null; then
				_awk_opt=""
			else
				if update-alternatives --query awk | \
					grep -i Value: | grep -q -i mawk
				then
					_awk_opt=" ${_awk_opt} -W sprintf=1048576 "
				fi
			fi
		fi
	else
		_awk_opt=""
	fi

	_names="$*"

	verbose "    YP_MATCH for ${_names} ..."

	if ! eval $(ypmatch $_names $netgroup_map 2>> $err_file | awk ${_awk_opt} '
		BEGIN{ user_list = name_list = "" }
		{
			for (i = 1; i <= NF; i++) {
				if ($i ~ /,/) {
					gsub(/\(|\)/, "", $i);
					if (split($i, entry, ",") != 3) exit 1;
					if (! user_list) user_list = entry[2];
					else user_list = sprintf("%s %s", \
						user_list, entry[2]);
				}
				else {
					if (! name_list) name_list = $i;
					else name_list = sprintf("%s %s", \
						name_list, $i);
					name[n++] = $i;
				}
			}
		}
		END{
			printf "_users=\"%s\";", user_list
			printf "_names=\"%s\";", name_list
		}' 2>> $err_file)
	then
		err 1 "Can't get values of keys (${_names}) from NIS !"
	fi

	_allowed_nis_users="${_allowed_nis_users}${_allowed_nis_users:+ }${_users}"
	
	if [ -n "$_names" ]; then
		get_allowed_nis_users $_names
	fi
}

get_nis_records()
{
	local _map_name _map _nis _allowed_nis_users _re _filter

	_map_name="$1"

	eval _map=\$${_map_name}_map
	eval _nis=\$${_map_name}_nis

	verbose "  Get records from NIS->${_map} map."

	case "${_map_name}" in
		master_passwd|passwd|shadow)
			_allowed_nis_users=""
			get_allowed_nis_users $netgroup_name

			if [ -z "${_allowed_nis_users}" ]; then
				verbose "  Allowed NIS records for" \
					"${netgroup_name} is not found."
				return
			fi

			verbose "    Allowed NIS records:" \
				"${_allowed_nis_users}."

			# This global variable will be useful to create home
			# directories.
			allowed_nis_users="${_allowed_nis_users}"

			if is_linux $os; then
				_re="^($(echo ${_allowed_nis_users} | \
				sed -re 's,(^ +| +$),,g; s, +,|,g')):"
			elif is_bsd $os; then
				_re="^($(echo ${_allowed_nis_users} | \
				sed -Ee 's,(^ +| +$),,g; s, +,|,g')):"
			else
				err 1 "unknown os $os";
			fi

			;;
		group|gshadow)
			_re=""
			;;
		*)
			err 1 "Unknown map: ${_map_name} !"
			;;
	esac

	if is_linux $os; then
		case "${_map_name}" in
			passwd)
				_filter='awk -F: "{ print \$1\":x:\"\$3\":\"\$4\":\"\$5\":\"\$6\":\"\$7;};"'
				;;
			shadow)
				_filter='awk -F: "{ print \$1\":*:14916:0:99999:7:::\";};"'
				;;
			group)
				_filter='awk -F: "{ print \$1\":x:\"\$3\":\"\$4;};"'
				;;
			gshadow)
				_filter='awk -F: "{ print \$1\":!::\"\$4;};"'
				;;
		esac 
	else
		_filter="cat"
	fi


	if ! ypcat $_map 2>> $err_file | grep -E "${_re}" 2>> $err_file | \
		eval $_filter 2>> $err_file > $_nis
	then
		err 1 "Can't get records from NIS->${_map} !"
	fi

	verbose "  Allowed records from NIS->${_map} are saved in ${_nis}."
}

merge_records()
{
	local _map_name _local _upper_delim _lower_delim _nis _filter _merged

	_map_name="$1"

	eval _local=\$${_map_name}_local
	eval _upper_delim=\$${_map_name}_upper_delim
	eval _lower_delim=\$${_map_name}_lower_delim
	eval _nis=\$${_map_name}_nis
	eval _filter=\$${_map_name}_filter
	eval _merged=\$${_map_name}_merged

	verbose "  Merge local and NIS records (${_map_name} map)."

	if ! awk -F: '! /^(#|$)/ { printf "/^%s:/d\n", $1; }' $_local \
		> $_filter 2>> $err_file
	then
		err 1 "Can't create filter file (${_map_name} map) !"
	fi

	if [ ! -s "${_filter}" ]; then
		err 1 "Can't get list of local records " \
			"(${_map_name} map) !"
	fi
 
	# linux sed fails with space after -i
	if ! sed -f $_filter -i$bak_suffix $_nis 2>> $err_file
	then
		err 1 "Can't clean NIS records by local ones " \
			"(${_map_name} map) !"
	fi

	if ! { cat $_local && echo $_upper_delim && cat $_nis && \
		echo $_lower_delim; } > $_merged 2>> $err_file
	then
		err 1 "Can't concatenate local and NIS records " \
			"(${_map_name} map) !"
	fi

	verbose "  Merged records are saved in ${_merged} (${_map_name} map)."
}

backup()
{
	local _file _bak_file _name _d _old_backups

	_file="$1"

	_name=$(basename $_file)
	_d=$(date +%s)
	_bak_file="${backup_dir}/${_name}.${_d}"

	if ! cp -p $_file $_bak_file 2>> $err_file; then
		err 1 "Can't backup ${_file} !"
	fi

	verbose "  Backup of ${_file}."

	_old_backups=$(ls -1t ${backup_dir}/${_name}.[0-9]* | \
		tail -n +$(($backup_count + 1)))

	if ! rm -f $_old_backups 2>> $err_file; then
		err 1 "Can't clean old backups of ${_name} (${_old_backups}) !"
	fi

	verbose "  Clean old backups of ${_file}."
}

install_records()
{
	local _map_name _file _merged _chk_cmd _inst_cmd _check

	_map_name="$1"

	eval _file=\$${_map_name}_file
	eval _merged=\$${_map_name}_merged
	eval _check=\$${_map_name}_check

	if is_linux $os; then 
		if ! awk -F: '! /^(#|$)/ { print $0; }' ${_merged} \
			> ${_check} 2>> $err_file
		then
			err 1 "Can't remove comments from merged records"
		fi 

		verbose "  Comments stripped from ${_map_name} map saved" \
			"to ${_check}."
	fi


	if diff -q $_file $_merged > /dev/null 2>&1; then
		verbose "  New file is identical with old ${_file}."
		return
	fi

	case "${_map_name}" in
		master_passwd)
			_chk_cmd="pwd_mkdb -C ${_merged}"
			;;
		passwd)
			_chk_cmd="pwck -q -r ${_check}"
			;;
		shadow)
			_chk_cmd="pwck -q -r ${passwd_check} ${_check} "
			;;
		group|gshadow)
			if is_linux $os; then
				#_chk_cmd="grpck -r ${_check}"
				_chk_cmd="true"
			elif is_bsd $os; then
				_chk_cmd="chkgrp ${_merged}"
			fi
			;;
		*)
			err 1 "Unknown map: ${_map_name} !"
			;;
	esac

	if ! $_chk_cmd > /dev/null 2>> $err_file; then
		err 1 "New file has checked with errors (${_map_name} map) !"
	fi

	backup $_file

	case "${_map_name}" in
		master_passwd)
			_inst_cmd="pwd_mkdb -p ${_merged}"
			;;
		passwd)
			_inst_cmd="true" #waiting for shadow check
			;;
		shadow)
			_inst_cmd="install -o root -g shadow -m 640 \
				${_merged} ${_file} && \
				install -o root -g root -m 644 \
				${passwd_merged} ${passwd_file}"
			;;
		group)
			if is_linux $os; then
				_inst_cmd="install -o root -g root -m 644 \
					${_merged} ${_file}"
			elif is_bsd $os; then
				_inst_cmd="install -o root -g wheel -m 644 \
					${_merged} ${_file}"
			else
				err 1 "Unknown os: $os"
			fi
			;;
		gshadow)
			_inst_cmd="install -o root -g shadow -m 640 \
				${_merged} ${_file}"
			;;
		*)
			err 1 "Unknown map: ${_map_name} !"
			;;
	esac

	if ! eval $_inst_cmd 2>> $err_file; then
		err 1 "Can't install new ${_file} !"
	fi

	verbose "  New ${_file} installed."
}

mk_homes()
{
	local _user_re _uids _uid _user _home

	if [ -z "${mk_homes}" ]; then
		return 0
	fi

	if [ ! -d "${home_root}/" ]; then
		verbose "${home_root} doesn't exist, skip creating homes."
		return 0
	fi

	verbose "Start creating home directories."

	_user_re=$(echo "^(${allowed_nis_users})\$" | sed -Ee "s/ +/|/g")

	if ! eval $(getent passwd | awk -F: -v home_root="${home_root}" \
		-v user_re="${_user_re}" \
		'
		$1 ~ user_re && $6 ~ home_root {
			user = $1; uid = $3; home_dir = $6
			homes[uid] = home_dir
			users[uid] = user
			if (uid_list) uid_list = sprintf("%s %s", \
				uid_list, uid)
			else uid_list = uid
		}
		END{
			printf "_uids=\"%s\";\n", uid_list
			for (uid in users) {
				printf "local _user_%d=\"%s\";\n", \
					uid, users[uid]
				printf "local _home_%d=\"%s\";\n", \
					uid, homes[uid]
			}
		}
		')
	then
		err 1 "Can't get info to create home directories !"
	fi

	for _uid in $_uids; do
		eval _user=\"\$_user_${_uid}\"
		eval _home=\"\$_home_${_uid}\"
		if [ -d "${_home}" ]; then
			continue
		fi
		if ! install -o $_user -g $home_group -m $home_perm -d $_home
		then
			err 1 "Can't create ${_home} for ${_user} !"
		fi
		verbose "  ${_home} for ${_user} is created."
	done

	verbose "All home directories are created."
}

cleanup()
{
	# A host can be rebooted between start and stop ${thiscmd} and we
	# can lose the list of temporary files so we remove temporary
	# files by mask. Value of ${tmp_prefix} depends on ${thiscmd} so
	# we check value of ${thiscmd} in check_depends.
	rm -f ${tmp_prefix}* 2> /dev/null || true
	verbose "Cleanup done."
}


#-- Variables ----------------------------------------------------------

thiscmd=$(basename $0)

os=$(get_os)

verbose=""

line_comment_length=72
line_comment_prefix="#-- "
line_comment_suffix=" "
line_comment_symbol="-"

tmp_prefix="/etc/${thiscmd}."
bak_suffix=".bak"

if is_bsd $os; then
	map_names="master_passwd group"
	master_passwd_file="/etc/master.passwd"
	master_passwd_map="master.passwd"
	master_passwd_upper_delim=$(line_comment NIS accounts)
	master_passwd_lower_delim=$(line_comment NIS accounts end)

	group_file="/etc/group"
	group_map="group"
	group_upper_delim=$(line_comment NIS groups)
	group_lower_delim=$(line_comment NIS groups end)

elif is_linux $os; then
	map_names="passwd shadow group gshadow"

	passwd_file="/etc/passwd"
	passwd_map="passwd"
	passwd_upper_delim=$(line_comment NIS accounts)
	passwd_lower_delim=$(line_comment NIS accounts end)

	shadow_file="/etc/shadow"
	shadow_map="passwd"
	shadow_upper_delim=$(line_comment NIS accounts)
	shadow_lower_delim=$(line_comment NIS accounts end)

	group_file="/etc/group"
	group_map="group"
	group_upper_delim=$(line_comment NIS groups)
	group_lower_delim=$(line_comment NIS groups end)

	gshadow_file="/etc/gshadow"
	gshadow_map="group"
	gshadow_upper_delim=$(line_comment NIS gshadows)
	gshadow_lower_delim=$(line_comment NIS gshadows end)

fi

netgroup_map="netgroup"
netgroup_name=$(hostname -s | tr '[:lower:]' '[:upper:]')

backup_count=7
backup_dir="/var/backups"

home_root="/home"
home_group="wheel"
home_perm="755"


#-- Main ---------------------------------------------------------------

get_options $*
check_options
check_depends

verbose "Selected os=$os"

random_sleep

trap cleanup EXIT

err_file=$(mk_tmp_file)

for map_name in $map_names; do
	if is_bsd $os; then
		suffixes="_local _nis _filter _merged"
	elif is_linux $os; then
		suffixes="_local _nis _filter _merged _check"; 
	fi

	for map_suffix in $suffixes; do 
		eval "${map_name}${map_suffix}=$(mk_tmp_file)" 
	done
done

allowed_nis_users=""

for map_name in $map_names; do
	verbose "Entering into map ${map_name}."

	get_local_records ${map_name}
	get_nis_records ${map_name}
	merge_records ${map_name}
	install_records ${map_name}

	verbose "Leaving map ${map_name}."
done

mk_homes

