# Author: Denis Korenevsky <denkoren@yandex-team.ru>
# This is a simple message "library" for printing messages to STDERROR
# They can be colorized, if client's terminal can represent colors

############################
#### "PUBLIC" VARIABLES ####
############################

# Are used to change functions behaviour from parent script.

# Connection timeout. Time (in seconds) to wait, before breaking the connnection
# attempt.
: ${fn_connect_timeout:=3}

# Define sed extended regexp flag
case $(uname) in
	[Ll]inux)
		: ${fn_sed_regex:='-r'}
	;;
	[Dd]arwin)
		: ${fn_sed_regex:='-E'}
	;;
	*)
		: ${fn_sed_regex:=''}
	;;
esac

export \
    fn_sed_regex \
    fn_connect_timeout

#########################
#### LOCAL VARIABLES ####
#########################

# Check for host:port availability from current host.
# Command format:
#   fn_check_availability <host> <port>
#
# Prints nothing
# Returns 0, if port available, 1 otherwise.
fn_check_availability()
{
	local _host _port

	_host=${1}
	_port=${2}
	_exitcode=1

	if which curl >/dev/null; then
		curl -s -k --connect-timeout ${fn_connect_timeout} \
		    ${_host}:${_port} >/dev/null \
		    && return 0 \
		    || return 1
	else
		nc -z ${_host} ${_port} 1>/dev/null 2>/dev/null \
		    && return 0 \
		    || return 1
	fi

	return 1
}

# Get host's IPv4 address using DNS.
# Command format:
#   fn_get_host_ip4 <host>
#
# Prints IPv4 address to STDOUT.
# Returns 0 on successful IP getting, 1 on fail.
fn_get_host_ip4()
{
	local _host
	local _host_ip

	_host="${1}"

	_host_ip="$(
	    host -t A "${_host}" \
		| egrep -o '\b[[:digit:]]{1,3}(\.[[:digit:]]{1,3}){3}\b'
	)"

	if [ -z "${_host_ip}" ]; then
		return 1
	else
		printf -- "${_host_ip}"
	fi

	return 0
}

# Get host's IPv6 address using DNS.
# Command format:
#   fn_get_host_ip6 <host>
#
# Prints IPv6 address to STDOUT.
# Returns 0 on successful IP getting, 1 on fail.
fn_get_host_ip6()
{
	local _host
	local _host_ip6

	_host="${1}"

	_host_ip6="$(
	    host -t AAAA "${_host}" \
		| egrep -o '\b([[:alnum:]]*:)+[[:alnum:]]+\b'
	)"

	if [ -z "${_host_ip6}" ]; then
		return 1
	else
		printf -- "${_host_ip6}"
	fi

	return 0
}

# Check that <host> string is a real host (it has DNS record with IPv4 or v6
# address)
# Command format:
#   fn_is_real_host <host>
# Prints nothing
# Returns 0 for real DNS name and 1 otherwise.
fn_is_real_host()
{
	local _host _host_ip

	_host="${1}"
	_host_ip="$(fn_get_host_ip4 "${_host}" || fn_get_host_ip6 "${_host}")"

	if [ -z "${_host_ip}" ]; then
		return 1
	fi

	return 0
}

# Check the variable value is a number
# Command format:
#   fn_is_number <data>
# Prints nothing
# Returns 0 for number and 1 for non-number
fn_is_number()
{
	local _data

	_data="${1}"

	if test "${_data}" -eq "${_data}" 2>/dev/null; then
		return 0
	else
		return 1
	fi
}

# Makes first character of each word capital.
# Can work as filter.
# Command format: 
#   echo 'make some text' | fn_capitalize
# Prints capitalized text from STDIN.
# Returns 'awk' exitcode.
fn_capitalize()
{
	if [ "$#" -ne "0" ]; then
		echo "$*" \
		    | awk '{
			for(i=1;i<=NF;i++)
			    sub(/./, toupper( substr($i,1,1) ),$i)
		}1'
	else
		awk '{
		    for(i=1;i<=NF;i++)
			    sub(/./, toupper( substr($i,1,1) ),$i)
		}1'
	fi
}

# Validates FQDN given as parameter.
# Command format:
#   fn_fqdn_validation 
fn_fqdn_validate()
{
	local _fqdn

	_fqdn="${1}"

	printf -- "${_fqdn}\n" \
	    | awk -F"." '
		BEGIN {
		    is_valid=0
		}
		(length($0) < 254) && /^[[:alnum:]][[:alnum:].\-]*$/ {
		    if (NF>1) is_valid=1
		    for(N=1; N<=NF; N++) if (length($N) > 63) is_valid=0
		}
		END {
		    if (is_valid == 1) exit 0
		    else exit 1
		}'
}
