#!/bin/sh

###########################
#### SHORT DESCRIPTION ####
###########################
# This is a simple interface wrapper for getting host's active mac address using mouse@ web-interface or
# Author: Korenevskiy Denis (denkoren@)
# Based on mzelenkov@ wiki page prednalivka (http://wiki.yandex-team.ru/MixailZelenkov/prednalivka)
#
# You are free to change this script if you're think, it will help other SEARCH admins.
# Please, don't forget to notify searchadm@ for changes.

set -u

##########################
#### GLOBAL VARIABLES ####
##########################

: ${YR:="/Berkanavt/bin/scripts/yr"}
: ${YR_GROUP_DOMAIN:="${DEFAULT_DOMAIN:-"yandex.ru"}"}


#########################
#### LOCAL VARIABLES ####
#########################

script_name="$(basename $0)"
script_dir="$(dirname $0)"
# Long option trigger. Defines whether print treating host's FQDN or not.
long_output=false
# Number of required parameters, specified after all options.
required_params=1

# Einstellung farm name to request for MAC
einstellung_farm=""
# Host's MAC address, given by Golem
host_mac_golem=""
# Host's MAC address, given by Einstellung farm
host_mac_farm=""

# Script's result exit code.
exitcode="0"

#########################
#### LOCAL FUNCTIONS ####
#########################

help () {
	cat <<EndOfHelp | ${PAGER:-less}

NAME
	$script_name  - prints host's active interface mac address using
			Einstellung API or Golem API.

SYNOPSIS
	$script_name [-l] [-v] [-e einstellung_farm] host
	$script_name [-l] [-v] [-e einstellung_farm] +YR_GROUP
	$script_name -h

DESCRIPTION
	This sctipt is a simple interface wrapper for getting host's active mac
	address using mouse@ web-interface or Golem API. It uses host_query.sh
	script to work with Golem API. If einstellung farm name specified,
	script tries to get information from both of Golem and Einstellung and
	fails on different non-empty answers from them. If no Einstellung farm
	specified, script tries to get host's MAC address from Golem. On -e
	option specified it tries to get MAC from both of sources.

OPTIONS
	-d	debug mode. Enables verbose and debug messages.

	-e einstellung_farm
		Einstellung farm name. Based on DC name of host. At the moment,
		this script has been last changed next Einstellung farms were
		avalible:
		    folgaine1	- Folga-A DC
		    folgaine2	- Folga-B DC
		    ugreine2	- Ugreshka-B
		    ivaine1	- Ivanteevka-A
		    ivaine2	- Ivanteevka-B
		    vegaine	- Las-Vegas-1
		    amseine	- Amsterdam-1
		    eine	- Red Rose

	-h	show this help and exit

	-q	quiet mode. Supresses warning messages.

	-l	print host FQDN before mac address. (<FQDN> <mac>)
		usefull for DHCP Web-interface "Batch mode".

	-v	verbose mode. Enables verbose messages

PARAMETERS

	host	host's FQDN

VARIABLES
	Script uses following variables to control it's behaviour

	YR	path to YR script executable.

	YR_GROUP_DOMAIN
		YR hostgroup domain name. It is appended to the
		YR-group's hostnames.

EXIT CODES
	Exit codes correspond to sysexits.h (/usr/include/sysexits.h)
	0	all MAC addresses had been obtained successfuly.
	64	Usage error. Not enough data to run.
	69	YR executable not found.
	70	Internal script error. MAC addresses for some hosts, obtained
		from golem and einstellung are different, or one of answers is
		empty.
	76	Exchange error. MAC address for one of hosts has not been
		obtained.

BSD				August 27, 2012				    BSD

EndOfHelp

}

reference () {
	cat <<EndOfReference

	Script requires ${required_params} parameter
	    $script_name host_FQDN

	For verbose help use -h flag:
	    $script_name -h

EndOfReference
}

. "${script_dir}/messages.sh"

# Get host mac from 2 sources and compare answers.
# If no answer from one of sources, print the other without comparsion.
# If answers are different, then ERROR.
# Get 1 parameter:
#	host_fqdn. FQDN of host to obtain MAC.
get_host_mac() {

	# Local variables definition
	# Stores treating host's FQDN
	local _host_fqdn
	# Stores commands exit codes
	local _exitcode
	# Code to return after execution
	local _returncode

	# Local variables initialization
	_host_fqdn="${1}"
	_exitcode="1"
	_returncode="0"


	# Print host's FQDN in long output mode.
	$long_output && printf "${_host_fqdn} "

	#-------------------------------------------------------------
	#--- GET MAC FROM GOLEM, CHECK AND TRANSFORM TO CANONICAL ----
	#-------------------------------------------------------------

	# Try to get host MAC using Golem API. host_query.sh script is used.
	host_mac_golem=$( "${script_dir}/host_query.sh" \
	    -c primary_mac ${_host_fqdn})
	_exitcode="$?"

	# Check whether Golem MAC request was successful
	if [ "${_exitcode}" -ne "0" -o -z "${host_mac_golem}" ]; then
		# If not
		if [ -z "${einstellung_farm}" ]; then
			# If no einstellung was given, then error.
			error "Can't get ${_host_fqdn} MAC from Golem and" \
			    "Einstellung farm has not been specified."
			_returncode=76
		else
			# If einstellung farm was specified, then just WARN
			warning "Can't get ${_host_fqdn} MAC from Golem."
			# Set return code to "different MAC's"
			_returncode=70
			# and clear recieved data to stay safe
			host_mac_golem=""
		fi
	else
		debug "Transforming the answer to canonical form"\
		    "(:-delimited, lower case)..."
		host_mac_golem=$( echo ${host_mac_golem} \
		    | sed -E 's/([[:alnum:]]{2})([[:alnum:]]{2})/\1.\2/g;s/\./:/g' \
		    | tr "[[:upper:]]" "[[:lower:]]" )
		verbose "Golem: ${_host_fqdn} active interface MAC is" \
		    "${host_mac_golem}"
	fi

	#--------------------------------------------
	#--- GET MAC FROM EINSTELLUNG, AND CHECK ----
	#--------------------------------------------

	if [ -n "${einstellung_farm}" ]; then
		debug "Trying to get ${_host_fqdn} MAC from Einstellung farm..."
		debug "Requesting for http://${einstellung_farm}.yandex-team.ru/computer/get_macs.cli..."
		host_mac_farm=$( wget -q -t 1 -T 2 -O - \
		    "http://${einstellung_farm}.yandex-team.ru/computer/get_macs.cli" \
		    --post-data "id=${_host_fqdn}&mac_type=0" \
		    | tr "[[:upper:]]" "[[:lower:]]" )
		_exitcode="$?"
		
		# If request was not sucessful
		if [ "${_exitcode}" -ne "0" -o -z "${host_mac_farm}" ]; then
			# And Golem was fail
			if [ -z "${host_mac_golem}" ]; then
				error "Can't get ${_host_fqdn} MAC neither from Golem nor from Einstallung."
				_returncode=76
			else
				# The other way is just warning.
				warning "Can't get ${_host_fqdn} MAC from Einstellung."
				# Set return code to "different MAC's"
				_returncode=70
				# And clear answer from einstellung to stay safe.
				host_mac_farm=""
			fi
		else
			# Just report on succesful request.
			verbose "Einstellung: ${_host_fqdn} active interface MAC is ${host_mac_farm}"
		fi
	fi

	#-------------------------------------------------
	#--- CHECK THAT TWO ANSWERS ARE NOT DIFFERENT ----
	#-------------------------------------------------

	debug "Comparing answers..."
	if [ "${host_mac_golem}" != "${host_mac_farm}" ]; then
		# And both of "MACs" are not empty
		if [ -n "${host_mac_golem}" -a -n "${host_mac_farm}" ]; then
			warning "${_host_fqdn}'s MACs, given by Golem and Einstellung are different."\
			    "Golem: ${host_mac_golem} Einstellung: ${host_mac_farm}"
			# Set return code to "different MAC's"
			_returncode=70
		else
			if [ -z "${host_mac_golem}" ]; then
				debug "Answer from Golem is empty. Using Einstellung answer only."
				echo "${host_mac_farm}"
			else
				debug "Answer from Einstellung is empty. Using Golem answer only."
				echo "${host_mac_golem}"
			fi
		fi

	else
		# If both of "MACs" are the same, print one of them
		echo "${host_mac_golem}"
	fi

	return ${_returncode}
}

#################################################
#### TESTING OPTIONS, CONFIGURING CURREN RUN ####
#################################################

# Testing -h or flag. If exists, show help end exit.
while getopts "de:hqlv" opt; do
	case $opt in
	d)
		msg_verbose_flag=true
		msg_debug_flag=true
		;;
	e)
		einstellung_farm="${OPTARG}"
		;;
	h)
		help
		exit 0
		;;
	q)
		msg_warning_flag=false
		;;
	l)
		long_output=true
		;;
	v)
		msg_verbose_flag=true
		;;
	\:)
		critical 64 "Option -${OPTARG} requires an argument"
		;;
	\?)
		reference
		exit 64
		;;
	esac
done
shift $(($OPTIND - 1))

# Check if we have not enouth parameters and input is a tty.
if [ "$#" -lt "$required_params" ] \
    && [ -t 0 ]; then
	reference
	exit 64
fi

# If working with more then 1 hostname, then force long output mode.
if [ "$#" -gt "1" ]; then
	long_output=true
fi

# Report about startup status.
debug "Debug mode had been activated."
${long_output} && verbose "Long output mode had been activated."
test -n "${einstellung_farm}" && verbose "Einstellung farm had been specified."

#######################
#### SCRIPT'S BODY ####
#######################

# If positional parameters had been specified, treat them
if [ "$#" -ge "$required_params" ]; then
	host_fqdn="${1}"

	if expr "${host_fqdn}" : "^\+" >/dev/null; then
		if [ ! -x ${YR} ]; then
			critical 69 "YR executable not found at '${YR}'."\
			    "Please, set 'YR' environment variable to point"\
			    "yr executable."
		fi
		yr_groups="${@}"
		# yr groups contan more then one host, so force long output mode.
		long_output=true
		${YR} ${yr_groups} LIST | while read h; do
			get_host_mac ${h}.${YR_GROUP_DOMAIN}
			ret_code="$?"
			# Test result exit code priority and change on
			# something more terrible happen.
			test "${ret_code}" -gt "${exitcode}" \
			    && exitcode=${ret_code}
		done
	else
		for host_fqdn in ${@}; do
			get_host_mac ${host_fqdn}
			ret_code="$?"
			# Test result exit code priority and change on
			# something more terrible happen.
			test "${ret_code}" -gt "${exitcode}" \
			    && exitcode=${ret_code}
		done
	fi
fi

# If our input is not a terminal, working with it
if [ ! -t 0 ]; then
	long_output=true
	while read host_fqdn; do
		get_host_mac ${host_fqdn}
		ret_code="$?"
		# Test result exit code priority and change on
		# something more terrible happen.
		test "${ret_code}" -gt "${exitcode}" \
		    && exitcode=${ret_code}
	done
fi

exit ${exitcode}
