#!/bin/sh -e
#
# Script for getting content of given netgroup.
#
# $Header$


PATH=/bin:/usr/bin:/usr/local/bin
export PATH

#-- Subroutines --------------------------------------------------------

err()
{
	local _exitval

	_exitval=$1
	shift

	echo 1>&2 "ERROR: $*"
	exit $_exitval
}

usage()
{
	echo 1>&2 "Usage: ${thiscmd} [-f <netgroup>] -g <group> [-r] [-d]"
	echo 1>&2 "Options are:"
	echo 1>&2 "  -f <netgroup> alternative path to netgroup" \
		"(default: ${default_netgroup})"
	echo 1>&2 "  -g <group>    netgroup name"
	echo 1>&2 "  -r            output RE instead of a hostname list" \
		"(default: ${default_re_out:-0})"
	echo 1>&2 "  -d            print domainnames in output" \
		"(default: ${default_domain_out:-0})"
	echo 1>&2 "Example:"
	echo 1>&2 "  ${thiscmd} -f ./netgroup -g wmconsgrp -r"
	exit 1
}

get_opts()
{
	local _opt

	while getopts "f:g:rd" _opt; do
		case "$_opt" in
			f) netgroup="$OPTARG" ;;
			g) group="$OPTARG" ;;
			r) re_out=1 ;;
			d) domain_out=1 ;;
			*) usage ;;
		esac
	done

	shift $(($OPTIND - 1))

	if [ $# -ne 0 ]; then
		usage
	fi
}

handle_opts()
{
	: ${netgroup:="${default_netgroup}"}
	: ${re_out:="${default_re_out}"}

	if [ -z "${group}" ]; then
		usage
	fi

	if [ ! -f ${netgroup} ]; then
		err 1 "Can't find ${netgroup} !"
	fi
}

add_to_tmp_files()
{
	tmp_files="${tmp_files}${tmp_files:+ }$*"
}

prepare_netgroup()
{
	netgroup_one_lines=$(mktemp -t ${thiscmd})
	add_to_tmp_files $netgroup_one_lines

	if ! awk '
		/(^$|^(#|\+))/ { next }
		{
			gsub(/[\t ]+/, " ", $0)
			gsub(/(^ +| +$)/, "", $0)
			gsub(/\)\(/,") (", $0)
			if (/\\$/) {
				sub(/ *\\$/, " ", $0)
				end=""
			}
			else {
				end="\n"
			}
			printf "%s%s", $0, end
		}
		' $netgroup > $netgroup_one_lines
	then
		err 1 "Can't prepare netgroup !"
	fi
}

getgroup()
{
	if ! awk -v group="${group}" -v re_out="${re_out}" \
		-v domain_out="${domain_out}" '
		function getgroup(_group, _arr,
	       		_d_arr, _i, _entry)
			# Arguments of this function are: _group, _arr
			# Rest ones are local variables.
		{
			if (_arr[_group]) {
				split(_arr[_group], _d_arr, FS)
				for (_i in _d_arr) {
					_entry = _d_arr[_i]
					if (_entry ~ /,/) {
						gsub(/(\(|\)|,)/, "", _entry)
						if (re_out) {
							gsub(/[0-9]/, "[0-9]", _entry)
						}
						if (! domain_out) {
							sub(/'${domain}'/, "", _entry)
						}
						res[_entry] = 1
					}
					else {
						getgroup(_entry, _arr)
					}
				}
			}
			else {
				res[_group] = 1
			}
			return
		}
		{
			key = $1
			val = $0
			sub(/^[^ ]+ +/, "", val)
			ng[key] = val
		}
		END{
			getgroup(group, ng)
			for (key in res) {
				printf "%s\n", key
			}
		}
		' $netgroup_one_lines | sort
	then
		err 1 "Can't get ${group} !"
	fi
}

cleanup()
{
	if [ -n "${tmp_files}" ]; then
		rm -f ${tmp_files} >/dev/null || true
	fi
}


#-- Variables ----------------------------------------------------------

thiscmd=$(basename $0)

default_netgroup="/etc/netgroup"
default_re_out=""
default_domain_out=""

domain=".yandex.ru"


#-- Main ---------------------------------------------------------------

trap cleanup 2 3 15 EXIT 

get_opts $@
handle_opts

prepare_netgroup
getgroup

