#!/bin/sh

# This is a CLI for golem API's host_query function
# (interface for getting information about host).
# API function desctiption can be found at
# http://golem.yandex-team.ru/api/host_query.sbml
# Author: Korenevsky Denis (denkoren@yandex-team.ru)
#
# You are free to change this script if you're think, it will help other SEARCH
# admins.
# Please, don't forget to notify searchadm@ for changes.

set -u

##########################
#### GLOBAL VARIABLES ####
##########################



#########################
#### LOCAL VARIABLES ####
#########################

script_name="$(basename $0)"
script_dir="$(dirname $0)"
# Number of required parameters, specified after all options.
required_params=1

# Golem API availability flag
golem_available=false
# Host search pattern/regexp/list
search_pattern=""
# List of columns to show. ","-separated
col_list=""
# Title line with columns names.
title=""
# Show title with columns names.
show_title=false
# Defines host search pattern type
search_type="hostname_pat"
# Command for requesting Golem HTTP API. May be wget or fetch
req_command="wget -q -O -"

#########################
#### LOCAL FUNCTIONS ####
#########################

help () {
	cat <<EndOfHelp | ${PAGER:-less}

NAME
	$script_name - CLI for host_query Golem API function

SYNOPSIS
	$script_name [-v] [-d] [-c col_list] [-l|-r|-s] [-H] [-T] search_patern
	$script_name -h

DESCRIPTION
	Script trying to get information about hosts using Golem API. Detailed
	information about API function can be found at
	http://golem.yandex-team.ru/api/host_query.sbml

OPTIONS
	-c col_list
		define columns list to show. Comma-separated.
		Default value: "name,dc"
		Available values:
		    Always defined:
			id - host's golem id
			name - host's FQDN
			primary_ip - host's main IP-address
				(IP of hostname PTR)
			dc_id - host's DC golem id
			dc - host's DC name
			short_dc - host's short DC name
			short_line - short DC's line name
			rack_id - hosts's rack golem id
			rack - host's rack name (number)
			kvm - KVM name
			turn_off_priority - turn-off line

		    Can be defined only for working hosts:
			switch - host's switch FQDN.
			switch_short - host's switch name.
			switch_id - host's switch golem id.
			switch_port - host's switchport name.
			primary_mac - primary interface's MAC-address.
			primary_vlan - primary interface's VLAN.

		For exapmple:
		    -c name,dc,kvm
		will print:
		<host fqdn>	<host's dc name>	<host's KVM>
		(tab-separated)

	-d	debug mode. Enables verbose and debug messages.

	-h	show this help and exit.

	-H	shortcut for '-c name,short_line,rack,switch_short,switch_port'
		Usable for creating tasks for helpdc@.

	-l	use comma-separated host list instead of simple shell-style
		search pattern.

	-q	quiet mode. Disables printing of alarming status messages:
		    criticals
		    errors
		    warnings.

	-r	use perl-regexp instead of simple shell-style search pattern
		(mach longer).

	-s	show information about all hosts, connected to specified switch.

	-T	show title with field names before output.

	-v	verbose mode. Enables printing of workprocess reports.

PARAMETERS
	search_pattern
		hostname search pattern:
		    1. simple shell-style search pattern
		       (if no -r, -l or -s option specified).
			Example:
			    leon[0-4][0,1,2,3].yandex.ru
		    2. perl-regexp host search pattern
		       (if -r option specified).
			Example:
			    leon[[:digit:]][0-3].yandex.ru
		    3. comma-separated list of hosts
		       (if -l option specified).
			Example:
			    leon1.yandex.ru,carandiru1.yandex.ru
		    4. switch FQDN.
		       (if -s option specified).

EXIT CODES
	0 on success

	1, when Golem API return's error message.

	not 0 on error (exit code conform to wget exit code)

BSD				 August 24, 2012			    BSD

EndOfHelp
}

# Print a short script usage description
reference ()
{
	cat <<EndOfReference

	$script_name requires ${required_params} parameters
	    $script_name search_pattern

	For verbose help use -h flag:
	    $script_name -h

EndOfReference
}

set_col_list ()
{
	col_list=${1}
	title="$(printf -- "${col_list}" \
	    | sed "s/,/	/g;
	    s/name/name/g;
	    s/switch_short/sw/g;
	    s/switch/sw/g;
	    s/short_line/line/g")"
}

. "${script_dir}/messages.sh"
. "${script_dir}/functions.sh"

set_col_list "name,dc"

#################################################
#### TESTING OPTIONS, CONFIGURING CURREN RUN ####
#################################################

while getopts "c:dhHlqrsTv" opt; do
	case $opt in
	c)
		set_col_list "${OPTARG}"
		;;
	d)
		msg_verbose_flag=true
		msg_debug_flag=true
		;;
	h)
		help
		exit 0
		;;
	H)
		set_col_list "name,short_line,rack,switch_short,switch_port"
		;;
	l)
		search_type="hostname_list"
		;;
	q)
		msg_error_flag=false
		msg_critical_flag=false
		;;
	r)
		search_type="hostname_re"
		;;
	s)
		search_type="switch"
		;;
	T)
		show_title=true
		;;
	v)
		msg_verbose_flag=true
		;;
	\:)
		critical 1 "Option -${OPTARG} requires an argument"
		;;
	\?)
                error "Unknown parameter given."
		reference
		exit 1
		;;
	esac
done
shift $(($OPTIND - 1))

if [ "$#" -lt "$required_params" ]; then
	reference
	exit 1
fi

#######################
#### SCRIPT'S BODY ####
#######################

search_pattern="${1}"

# Defining utils to use for requesting golem
case "$(uname)" in
[Ll]inux*)
	req_command="wget -q -O -"
	;;
[Dd]arwin)
	req_command="curl -s"
	;;
*)
	req_command="fetch -q -o -"
	;;
esac

debug "Checking Golem API availability..."
fn_check_availability ro.admin.yandex-team.ru 80 \
    && golem_available=true

if ${golem_available}; then
	debug "Golem API is available."
else
	critical 1 "Golem API is not available."
fi

verbose "Requesting Golem for host's information." \
    "Following fields will be requested: ${col_list}."
${show_title} && printf "${title}\n"
request_result="$(
    ${req_command} \
	"http://ro.admin.yandex-team.ru/api/host_query.sbml?${search_type}=${search_pattern}&columns=${col_list}"
    )"
exitcode=$?

if echo "${request_result}" | grep -iq "error"; then
	critical 1 "Golem returned error message: '${request_result}'."
else
	/bin/echo "${request_result}"
fi

exit ${exitcode}
