#!/bin/sh

usage()
    {
    echo "Usage: ./`basename $0` start|stop [ad|da|ada][0-9]"
    }

exit_error()
    {
    echo
    echo "Error occured, check output above."
    echo
    exit 1
    }

get_prov_name()
    {
    local _mirror _prov_name _prov_diskn
    _mirror=$1

    # get existing 'prov' name from mirror, and replace its disk number to DISKN
    _prov_name=`gmirror status -s ${_mirror} | awk '{print $3 ; exit 0}'`
    _prov_diskn=`echo ${_prov_name} | sed 's/[^0-9]//g'`
    echo ${_prov_name} | sed 's/'${_prov_diskn}'/'$DISKN'/'

    return 0
    }

case $1 in
    start|stop)
        ;;
    *)
        usage
        exit 1
        ;;
esac

case $2 in
    ad[0-9])
        ;;
    da[0-9])
        ;;
    ada[0-9])
        ;;
    *)
        usage
        exit 1
        ;;
esac

if [ "$USER" != "root" ] ; then
    echo "current user is not root, running sudo..."
    sudo $0 $@
    exit 0
fi

# init vars
DISK=${2}
DISKN=`echo $DISK | sed 's/[^0-9]//g'`
DISKNAME=`echo $DISK | tr -d '[0-9]'`
MIRR_LIST=`gmirror status -s | awk '{print $1}' | awk -F/ '{print $2}' | uniq`
MIRR_INSERT="" # list of mirrors to 'gmirror insert'

echo "---- init"

case $1 in
    start)
        # check if disk doesn't even exist in system
        if [ ! -e /dev/$DISK ] ; then
            echo "error: /dev/$DISK doesn't exist. Try to raise it up with 'camcontrol rescan all' or 'atacontrol attach ataX'"
            exit 1
        fi

        # setting disk modes
        if [ $DISKNAME = "da" ] ; then
                echo "WCE: 1" | camcontrol modepage -e -n da -u $DISKN -m 8 -P 3
                camcontrol negotiate -n da -u $DISKN -O 200 -W 16 -T enable -R 160.000 -a
        fi

        # check for existing mirrors with given disk, fail if so
        # also grep will print found element(s)
        gmirror status -s | grep $DISK && exit_error

        # flush partition table
        dd if=/dev/zero of=/dev/$DISK count=1000 || exit_error

        # search for good source disk for disklabel
        # any mounted mirror or stripe member contains good candidates
        DISK_SRC=""
        for mirror in $MIRR_LIST ; do
            if mount | grep -q "/${mirror} on /" || gstripe status -s | grep -q " mirror/${mirror}" ; then
                DISK_SRC=`gmirror status -s $mirror | awk '{print substr($3,1,length($3)-1) ; exit 0}'`
                break
            fi
        done
        if [ "$DISK_SRC" = "" ]; then
            echo "error: it seems that there are no mirror which is mounted or is member of raid10"
            echo "so, can't find good source for disklabel"
            echo "please first fix this by hand"
            exit 1
        fi

        # copy disklabel info from source disk
        echo "---- disklabel"
        disklabel -Brw $DISK auto || exit_error
        disklabel $DISK_SRC | grep -v "c:" | disklabel -B -R $DISK /dev/stdin || exit_error

        ######################
        # disk init completed
        # now we'll make some deep checks of raid config, to prevent doing something really wrong

        echo "---- disk init completed, checking raid config"

#raid0
        if [ `gstripe status -s | grep -v "mirror/" | wc -l` -gt 0 ]; then
            echo "error: non-standard config, plain raid0 array(s) are present"
            echo "you need to do everything related to raid manually"
            exit 1
        fi

#raid1
        if [ `gstripe status -s | wc -l` -eq 0 ]; then
            # gstripe not configured
            # check if there are mirrors containing digit in name
            if echo "$MIRR_LIST" | egrep -q '[1-9]' ; then
                echo "error: there are mirror(s) - possible parts of raid10 array, but no active raid10 present"
                echo "you need to do everything related to raid manually"
                exit 1
            fi
            # raid1 only config - just insert disk to every found mirror
            echo "---- inserting"
            for mirror in $MIRR_LIST ; do
                # add element to list
                MIRR_INSERT="$MIRR_INSERT $mirror"
            done
        fi

#raid10
        if gstripe status -s | grep -q "mirror/" ; then
            # raid10 config

            mirrors_processed="" # list of processed mirrors
            for mirror in $MIRR_LIST ; do
                # 'mirror_set' var contains just 'mirror' if it is plain raid1, or 'mirror set' used in raid10 (e.g. "root" or "db1 db2")
                mirror_set="$mirror"
                # starting deep checks
                if echo $mirror_set | egrep -q '[1-5]'; then
                    # if 'mirror_set' contains number, replace its value to list of mirrors: "db2" -> "db1 db2"
                    mirror_basename=`echo $mirror_set | tr -d '12345'` # mirror without number
                    mirror_set=`gstripe status -s | egrep " mirror/${mirror_basename}[1-5]" | awk -F/ '{printf "%s ", $(NF)}'`

                    # looking for some non-proper situations, e.g. db1 without db2, db1+db2 not in raid10, bogus mirror names...
                    # there may be only one valid config: exactly two elements name1 + name2, both in active raid0
                    disk_count=`gmirror status -s $mirror_set | wc -l`
                    mirr_count=`echo $mirror_set | wc -w`
                    if [ "$mirr_count" -lt 2 ] || [ "$disk_count" -lt "$mirr_count" ]; then
                        echo "error: misconfiguration detected, ${mirror_basename} - check config"
                        exit 1
                    fi
                fi
                # lets check if this mirror has been processed already
                if echo " $mirrors_processed " | grep -q " $mirror " ; then
                    continue;
                fi

                # check if all mirror members have same disklabel names
                if [ `gmirror status -s $mirror_set | awk '{print substr($3,length($3),1)}' | sort | uniq | wc -l` -gt 1 ] ; then
                    echo "error: non-typical mirror structure, exiting: " $mirror
                    echo "usually it means that server has only 3 disks"
                    echo "you need to do everything related to raid manually"
                    exit 1
                fi
                # config check for this mirror is now complete

                # check type of mirror
                if echo $mirror | egrep -q '[1-5]'; then
                    # mirror is a part of raid10
                    # mirror_basename and mirr_count are already set for this case, see above!

                    # get expected number of mirror
                    # trying to guess, using round-robin
                    for mirror_number in $(expr $(expr $DISKN % $mirr_count) + 1) $(jot $mirr_count 1) ; do
                        if [ `gmirror status -s ${mirror_basename}${mirror_number} | wc -l` -eq 1 ]; then
                            break
                        fi
                    done

                    # add element to list
                    MIRR_INSERT="$MIRR_INSERT ${mirror_basename}${mirror_number}"
                else
                    # plain raid1 mirror, just add to list
                    MIRR_INSERT="$MIRR_INSERT $mirror"
                fi

                # marking mirrors in `mirror_set` as processed
                mirrors_processed="$mirrors_processed $mirror_set"
            done
        fi

        # finally insert mirrors by collected list
        for mirror in $MIRR_INSERT; do
            echo $mirror
            gmirror insert $mirror `get_prov_name $mirror` || exit_error
        done

        # check if swap must be placed to partition directly, and turn it on
        swapname=`awk '/swap/ && ! /#/ {print $1}' /etc/fstab | grep $DISK`
        if [ "$?" = "0" ]; then
            swapon $swapname
        fi

        ;;

    stop)
        # check if swap placed to partition directly, and turn it off
        swapname=`swapinfo | awk '{print $1}' | grep $DISK`
        if [ "$?" = "0" ]; then
            echo "turning off swap"
            swapoff $swapname
            sleep 1
        fi

        # do "forget" to all degraded mirrors
        echo "running forget.."
        gmirror status | awk '/DEGRADED/ {print $1}' | awk -F/ '{print $2}' | xargs gmirror forget

        # remove disk from mirrors
        echo "running remove.."
        for mirror in $MIRR_LIST ; do
            # save mirror status to var
            mirror_status="`gmirror status -s $mirror`"
            # check if mirror doesnt contain selected disk
            if ! echo "$mirror_status" | grep -q $DISK ; then
                continue
            fi

            # check if disk is last member for mirror
            if [ `echo "$mirror_status" | grep -v $DISK | grep -v '%' | wc -l` -eq 0 ] ; then
                echo "error: disk $DISK is last good member for mirror $mirror, exiting"
                exit 1
            fi
            # finally remove partition from mirror
            echo $mirror
            gmirror remove $mirror `echo "$mirror_status" | grep $DISK | awk '{print $3}'` || exit_error
        done

        # fill beginning of disk with zeros (before detach)
        sleep 1
        dd if=/dev/zero of=/dev/$DISK count=1000

        # trying to detach ATA channel..
        # get ATA channel for disk. "exit 1" in awk is a signal for sh that record found :)
        # if disk is not ATA, nothing will be found and executed, so no need for extra checks
        ATA_N=`atacontrol list | grep -B2 $DISK | awk '/ATA channel/ {print substr($3,1,1) ; exit 1 }'`
        # double-checking before final detach
        # 1: check if there is only one disk on channel, not two
        # 2: check if disk is really on this channel, not another
        if [ "$?" = "1" ] && [ `atacontrol info ata${ATA_N} | grep " ${DISKNAME}" | wc -l` -eq 1 ] && atacontrol info ata${ATA_N} | grep -q " ${DISK} " ; then
            echo "---- detaching ata${ATA_N}"
            sync
            sleep 1
            atacontrol detach ata${ATA_N}
        fi
        ;;
esac

# if we are here, it seems that all ok :)
echo "---- all done, please check results:"
echo
echo "# gmirror status"
gmirror status
