#!/usr/bin/env python
# coding=UTF-8
from __future__ import absolute_import
import string
"""
Author: artanis@yandex-team.ru
"""
import json
import sys
import time
import argparse
import getpass
from socket import timeout, gethostbyaddr, error, getaddrinfo
from re import search, sub
from functools import wraps
from os import path
from select import select
from multiprocessing import Pool

#python 2+3 compatible imports
try:
    from urllib2 import urlopen, Request, HTTPError, URLError
    from urllib import urlencode
except ImportError:
    from urllib.error import HTTPError, URLError
    from urllib.request import urlopen, Request
    from urllib.parse import urlencode

# from netaddr import IPAddress, IPNetwork imported in racktables_get_hosts!!

BOT_HW_HOSTS = "https://bot.yandex-team.ru/api/view.php?name=view_oops_hardware&format=json"
BOT_API = "https://eine.yandex-team.ru/computer/list.json?f[]="
HW = "https://bot.yandex-team.ru/api/consistof.php?name="
HW_INV = "https://bot.yandex-team.ru/api/consistof.php?inv="
BOT_PLANNER_TO_HW = "https://bot.yandex-team.ru/api/services.php?&act=info&planner_id="
CONDUCTOR_HOSTS2GROUPS = "https://c.yandex-team.ru/api/hosts2groups/"
CONDUCTOR_HOSTS2TAGS = "https://c.yandex-team.ru/api/hosts2tags/"
CONDUCTOR_GROUP2HOSTS = "https://c.yandex-team.ru/api/groups2hosts/"
CONDUCTOR_TAG2HOSTS = "https://c.yandex-team.ru/api/tag2hosts/"
CONDUCTOR_PROJECT2HOSTS = "https://c.yandex-team.ru/api/projects2hosts/"
RACKTBLES_API_NETWORKS = "https://racktables.yandex.net/export/networklist.php?report=vlans"
RACKTBLES_API_HOSTS = "https://ro.racktables.yandex.net/export/netmap/L123"
PLANNER_API = "https://abc-api.yandex-team.ru/v2/" + "services?"
GOLEM_HOST_ADMINS = "https://golem.yandex-team.ru/api/get_host_resp.sbml?host="
# dont forget to modify prepare_columns()
COLUMNS = ['inv', 'fqdn', 'cpu', 'mem_amount', 'disk_space', 'dc', 'placement', 'status',
           'mem_count', 'ssd_count', 'nvme_count', 'disk_count', 'disk_controllers', 'disk_controllers_count',
           'shelfs', 'shelfs_disk_cnt', 'shelfs_ssd_cnt', 'shelf_total_space', 'switch', 'port', 'c_group', 'c_tags', 'golem_admins'
          ]
BOT_TRANSLATOR = {"fqdn": "XXCSI_FQDN", "status": "status_name", "cpu": "item_segment2", "inv": "instance_number",
                  "dc": "loc_segment4", "placement": "loc_segment5"}
TOKEN_PATH = (path.join(path.expanduser('~'), '.invent.token'))



def return_error(text):
    '''
    Exit from script with text
    '''
    sys.exit(text)


def retry(ExceptionToCheck, tries=4, delay=3, backoff=2):
    """Retry calling the decorated function using an exponential backoff.
    Original: https://github.com/saltycrane/retry-decorator/blob/master/decorators.py
    """
    def deco_retry(f):
        @wraps(f)
        def f_retry(*args, **kwargs):
            mtries, mdelay = tries, delay
            while mtries > 1:
                try:
                    return f(*args, **kwargs)
                except ExceptionToCheck as e:
                    time.sleep(mdelay)
                    mtries -= 1
                    mdelay *= backoff
            return f(*args, **kwargs)
        return f_retry  # true decorator
    return deco_retry


@retry((URLError, HTTPError), tries=3, delay=1, backoff=1)
def get_data(url, is_json=False):
    """ get content of utl and save it;
    url - url; json - should i json.loads answer """
    try:
        data = urlopen(url).read()
    except (timeout, HTTPError, URLError) as e:
        if e.code == 404:
            return False
        if type(url) == str:
            print("Socket timed out on {0}".format(url))
        raise
    if is_json:
        data = json.loads(data.decode())
    return data


def get_token(link, bad_token=False):
    """
    return string with token
    option "bad_token=True", re-request token
    """

    def obtain_token():
        """
        Get OAuth token and save+return
        """
        with open(TOKEN_PATH, "w") as token_file:
            token = get_oauth_token()
            token_file.write(token)
            token_file.close()
            print("Token saved to " + TOKEN_PATH)
            return token


    @retry(HTTPError, tries=3, delay=0, backoff=0)
    def get_oauth_token():
        """
        Get OAuth token.
        Asks user for a password.
        Author: saku@
        """
        user = getpass.getuser()
        print('User: %s' % user)
        password = getpass.getpass()
        auth = urlencode({'grant_type': 'password', 'username': user,
                          'password': password,
                          'client_id': 'dd8752543fd646e78934d15688dfbd87',
                          'client_secret': 'eb363d59d50c41d29b9332e8e8aaf73f'
                         })
        request = Request(url='https://oauth.yandex-team.ru/token',
                                  data=auth)
        try:
            oauth_token = get_data(request, is_json=True)
            return oauth_token['access_token']
        except HTTPError:
            print('Invalid password, try again')
            raise

    if not bad_token:
        try:
            with open(TOKEN_PATH, "r") as token_file:
                token = token_file.read().splitlines()[0]
                token_file.close()
                return token
        except (IOError, IndexError):
            print("Need to request token, for access: \n" + str(link))
            return obtain_token()
    else:
        print("Can't access to {0}\n".format(str(link)) +
               "Trying to re-get token")
        return obtain_token()


def parse_args():
    """
    Check argument from stdin
    If argument not specified or equlment -h, --help print help and exit.
    """
    options = argparse.ArgumentParser(description='''Script to assist in finding hosts''',
                                      epilog='''If you want to use -v (hosts in vlan), you\
                                      should: apt-get install python-netaddr ''')
    options.add_argument('-c', '--conductor-group', nargs='+', default=None,
                         help="""Specify the conductor group(s) separated by a space to search on.
                         """)
    options.add_argument('-t', '--conductor-tag', nargs='+', default=None,
                         help="""Specify the conductor tag(s) separated by a space to search on.
                         """)
    options.add_argument('-p', '--conductor-project', nargs='+', default=None,
                         help="""Specify the conductor project(s) separated by a space to search on.
                         """)
    options.add_argument('-b', '--bot_name_pattern', default=None,
                         help="""Name pattern to search on bot.
                         """)
    options.add_argument('-a', '--abc', nargs='+', default=None,
                         help="""All hosts belongs to abc project(s). Example --abc devhw kompyuternoezrenie.
                         You can find name from abc link: https://abc.yandex-team.ru/services/devhw/ for devhw
                         """)
    options.add_argument('-r', '--host_range', nargs='+', default=None,
                         help="""Unix regexp of hosts list. Example - man1-{0600..0915}.search.yandex.net
                         """)
    options.add_argument('-f', '--files', nargs='+', default=None,
                         help="""File(s) with hostnames, separated by space.
                         """)
    options.add_argument('--columns', nargs='+', choices=COLUMNS,
                         help="Columns to show"
                        )
    options.add_argument('--sort', choices=COLUMNS, default='fqdn', help="Column to sort by")
    options.add_argument('-e', '--export', default=None,
                         help="""Export result to csv file. You should provide filename
                         """)
    options.add_argument('-v', '--vlan', default=None,
                         help="""Vlan to search on mapped hosts in racktables
                         """)
    options.add_argument('-s', '--switchport', action='store_true',
                         help="""show switch_port of founded hosts
                         """)
    options.add_argument('-i', '--intersecting', action='store_true',
                         help="""Intersecting sources. Use only intersecting hosts from used sources. Example = hosts in 696vlan AND conductor group
                         Example: -v 696 -c search_dev-main -i
                         """)
    options.add_argument('-u', '--uniq-hosts', action='store_true',
                         help="""Use only UNIQ hosts from used sources. Example = hosts found only in 696vlan AND OR only in conductor group
                         Example: -v 696 -c search_dev-main -u
                         """)
    options.add_argument('--exclude', nargs='+', default=None, action='append', help="""
                         Exclude strings which contain COLUMN == values. Example: -c search_dev-main --exclude cpu XEON5530 XEON5620
                         Columns may be: inv fqdn cpu mem_amount disk_space dc placement status mem_count ssd_count disk_count
                         You can use both --exclude AND --include. First runs hosts included, than excluded""")
    options.add_argument('--include', nargs='+', default=None, action='append', help="""
                         Show only strings which contain COLUMN == values. Example: -c search_dev-main --include disk_count 4 2
                         Columns may be: inv fqdn cpu mem_amount disk_space dc placement status mem_count ssd_count disk_count
                         You can use both --exclude AND --include. First runs hosts included, than excluded""")
    options.add_argument('--with-shelfs', action='store_true', help="""
                         Get additional info about connected disk shelfs""")
    options.add_argument('--no-hw-info', action='store_true', help="""
                         Do not get bot HW info, just print out hostnames""")
    options.add_argument('--detailed-hd-mem', action='store_true', help="""
                         Show cpu and disks as 2x2tb 4x8gb and so on.""")
    opts = options.parse_args()
    return opts

def rt_parse_ethermap(ip):
    """
    get one str(ip), return str(ip), if matched in subnets. For map
    SLOW, SHOULD BE REWRITTEN
    """
    addr = IPAddress(ip.decode())
    for subnet in subnets:
        if addr in subnet:
            return ip
    return False

def rt_resolve_ips(ip):
    """
    get one str(ip), return str(hostname), if matched in subnets. For map
    """
    try:
        return gethostbyaddr(ip)[0]
    except error:
        return False

def racktables_get_hosts(racktables_ips, subnets):
    """ return all netmapped hosts in vlan """
    hostnames = []
    hosts = []

    pool = Pool()
    hosts = filter(bool, pool.map(rt_parse_ethermap, racktables_ips))
    hostnames = filter(bool, pool.map(rt_resolve_ips, hosts))
    pool.close()
    pool.join()
    return hostnames


def get_abc_hosts(projects):
    """
    Get list of project names, return servers list
    """
    def get_project_ids(projects):
        """
        get list of abc projects names, return list of id's
        """
        result = []
        for project in projects:
            url = PLANNER_API + u"slug=" + project
            header = {'Authorization' : "OAuth " + get_token(url)}
            request = Request(url=url, headers=header)
            try:
                data = get_data(request, is_json=True)
            # ValueError coul be, when trying to json_load answ with
            # bad token, should try to re-get token
            except (HTTPError, ValueError):
                header = {'Authorization' : "OAuth " + get_token(url, bad_token=True)}
                request = Request(url=url, headers=header)
                data = get_data(request, is_json=True)
            try:
                result.append(data["result"][0]["id"])
            except (KeyError, IndexError):
                print("WARNING: Can't find project name " + project)
        return result


    def get_server_list(projects_ids):
        """
        get list of abc id's, return plain list of hostnames
        """
        result = []
        for project_id in projects_ids:
            try:
                url = BOT_PLANNER_TO_HW + str(project_id)
                url = url + "&token=" + get_token(url)
                data = get_data(url, is_json=True)
            except HTTPError:
                url = BOT_PLANNER_TO_HW + project_id
                url = url + "&token=" + get_token(url, bad_token=True)
                data = get_data(url, is_json=True)
            try:
                for server in data['servers']:
                    if server.get("fqdn", None) is not None:
                        result.append(server.get("fqdn", ""))
                    else:
                        print("WARNING: server with inv {0} has no fqdn".format(server.get("instance_number", "unknown")))
            except KeyError:
                print("Trouble with getting bot hw for project id " + project_id)
        return result

    return get_server_list(get_project_ids(projects))

def get_hosts_from_files(files):
    """
    Get list of fs path, return flat list of hosts.
    """
    result = []
    for file in files:
        with open(file, 'r') as f:
            for host in f.readlines():
                result.append(host.split())

    result = [host.decode() for sublist in result for host in sublist]
    return result



def gen_hosts_list(keys, racktables_ips=False, subnets=False):
    """
    Generate plain uniq list of all hosts from all sources, intersect winth -i
    """
    
    bot_hosts = host_range = conductor_group_hosts = conductor_project_hosts = rt_hosts =\
    host_list = abc_hosts = hosts_stdin = conductor_tag_hosts =\
    files_hosts = []
    # Get hosts from stdin, if so.
    if sys.stdin in select([sys.stdin], [], [], 0)[0]:
        hosts_stdin = []
        for host in sys.stdin.readlines():
            hosts_stdin.append(host.split())
        # make hosts_stdin flat
        hosts_stdin = [host.decode() for sublist in hosts_stdin for host in sublist]
        host_list = host_list + hosts_stdin
    
    # Gather info from other sources.
    if not keys.conductor_group and not keys.conductor_tag and not keys.conductor_project\
    and not keys.bot_name_pattern and not keys.host_range and not\
    keys.vlan and not keys.abc and not hosts_stdin and not keys.files:
        return_error("Please, use at least one of -c -b -r -v -a -t -f hosts sources, or send hosts to stdin")
    if keys.abc:
        abc_hosts = get_abc_hosts(keys.abc)
        host_list = host_list + abc_hosts
    if keys.files:
        files_hosts = get_hosts_from_files(keys.files)
        host_list = host_list + files_hosts
    if keys.conductor_group:
        try:
            conductor_group_hosts = [host.decode() for group in keys.conductor_group for host in get_data(CONDUCTOR_GROUP2HOSTS + group).split()]
        except AttributeError:
            print(u'One of groups from {} does not exist'.format(keys.conductor_group))
            pass
        host_list = host_list + conductor_group_hosts
    if keys.conductor_tag:
        try:
            conductor_tag_hosts = [host.decode() for group in keys.conductor_tag for host in get_data(CONDUCTOR_TAG2HOSTS + group).split()]
        except AttributeError:
            print(u'One of tag from {} does not exist'.format(keys.conductor_tag))
        host_list = host_list + conductor_tag_hosts
    if keys.conductor_project:
        conductor_project_hosts = [host.decode() for group in keys.conductor_project for host in get_data(CONDUCTOR_PROJECT2HOSTS + group).split()]
        host_list = host_list + conductor_project_hosts
    if keys.bot_name_pattern:
        bot_hosts = [host.get("hostname", "no_hostname") for host in get_data(BOT_API + "*" + keys.bot_name_pattern + "*", True)
                     if host.get("status", "inreserv") != "inreserv"]
        host_list = host_list + bot_hosts
    if keys.host_range:
        host_range = keys.host_range
        host_list = host_list + host_range
    if keys.vlan:
        rt_hosts = racktables_get_hosts(racktables_ips, subnets)
        host_list = host_list + list(rt_hosts)

    if keys.intersecting:
        def intersect(*d):
            return set(d[0]).intersection(*d[1:])

        selected_sources = filter(None, [conductor_group_hosts, bot_hosts,
                                         host_range, rt_hosts, abc_hosts,
                                         hosts_stdin, conductor_tag_hosts,
                                         conductor_project_hosts, files_hosts])
        host_list = intersect(*selected_sources)

    if keys.uniq_hosts:
        def uniq_list(host_list):
            result = []
            for i in host_list:
                if host_list.count(i) == 1:
                    result.append(i)
            return result
        return uniq_list(host_list)

    return list(set(host_list))


def parse_bot_hw(bot_answ):
    """
    Parse bot answer, return parsed dict, use some logich, should be rewritten
    """
    #TODO: причесать тут, ооп все дела. SHAME:(
    parsed_answ = {}
    disk_count = 0
    disk_space = 0
    mem_count = 0
    mem_amount = 0
    ssd_count = 0
    nvme_count = 0
    disk_controllers = []
    disk_controllers_count = 0
    shelfs_ssd_cnt = 0
    shelf_total_space = 0
    shelfs_disk_cnt = 0
    shelfs = ""
    if keys.detailed_hd_mem:
        disk_count = []
        ssd_count = []
        mem_count = []
        nvme_count = []
    # fill known params
    for my, bot in BOT_TRANSLATOR.items(): #inefficient on py2
        try:
            parsed_answ[my] = bot_answ.get("data").get(bot)
        except AttributeError:
            return "ERR"
    # count devices
    try:
        for device in bot_answ.get("data").get("Components"):
            dev_type = device.get("item_segment3", 'NA')
            dev_desc = device.get("item_segment2", 'NA')
            if dev_type == "RAM":
                module_size = str(sub("\D", "", dev_desc))
                if not keys.detailed_hd_mem:
                    mem_count += 1
                else:
                    mem_count.append(module_size)
                mem_amount += int(module_size)
            if dev_type == "DISKDRIVES":
                dev_conn = device.get("attribute15", 'SATA')
                drive_size = str(device.get("attribute14", str(sub("\D", "", dev_desc.split("/", 1)[0]))))
                if "U.2" in dev_conn:
                    if not keys.detailed_hd_mem:
                        nvme_count += 1
                    else:
                        nvme_count.append(drive_size)
                else:
                    if search("SSD", dev_desc):
                        if not keys.detailed_hd_mem:
                            ssd_count += 1
                        else:
                            ssd_count.append(drive_size)
                    else:
                        if not keys.detailed_hd_mem:
                            disk_count += 1
                        else:
                            disk_count.append(drive_size)
                disk_space += int(drive_size)
            if dev_desc == "DISKCONTROLLERS":
                disk_controllers.append(device.get("item_segment1", 'NA'))
                disk_controllers_count += 1
    except: #:(
        return "ERR"
    # Match @connected devices@ if --with-shelfs key added
    if keys.with_shelfs:
        if bot_answ.get("data").get("Connected", False):
            for attached_device in bot_answ.get("data").get("Connected"):
                attached_dev_type = attached_device.get("item_segment3", 'NA')
                attached_dev_inv = attached_device.get("instance_number", 'NA')
                if attached_dev_type in ("STORAGES", "NODE-STORAGE"):
                    shelfs = shelfs + " " + attached_dev_inv
                    attached_bot_hw = get_data(HW_INV + attached_dev_inv + "&format=json", True)
                    try:
                        for attached_device_device in attached_bot_hw.get("data").get("Components"):
                            attached_device_device_type = attached_device_device.get("item_segment3", 'NA')
                            attached_device_device_desc = attached_device_device.get("item_segment2", 'NA')
                            if attached_device_device_type == "DISKDRIVES":
                                if search("SSD", attached_device_device_desc):
                                    shelfs_ssd_cnt += 1
                                else:
                                    shelfs_disk_cnt += 1
                                shelf_total_space += int(sub("\D", "", attached_device_device_desc.split("/", 1)[0]))
                    except: #:( again
                        pass
    else:
        shelf_total_space = shelfs_disk_cnt = shelfs_ssd_cnt = shelfs = "Use --with-shelfs"
    
    if keys.detailed_hd_mem:
        tr = {"mem_count": mem_count,
             "ssd_count": ssd_count,
             "disk_count": disk_count,
             "nvme_count": nvme_count,
             }
        for k, v in tr.items():
            r = ''
            for l in set(v):
                r = r + str(v.count(l)) + 'x' + l + ' '
            parsed_answ[k] = r[0: -1]
    else:
        parsed_answ["ssd_count"] = ssd_count
        parsed_answ["nvme_count"] = nvme_count
        parsed_answ["disk_count"] = disk_count
        parsed_answ["mem_count"] = mem_count

    parsed_answ["mem_amount"] = mem_amount
    parsed_answ["disk_space"] = disk_space
    parsed_answ["disk_controllers"] = ', '.join(list(set(disk_controllers)))
    parsed_answ["disk_controllers_count"] = disk_controllers_count
    parsed_answ["shelf_total_space"] = shelf_total_space
    parsed_answ["shelfs_disk_cnt"] = shelfs_disk_cnt
    parsed_answ["shelfs_ssd_cnt"] = shelfs_ssd_cnt
    parsed_answ["shelfs"] = shelfs
    parsed_answ["switch"] = 'NA'
    parsed_answ["port"] = 'NA'
    return parsed_answ

def get_swport(host):
    """
    Function for map. Get dict(host), match dict['fqdn']it in racktables_ethermap.
    return "patched" dict
    """
    try:
        ipaddr = getaddrinfo(host.get('fqdn', u'NA'), 0)[0][-1][0]
        swport = racktables_ethermap_dict.get(ipaddr.encode())
        if swport:
            a = swport[0].decode().split('/')
            host['switch'] = a[0]
            a.pop(0)
            host['port'] = '/'.join(a)
    except:
        host['switch'] = u'NA'
        host['port'] = u'NA'
    return host


def get_c_group(host):
    """
    Function for map. Get dict(host), match dict['fqdn'] to conductor group by conductor API.
    Return "patched" dict
    """
    try:
        c_groups = get_data('{}{}'.format(CONDUCTOR_HOSTS2GROUPS, host.get('fqdn'))).decode()
        last_c_group = c_groups.split('\n')[-2]
    except AttributeError:
        last_c_group = ''
    host['c_group'] = last_c_group

    return host

def get_c_tags(host):
    """
    Function for map. Get dict(host), match dict['fqdn'] to conductor group by conductor API.
    Return "patched" dict
    """
    try:
        c_tags = get_data('{}{}'.format(CONDUCTOR_HOSTS2TAGS, host.get('fqdn'))).decode()
        host['c_tags'] = " ".join(c_tags.splitlines())
    except AttributeError:
        host['c_tags'] = " "

    return host


def get_golem_admins(host):
    """
    Function for map. Get dict(host), match dict['fqdn'] to conductor group by golem API.
    Return "patched" dict
    """
    host["golem_admins"] = get_data('{}{}'.format(GOLEM_HOST_ADMINS, host.get('fqdn'))).decode().strip().replace(',', ' ')

    return host

def get_hosts_info(all_hosts, keys):
    """gets list of fqdns and, with help of consistof.php, creates dict"""
    parsed_hosts = []
    err_hosts = []
    i = 0
    for host in all_hosts:
        i += 1
        bot_hw = get_data(str(HW) + str(host) + u"&format=json", True)
        host_parsed = parse_bot_hw(bot_hw)
        if host_parsed != "ERR":
            parsed_hosts.append(host_parsed)
        else:
            err_hosts.append(host)
        sys.stderr.write("\rGet info from bot [{0}/{1}]              \
                         ".format(i, len(all_hosts)))
        sys.stderr.flush()

    if keys.switchport:
            sys.stderr.write('\nStart to fetch switch/port')
            pool = Pool()
            parsed_hosts = pool.map(get_swport, parsed_hosts)
            pool.close()
            pool.join()

    if keys.columns is not None and 'c_tags' in keys.columns:
            sys.stderr.write('\nStart to fetch conductor tags per host...')
            pool_c = Pool()
            parsed_hosts = pool_c.map(get_c_tags, parsed_hosts)
            pool_c.close()
            pool_c.join()

    if keys.columns is not None and 'c_group' in keys.columns:
            sys.stderr.write('\nStart to fetch conductor groups per host...')
            pool_c = Pool()
            parsed_hosts = pool_c.map(get_c_group, parsed_hosts)
            pool_c.close()
            pool_c.join()

    if keys.columns is not None and 'golem_admins' in keys.columns:
            sys.stderr.write('\nGet golem server admins...')
            pool_g = Pool()
            parsed_hosts = pool_g.map(get_golem_admins, parsed_hosts)
            pool_g.close()
            pool_g.join()

    sys.stderr.write("\n\r                                             \r\n")
    sys.stderr.flush()
    if err_hosts:
        sys.stderr.write("\nNot found {0}\n".format(err_hosts))
    return parsed_hosts


def include_exclude(all_hosts):
    """
    get dict, return dict. white/blacklisted by keys-values
    """
    excluded = []
    included = []
    keys_check = []
    # is there in exclude/include keys right values of columns?
    if keys.exclude:
        keys_check += keys.exclude[0]
    if keys.include:
        keys_check += keys.include[0]
    c = list(set(keys_check) & set(COLUMNS))

    if c:
        if keys.include:
            incl_values = []
            for item in keys.include[0]:
                if item in COLUMNS:
                    key = item
                else:
                    incl_values.append(item)
            for host in all_hosts:
                if str(host.get(key)) in incl_values:
                    included.append(host)
            all_hosts = included
        if keys.exclude:
            excl_values = []
            for item in keys.exclude[0]:
                if item in COLUMNS:
                    key = item
                else:
                    excl_values.append(item)
            for host in all_hosts:
                if str(host.get(key)) not in excl_values:
                    excluded.append(host)
            all_hosts = excluded
    # if not - print and return []
    else:
        print("For include or exclude, plese, use one of: {0}".format(COLUMNS))
        all_hosts = []

    return all_hosts


def table_print(data, title_row):
    """
    data: list of dicts,
    title_row: e.g. [('name', 'Programming Language'), ('type', 'Language Type')]
    """
    max_widths = {}
    data_copy = [dict(title_row)] + list(data)
    for col in data_copy[0].keys():
        try:
            max_widths[col] = max([len(str(row[col])) for row in data_copy])
        except KeyError:
            max_widths[col] = 10
    cols_order = [tup[0] for tup in title_row]

    def custom_just(col, value):
        if type(value) == int:
            return str(value).rjust(max_widths[col])
        else:
            return value.ljust(max_widths[col])
    for row in data_copy:
        try:
            row_str = " | ".join([custom_just(col, row[col]) for col in cols_order])
        except KeyError:
            pass
        print("| %s |" % row_str)
        if data_copy.index(row) == 0:
            underline = "-+-".join(['-' * max_widths[col] for col in cols_order])
            print('+-%s-+' % underline)


def export_csv(data, title_row, file):
    ''' data: list of dicts
    title_row: e.g. [('name', 'Programming Language'), ('type', 'Language Type')]
    '''
    if file == '-':
        f = sys.stdout
    else:
        f = open(file, 'w')

    title = dict(title_row)
    cols_order = [tup[0] for tup in title_row]
    # print head of table
    head_list = [ title[head] for head in cols_order ]
    head_output = ','.join(head_list)
    f.write('{}\n'.format(head_output))
    for server in data:
        server_list = []
        for key in cols_order:
            try:
                server_list.append(str(server.get(key)))
            except:
                server_list.append('{} NA'.format(key))
        server_output = ','.join(server_list)
        f.write('{}\n'.format(server_output))

    if file != '-':
        sys.stderr.write('Exported to ' + file)


def prepare_columns(selected):
    """
    Get list of keys and return list of tuples for table_print()
    """
    if not selected:
        selected = [ 'inv', 'fqdn', 'cpu', 'mem_amount', 'disk_space', 'dc', 'placement', 'mem_count',
                     'ssd_count', 'nvme_count', 'disk_count' ]

    unprepared = {'inv': ("inv", "Inventory N"), 'fqdn': ("fqdn", "Hostname"), 'cpu': ("cpu", "Cpu model"),
                  'mem_amount': ("mem_amount", "Mem GB"), 'disk_space': ("disk_space", "Disk GB"),
                  'dc': ("dc", "DC"), 'placement': ("placement", "Rack"), 'status': ("status", "Host status"),
                  'mem_count': ("mem_count", "Mem cnt"), 'ssd_count': ("ssd_count", "SSDs"), 'nvme_count': ("nvme_count", "NVMEs"),
                  'disk_count': ("disk_count", "Disks"), 'disk_controllers': ("disk_controllers", "Disk Contr."),
                  'disk_controllers_count': ("disk_controllers_count", "Dsk Cntr. cnt."), 'shelfs': ("shelfs", "Shelfs inv"),
                  'shelfs_disk_cnt': ("shelfs_disk_cnt", "Total Disks in shelfs"),
                  'shelfs_ssd_cnt': ("shelfs_ssd_cnt", "Total ssd in shelfs"),
                  'shelf_total_space': ("shelf_total_space", "Total shelf disk space"),
                  'switch': ("switch", "Switch name"), 'port': ("port", "Port Number"), 'c_group':("c_group", "Conductor group"),
                  'c_tags':("c_tags", "Conductor tags"), 'golem_admins':("golem_admins", "Golem Admins"),
                  }
    result = []
    if keys.with_shelfs and not keys.columns:
        selected.extend(["shelfs", "shelfs_disk_cnt", "shelf_total_space"])
    if keys.switchport and not keys.columns:
        selected.extend(["switch", "port"])
    for column in selected:
        result.append(unprepared.get(column))
    return result


def render_output(all_hosts, keys):
    title = prepare_columns(keys.columns)
    if keys.export:
        export_csv(all_hosts, title, keys.export)
    elif keys.no_hw_info is True:
        print('\n'.join(all_hosts))
    else:
        table_print(all_hosts, title)


keys = parse_args()
# get all hosts to plain list
sys.stderr.write("Get hosts from selected sources...\n")

# prepare data for rt multiprocessing parse
#TODO: IPSet ускорит поиск вхождения в нужные подсети, наверное.
if keys.vlan or keys.switchport:
    if not keys.vlan:
        keys.vlan = ''
    try:
        from netaddr import IPAddress, IPNetwork, IPSet
    except ImportError:
        return_error("Please, install netaddr for interract with racktables, otherwise -v / -s will be ignored")
    
    subnets = []
    racktables_ethermap_dict = {}
    
    racktables_ethermap = [line.split() for line in get_data(RACKTBLES_API_HOSTS).splitlines()]
    #make dict {ip: [swport, mac]}
    if keys.switchport:
        for i in racktables_ethermap:
            racktables_ethermap_dict[i[2]] = [i[0], i[1]]
    racktables_ips = [host[2] for host in racktables_ethermap]
    racktables_subnets = [network.decode().split('\t', 1)[0] for network in get_data(RACKTBLES_API_NETWORKS).splitlines()
                          if search(u"\t" + keys.vlan, network.decode())]
    racktables_subnets = list(set(racktables_subnets))
    for subnet in racktables_subnets:
        subnets.append(IPNetwork(subnet))
else:
    subnets = racktables_ips = []

# Generate host_list from all sources
all_hosts = gen_hosts_list(keys, racktables_ips, subnets)
sys.stderr.flush()

# get values from bot if no --no-hw-info
if keys.no_hw_info is False:
    sys.stderr.write("Parsing bot answers..")
    all_hosts = get_hosts_info(all_hosts, keys)
    if keys.include or keys.exclude:
        all_hosts = include_exclude(all_hosts)
    #ADD NULL FOR COLUMNS WITHOUT KEY
    sys.stderr.flush()

# sort hosts by key
if not all_hosts:
    print("no hosts found :(")
else:
    if keys.no_hw_info is False:
        all_hosts = sorted(all_hosts, key=lambda k: k[keys.sort])
    # output
    render_output(all_hosts, keys)
