#!/bin/sh -e
#
# Temporary script for setting up local firewall on spiders.
#
# $Id$

PATH=/sbin:/bin:/usr/sbin:/usr/bin:/usr/local/sbin:/usr/local/bin
export PATH

#-- Subroutines --------------------------------------------------------

err()
{
	local _exitval

	_exitval=$1
	shift

	echo 1>&2 "ERROR: $*"
	exit $_exitval
}

usage()
{
	echo 1>&2 "Usage: ${thiscmd} [-q] [start|stop|restart|update]"
	exit 1
}

get_opts()
{
	local _opt

	while getopts "q" _opt; do
		case "$_opt" in
			q) quiet=1 ;;
			*) usage ;;
		esac
	done

	shift $(($OPTIND - 1))

	case "$#" in
		1) mode="$1" ;;
		0) mode="${default_mode}" ;;
		*) usage ;;
	esac
}

warn()
{
	if [ -z "${quiet}" ]; then
		echo 1>&2 "WARNING: $*"
	fi
}

add_to_cleanup()
{
	cleanup_files="${cleanup_files}${cleanup_files:+ }$*"
}

mk_tmp_file()
{
	local _var _tmp_file
	_var="$1"
	_tmp_file=$(mktemp -t $thiscmd) || err 1 "Can't make temporaty file !"
	add_to_cleanup $_tmp_file
	eval $_var=\"\$_tmp_file\"
}

is_file()
{
	if echo "$1" | grep -Eq "^/"; then
		return 0
	fi
	return 1
}

is_macro()
{
	if echo "$1" | grep -Ewq "${macro_re}"; then
		return 0
	else
		return 1
	fi
}

resolve_files()
{
	local _var _input _entry _entry_macros _all_macros

	_var="$1"; shift
	_input="$*"

	for _entry in $_input; do
		if is_file "${_entry}"; then
			_entry_macros=$(sed -Ee 's/[       ]*#.*//; /^$/d' \
				$_entry || true)
		else
			_entry_macros="${_entry}"
		fi
		if [ -z "${_entry_macros}" ]; then
			err 1 "Can't handle entry ${_entry} !"
		fi
		_all_macros="${_all_macros}${_all_macros:+ }${_entry_macros}"
	done

	eval $_var=\"\$_all_macros\"
}

resolve_macros()
{
	local _var _input _entry _entry_nets _all_nets

	_var="$1"; shift
	_input="$*"

	for _entry in $_input; do
		if is_macro "${_entry}"; then
			# We remove all IPv6 networks from a macro
			_entry_nets=$( (cat $fw_macros_file && echo $_entry) | \
				m4 | \
				sed -E -e 's/(^ +| +$)//g' -e 's/ +or +/ /g' \
					-e 's/ *[^ ]+::[^ ]+ */ /g')
			if is_macro "${_entry_nets}"; then
				err 1 "Can't resolve macro ${_entry} !"
			fi
		else
			_entry_nets="${_entry}"
		fi
		if [ -z "${_entry_nets}" ]; then
			err 1 "Can't resolve entry ${_entry} !"
		fi
		_all_nets="${_all_nets}${_all_nets:+ }${_entry_nets}"
	done

	eval $_var=\"\$_all_nets\"
}

add_to_table()
{
	local _tbl _macros _nets _net _out
	_tbl="$1"; shift
	resolve_files _macros $*
	resolve_macros _nets $_macros
	mk_tmp_file _out
	for _net in $_nets; do
		if ! $fwcmd table $_tbl add $_net 2>$_out; then
			warn_nets="${warn_nets}${warn_nets:+ }${_net}"
			if [ -z "${quiet}" ]; then
				cat $_out || true
			fi
			warn "Can't add ${_net} to table ${_tbl} !"
		fi
	done
}

remove_from_table()
{
	local _tbl _nets _net
	_tbl="$1"; shift
	_nets="$*"
	for _net in $_nets; do
		if ! $fwcmd table $_tbl delete $_net; then
			warn "Can't remove ${_net} from table ${_tbl} !"
		fi
	done
}

mk_tmp_table()
{
	# Without arguments it just flushes temporary table
	local _nets
	_nets="$*"
	if ! $fwcmd table $tmp_tbl_num flush; then
		err 1 "Can't flush temporary table ${tmp_tbl_num} !"
	fi
	if [ -z "${_nets}" ]; then
		return
	fi
	add_to_table $tmp_tbl_num $_nets
	if [ -n "${warn_nets}" ]; then
		warn_nets=""
		return 1
	fi
}

tbl_to_file()
{
	local _file _file_var _tbl

	_file_var="$1"
	_tbl="$2"

	mk_tmp_file _file

	if ! $fwcmd table $_tbl list > $_file; then
		err 1 "Can't safe table ${_tbl} to file ${_file} !"
	fi

	eval $_file_var=\"\${_file}\"
}

merge_tables()
{
	local _cur _tmp _cur_file _tmp_file _added_nets _removed_nets

	_cur="$1"
	_tmp="$2"

	if [ -z "${_cur}" -o -z "${_tmp}" ]; then
		err 1 "There are not enough arguments !"
	fi

	tbl_to_file _tmp_file $_tmp
	tbl_to_file _cur_file $_cur

	eval $(diff -u $_cur_file $_tmp_file | awk '
		function add_net(_list, _net) {
			if (_list) _list = sprintf("%s %s", _list, _net)
			else _list = _net
			return _list
		}
		/^\+[^\+]/	{
			sub(/^\+/,"",$1) 
			a_nets = add_net(a_nets, $1)
		}
		/^-[^-]/	{
			sub(/^-/,"",$1) 
			r_nets = add_net(r_nets, $1)
		}
		END{
			printf("_added_nets=\"%s\";\n", a_nets)
			printf("_removed_nets=\"%s\";\n", r_nets)
		}
	')

	add_to_table $_cur $_added_nets
	remove_from_table $_cur $_removed_nets
}

update_table()
{
	local _tbl _nets

	_tbl="$1"; shift
	_nets="$*"

	if mk_tmp_table $_nets; then
		merge_tables $_tbl $tmp_tbl_num
	else
		warn "Table ${_tbl} won't be updated !"
	fi
}

subs_rules()
{
	local _port _tbl_num _all_subs_ports

	if ! eval $(awk '
		/name="Destination"/ {
			sub(/.*value="/, "", $0)
			sub(/".*/, "", $0)
			if (uniq[$0]) next
			uniq[$0] = 1
			split($0, ip_port, ":")
			host = ip_port[1]
			port = ip_port[2]
			if (! tbl[port]) tbl[port] = host
			else tbl[port] = sprintf("%s %s", tbl[port], host)
		}
		END{
			for (port in tbl) {
				if (!p_list) p_list = port
				else p_list = sprintf("%s %d", p_list, port)
				printf("subs_%d_hosts=\"%s\";\n", port, \
					tbl[port])
			}
			printf("subs_port_list=\"%s\";\n", p_list)
		}
		' $subs_conf)
	then
		err 1 "Can't get subscriber clients !"
	fi

	if [ -z "${subs_port_list}" ]; then
		warn "There are no subscriber clients !"
	fi

	# We create tables and rules for all possible subscribers' ports
	# because it's much easier not to care about set of rules during
	# updates.
	_all_subs_ports=$(jot $(($subs_port_max - $subs_port_min + 1)) \
		$subs_port_min)

	for _port in $_all_subs_ports; do
		eval subs_hosts=\"\${subs_${_port}_hosts}\"
		_tbl_num=$(($subs_tbl_base_num + $_port - $subs_port_min))
		case "${mode}" in
			start)
				add_to_table $_tbl_num $subs_hosts
				$fwcmd add allow ip4 from me to \
					table\(${_tbl_num}\) ${_port} \
					setup proto tcp uid $robot_user \
					// Subscribe clients
				;;
			update)
				update_table $_tbl_num $subs_hosts
				;;
		esac
	done
}

check_start_depends()
{
	local _pf_enable

	_pf_enable=$(. /etc/rc.subr; \
		load_rc_config pf; \
		if checkyesno pf_enable; then \
			echo yes; \
		fi
		)

	if [ -n "${_pf_enable}" ]; then
		# This host uses pf so we don't need to do anything here.
		exit 0
	fi
}

ipfw_spider_start()
{
	local _tbl _tbl_num _nets

	check_start_depends

	echo "Starting local firewall."

	# XXX Dirty hack for langnet
	langnet_rule=$($fwcmd list 550 2>/dev/null || true)

	$fwcmd -f flush
	$fwcmd table all flush

	# XXX Dirty hack for langnet
	if [ -n "${langnet_rule}" ]; then
		$fwcmd add $langnet_rule
	fi

	for _tbl in $tbl_list; do
		eval _tbl_num=\$${_tbl}_tbl_num
		eval _nets=\"\$${_tbl}_nets\"
		add_to_table $_tbl_num $_nets
	done

	$fwcmd add allow ip4 from any to any \
		established proto tcp \{ uid $robot_user or uid $proxy_user \}

	$fwcmd add allow ip4 from me to any 53 \
		setup proto tcp \{ uid $robot_user or uid $proxy_user \}

	$fwcmd add allow ip4 from me to any 873 setup proto tcp uid $robot_user

	$fwcmd add allow ip4 from me to zora.yandex.net 33444 \
		setup proto tcp uid $robot_user // Zora for LogicalDocs

	$fwcmd add allow ip4 from me to table\($spider_tbl_num\) 8181 \
		setup proto tcp uid $robot_user // Spider Monitoring

	$fwcmd add allow ip4 from me to table\($mlspider_tbl_num\) 3128 \
		setup proto tcp uid $robot_user // MultiLingual Proxies

	subs_rules

	$fwcmd add allow ip4 from me to table\($arcadiasvn_tbl_num\) 22 \
		setup proto tcp uid $robot_user // SVN \& CVS

	$fwcmd add allow ip4 from me to table\($larix_tbl_num\) 6555 \
		setup proto tcp uid $robot_user // MapReduce Proxies

	$fwcmd add allow ip4 from me to table\($slbpublic_tbl_num\) \
		setup proto tcp \{ uid $robot_user or uid $proxy_user \} \
		// Public SLB

	$fwcmd add allow ip4 from me to table\($nonslbs_tbl_num\) 80,443 \
		setup proto tcp \{ uid $robot_user or uid $proxy_user \} \
		// Services without SLB

	$fwcmd add allow ip4 from me to table\($nonslb_tbl_num\) 80 \
		setup proto tcp \{ uid $robot_user or uid $proxy_user \} \
		// Services without SLB

	$fwcmd add deny log logamount 1024 ip4 from me to \
		table\($yandex_tbl_num\) setup proto tcp \
		\{ uid $robot_user or uid $proxy_user \} // Yandex IPv4 Nets

	# ipfw can't match UID for IPv6 so we have to use separate rules
	# for IPv6
	$fwcmd add allow ip6 from me to any 873 setup proto tcp 
	$fwcmd add deny log logamount 1024 ip6 from me to any setup proto tcp

	echo "Local firewall started."
}

ipfw_spider_update()
{
	local _tbl _tbl_num _nets

	for _tbl in $tbl_list; do
		eval _tbl_num=\$${_tbl}_tbl_num
		eval _nets=\"\$${_tbl}_nets\"
		update_table $_tbl_num $_nets
	done

	subs_rules
}

cleanup()
{
	if [ -n "${cleanup_files}" ]; then
		rm -f $cleanup_files || true
	fi
	mk_tmp_table
}


#-- Variables ----------------------------------------------------------

thiscmd=$(basename $0)

fwcmd="/sbin/ipfw -q"

fw_macros_file="/usr/local/etc/fw/macros-inc.m4"
macro_re='^_[A-Z0-9_]+_$'

robot_user="webbase"
proxy_user="squid"

spider_fw_dir="/Berkanavt/config/fw"

yandex_tbl_num="1";	yandex_nets="_YANDEXNETS_"
slbpublic_tbl_num="2";	slbpublic_nets="_SLBPUBLICSUPERNETS_"
arcadiasvn_tbl_num="3";	arcadiasvn_nets="_ARCADIASVN_ tree.yandex.ru"
spider_tbl_num="4";	spider_nets="_SPIDERSRV_ _SPIDERNETS_ 127.0.0.1"
mlspider_tbl_num="5";	mlspider_nets="${spider_nets} _SEARCHMLSPIDERNETS_"
larix_tbl_num="6";	larix_nets="_MRPROXY01SRV_"
nonslbs_tbl_num="7";	nonslbs_nets="${spider_fw_dir}/allow-80_443"
nonslb_tbl_num="8";	nonslb_nets="${spider_fw_dir}/allow-80"

tbl_list="yandex slbpublic arcadiasvn spider mlspider larix nonslbs nonslb"

subs_port_min="33450"		# These ports must be synchronized with
subs_port_max="33459"		# what we have in NOC firewall.
subs_tbl_base_num="50"
subs_conf="/Berkanavt/spider/config/logsubs.conf.xml"

tmp_tbl_num="127"

default_mode="start"


#-- Main ---------------------------------------------------------------

trap cleanup EXIT

get_opts $@

warn_nets=""

case "$mode" in
	start|restart) ipfw_spider_start ;;
	update) ipfw_spider_update ;;
	stop) exit 0 ;;
	*) usage ;;
esac

