#!/usr/bin/env python3
# -*- encoding: utf-8 -*-
#
# This script generates IPv6 EUI-64 address for any host.
# It gets active MAC from http://eine.yandex-team.ru
#
# TODO: MAC can be set by hands.

import urllib.request
import getopt
import sys
import ipaddress

# Take string value and conver this value to bits.
conv_to_bin = lambda data, base, fill: bin(int(data, base))[2:].zfill(fill)

# Take string value and conver this value to hex.
conv_to_hex = lambda data, base: hex(int(data, base))


def main():
    prefix = ""
    hostlist = []
    macs = []
    try:
        opts, args = getopt.getopt(sys.argv[1:], 'p:H:h')

    except getopt.GetoptError as e:
        print(str(e))
        usage()

    for o, a in opts:
        if o == '-h':
            usage()
        elif o == '-p':
            prefix = a
        elif o == '-H':
            hostlist = a.split()
        else:
            usage()

    if not prefix or not hostlist:
        usage()
    else:
        for host in range(0, len(hostlist)):
            mac = get_active_mac(hostlist[host])
            if mac:
                eui64 = generate_eui64(mac, prefix)
                try:
                    ipaddress.IPv6Address(eui64)
                    print("%s %s" % (hostlist[host], eui64))
                except ipaddress.AddressValueError as AddrErr:
                    print("ERROR: Generated address for %s in not valid: %s" % (hostlist[host], eui64))
    return 0


def get_active_mac(host):
    '''Get active MAC from eine.'''
    act_mac = ""
    gen_url = "http://eine.yandex-team.ru/computer/get_macs.cli?id=%s&mac_type=0" % (host)
    try:
        req = urllib.request.urlopen(gen_url)
        act_mac = req.read().decode('utf-8')[:-1]
    except urllib.error.HTTPError:
        print("Could not get active MAC for %s." % (host))
    return act_mac


def invert(bit):
    ''' Just invert a bit.'''
    if bit == '0':
        bit = '1'
    elif bit == '1':
        bit = '0'
    return bit


def convert_first_byte(byte_splited_maclist):
    ''' Do RFC2373 magic with first byte of MAC addess'''
    first_byte_bin = conv_to_bin(byte_splited_maclist[0], 16, 8)
    inverted_bit = invert(first_byte_bin[-2])
    new_first_byte_bin = first_byte_bin[:-2] + inverted_bit + first_byte_bin[-1:]
    new_first_byte_hex = conv_to_hex(new_first_byte_bin, 2)
    return new_first_byte_hex


def gen_empty_list(n):
    ''' Generate list with (n) empty strings in it.'''
    try:
        List = []
        for i in range(0, n):
            List.append('')
        return List
    except ValueError as err:
        print(err)


def generate_eui64(mac, net_prefix):
    ''' Do the magic for each geted MAC:
    Generates byte list of geted MAC by separating it by bytes;
    Change universal/local bit in first byte;
    Insert 0xff and 0xfe bytes in proper places of byte list;
    '''
    # TODO: Optimize some logic; separate this to few functions(?).

    # Generate empty list for our result address
    pref_address = gen_empty_list(8)
    mac_address = gen_empty_list(4)

    # Split prefix and MAC strings to get lists of "octets" values
    mac_splited = mac.split(':')
    prefix_splited = net_prefix.split(':')

    # Fill first part of resulting address with prefix
    for i in range(0, len(prefix_splited)):
        pref_address[i] = prefix_splited[i]

    # Convert first byte (by flipping 7th bit)
    first_byte = convert_first_byte(mac_splited)

    # Replace first byte in MAC list
    if len(first_byte) == 3:
        mac_splited[0] = "0" + first_byte[-1]
    else:
        mac_splited[0] = first_byte[-2:]

    # Insert 0xff and 0xfe bytes in center of MAC address list
    mac_splited.insert(3, "ff")
    mac_splited.insert(4, "fe")

    # Add with fully converted MAC list, but in propper format
    for i in range(0, len(mac_splited), 2):
        mac_address.append(mac_splited[i]+mac_splited[i+1])

    # Generate final list with proper values of "octets"
    eui64_list = []
    for i in range(0, 8):
        eui64_list.append(pref_address[i] + mac_address[i])

    # Forming result as a string
    result_str = ""
    for i in range(0, len(eui64_list)):
        result_str += eui64_list[i]
        if i != len(eui64_list) - 1:
            result_str += ":"
    return result_str


def usage():
    print("Usage: %s -p <net prefix> -H <hostlist>" % sys.argv[0])
    sys.exit(1)

if __name__ == '__main__':
    main()
