#!/usr/bin/env python2.7
# -*- coding: utf-8 -*-

"""
This module is capable to:
    generate tickets for helpdesk using BOT-API;
    generate unformalized requests using email;
    show information about servers and connected storages;
    manage downtime for hosts;
    show information about actions with servers.

For any suggestions or improvements, please, contact saku@yandex-team.ru
"""
import urllib2
from urllib2 import urlopen
import getpass
import urllib
import socket
import os
import sys
import re
import json
import argparse
import pwd
import datetime
import ConfigParser
import smtplib
from email.mime.text import MIMEText


class BotRequest(object):
    """
    Main class for initiation formalized requests and other requests
    by BOT-API.
    """
    def __init__(self, email_from='', email_cc='', email_replyto='',
                 read_only=False, initiator='', reboot=False, boot='',
                 needcall=False, oauth_token='', force_mode=False):
        self.bot_api = 'https://bot.yandex-team.ru'
        self.email_from = email_from
        self.email_cc = email_cc
        self.email_replyto = email_replyto
        self.read_only = read_only
        self.initiator = initiator
        self.boot = boot
        self.oauth_token = oauth_token
        self.force_mode = force_mode

        if needcall:
            self.needcall = 'yes'
        else:
            self.needcall = 'no'

        if reboot:
            self.reboot = 'yes'
        else:
            self.reboot = 'no'

    # FORMALIZED REQUESTS
    def req_kvm(self, server, comment=''):
        """
        Show current connected KVM, otherwise generate KVM request.
        Servers with IPMI returns link to ipmi and ipmi-console.
        """
        bot_id, st_id = '', ''
        if server.host_has_ipmi():
            golem_ipmi_link = ('http://golem.yandex-team.ru/launch-ipmi'
                               '.sbml?h=%s' % server.name)
            print_bold('IPMI', 'http://%s.ipmi.yandex-team.ru' % server.name)
            print_bold('GOLEM', golem_ipmi_link)
            server.console = golem_ipmi_link
        else:
            current_kvm = open_url(url=self.bot_api, data='/api/kvm.php?name=%s'
                                   % server.name).readlines()[0]
            if 'FAILED' in current_kvm or self.force_mode:
                if self.read_only and not self.force_mode:
                    print_bold('MESSAGE', 'To server %s is not connected KVM. '
                               'Need to send a request' % server.name)
                else:
                    bot_id, st_id = self.make_request(server.name,
                                                        'KVM', comment)
            else:
                server = self.get_dc(server)
                queue_num = current_kvm.split()[1].split('/')[0]
                if queue_num in server.queue:
                    kvm = '%s-%s' % (server.dc, current_kvm.split()[1])
                    golem_kvm_link = ('https://golem.yandex-team.ru/'
                                      'launch-kvm.sbml?kvm=%s' % kvm)
                    print_bold('KVM', kvm)
                    print_bold('GOLEM', golem_kvm_link)
                    server.kvm = kvm
                    server.console = golem_kvm_link
        return bot_id, st_id

    def req_memory(self, server, comment=''):
        """
        Generate change MEMORY request. If you know the name of bad module
        you can specify it, using field "comment".
        """
        bot_id, st_id = self.make_request(server.name, 'MEMORY', comment)
        return bot_id, st_id

    def req_psu(self, server, comment=''):
        """
        Generate PSU change request.
        """
        bot_id, st_id = self.make_request(server.name, 'PSU', comment)
        return bot_id, st_id

    def req_reboot(self, server, comment=''):
        """
        Generate REBOOT request. If host has ipmi, it returns message
        about ipmi on server.
        """
        bot_id, st_id = '', ''
        if server.host_has_ipmi() and not self.force_mode:
            print_bold('MESSAGE', 'Host %s gas IPMI. You can use ipmitool '
                       'for reboot this host' % server.name)
        else:
            bot_id, st_id = self.make_request(server.name, 'REBOOT', comment)
        return bot_id, st_id

    def req_lan(self, server, problem='down', comment=''):
        """
        Generate LAN request. Problem can be specified using
        option "problem".
        """
        if ('100mb' not in problem and 'switchport' not in problem
            and 'down' not in problem):
            problem = 'down'
        bot_id, st_id = self.make_request(server.name, 'LAN', comment,
                                            problem=problem)
        return bot_id, st_id

    def req_ipmi(self, server, problem='ipmidown', comment=''):
        """
        Generate IPMI request. Problem can be specified using
        option "problem".
        """
        if 'ipmidown' in problem:
            ipmi_problem = '108'
        elif 'dns' in problem:
            ipmi_problem = '107'
        elif 'nosignal' in problem:
            ipmi_problem = '106'
        elif 'poweroff' in problem:
            ipmi_problem = '105'
        else:
            ipmi_problem = '108'
        bot_id, st_id = self.make_request(server.name, 'IPMI', comment,
                                            problem=ipmi_problem)
        return bot_id, st_id

    def req_hdd(self, server, slot, shelf_serial='',
                disk_serial='', comment=''):
        """
        Generate HDD request. HDD requests can be made for server and
        for storage. Requires parameter "slot".
        For changing disk in storage needs parameter "serial".
        """
        bot_id, st_id = '', ''
        if shelf_serial:
            storages = self.get_storages_info(server)
            if storages:
                for storage in storages:
                    for line in storages[storage]:
                        if shelf_serial[-8:] in line.split()[-1]:
                            bot_id, st_id = self.make_request(server.name,
                                                                'HDD', comment,
                                                                inv=storage,
                                                                slot=slot)
                            break
            else:
                print_bold('MESSAGE', 'To server is not connected shelf')
        else:
            # Server hdd request.
            bot_id, st_id = self.make_request(server.name, 'HDD',
                                                comment, serial=disk_serial,
                                                slot=slot)
        return bot_id, st_id

    # OTHER BOT REQUESTS
    def req_info(self, server):
        """
        Show information about host: Placement, hardware and connected
        storages.
        """
        server = self.get_dc(server)
        # Basic info
        print_bold('PLACEMENT', '%s%s/%s-%s' % (server.dc, server.queue,
                                                server.rack, server.unit))
        print_bold('INVENTORY', server.inv)

        # Information about body
        print_bold('', '')
        print_bold('INFORMATION FROM BOT ', '')
        request = open_url(url=self.bot_api, data='/api/consistof.php?name=%s'
                           % server.name).readlines()
        server.body_info = remove_tags(request)
        for line in server.body_info:
            print_bold('', line)
        print_bold('', '')
        # Server storages
        storages = self.get_storages_info(server)
        server.storage_info = storages
        if storages:
            print_bold('SERVER STORAGES', '')
            for storage in storages:
                for line in storages[storage]:
                    print_bold('', line)
            print_bold('TOTAL STORAGES', str(len(storages)))
        return server

    def req_actions(self, server, last_count='10'):
        """
        Show information about last actions: formalized and unformalized.
        Return only formalized actions.
        """
        if self.oauth_token:

            # Formalized actions
            server_actions = self.get_actions(server=server.name, count=last_count)
            if server_actions:
                print_bold('',
                           '\033[1m%-10s %16s %18s %8s %12s %-15s\033[0m'
                           % ('ACTION', 'DATE', 'STARTREK',
                              'STATUS', 'INITIATOR', 'COMMENT'))
                print_actions(server_actions)
            else:
                print_bold('MESSAGE', 'No actions found for server')

            # Actions with connected storages
            storages = self.get_storages_info(server)
            for storage in storages:
                storage_actions = self.get_actions(inventory=storage, count=last_count)
                if storage_actions:
                    if not server_actions:
                        print_bold('',
                                   '\033[1m%-10s %16s %18s %8s %12s %-15s\033[0m'
                                   % ('ACTION', 'DATE', 'STARTREK',
                                      'STATUS', 'INITIATOR', 'COMMENT'))
                    print_bold('\nSTORAGE', storage)
                    print_actions(storage_actions)

            # Unformalized actions
            crm_actions = self.get_actions(server=server.name, count=last_count,
                                           unformalized=True)
            if crm_actions:
                print_bold('\nUNFORMALIZED ACTIONS', '')
                for action in crm_actions:
                    print_bold('', '%-10s %16s %18s %8s %12s %-15s'
                               % (action_by_id(action['RequestType']),
                                  time_from_ts(action['TimeStampCreate']), '',
                                  action['Status'], action['Initiator'],
                                  action['ResultComment']))
        else:
            print_bold('ERROR', 'OAuth token was not found. You can get '
                                'it using option --get-token')
        return server_actions

    def get_actions(self, server='', inventory='', count='5', unformalized=False):
        """
        Get information about last actions with specified server or inventory.
        """
        data = ''
        if server:
            data = 'OsName=%s' % server
        if inventory:
            data = 'OsInv=%s' % inventory

        if unformalized:
            request = open_url(url=self.bot_api, data='api/scrm/export.php?oauth_token=%s'
                               '&osname=%s&limit=%s&format=json' % (self.oauth_token,
                                                                    server, count))
        else:
            request = open_url(url=self.bot_api, data='adm/requestshow.php?Limit=%s&%s'
                               '&oauth_token=%s&export=json' % (count, data,
                                                                self.oauth_token))
        try:
            actions = json.load(request)
        except ValueError:
            print_bold('ERROR', 'Failed to get actions. Perhaps you don\'t have '
                                '"responsible-admin" role')
            sys.exit(1)
        return actions

    def get_request_status(self, bot_id):
        request_status = open_url(url=self.bot_api,
                                  data='/api/status.php?id=%s&out=json' % bot_id)
        task_info = json.load(request_status)
        status = task_info.get('Status')
        return status

    def get_storages_info(self, server):
        """
        Get information about connected storages.
        """
        storages = {}
        if not server.inv:
            server.inv = open_url(url=self.bot_api,
                                  data='/api/osinfo.php?fqdn=%s&output=instance_number'
                                       '&format=simple'
                                       % server.name).readlines()[-1].split()[0]
        storage_inv_list = open_url(url=self.bot_api,
                                    data='/api/storages-by-serverinv.php?inv=%s'
                                    % server.inv).read().split()
        if 'YES' in storage_inv_list[0]:
            for inv in storage_inv_list[1:]:
                storage_info = open_url(url=self.bot_api,
                                        data='/api/consistof.php?inv=%s' % inv
                                        ).readlines()
                storages[inv] = remove_tags(storage_info)
        return storages

    def get_dc(self, server):
        """
        Get information about DC placement.
        """
        try:
            result = open_url(url=self.bot_api,
                              data='/api/osinfo.php?fqdn=%s&output=loc_segment4'
                                   '|loc_segment5|loc_segment6|instance_number'
                                   % server.name)
            host_dc = json.load(result).get('os')[0]
            server.dc = str(host_dc.get('loc_segment4').split('-')[0])
            server.queue = str(host_dc.get('loc_segment4').split('-')[1])
            server.rack = str(host_dc.get('loc_segment5'))
            server.unit = str(host_dc.get('loc_segment6'))
            server.inv = str(host_dc.get('instance_number'))
        except (AttributeError, IndexError):
            print_bold('ERROR', 'Failed to get information about host')
        return server

    def rename_server(self, server, fqdn):
        """
        Rename host function. Requires fqdn as argument.
        """
        print_bold('ACTION', 'Rename')

        server = self.get_dc(server)
        request_data = urllib.urlencode({"oauth_token": self.oauth_token,
                        "data": '[{"new_fqdn":"%s","inv":"%s"}]' % (fqdn, server.inv)})
        try:
            request = urllib2.Request(url='%s/adm/rename.php' % self.bot_api, data=request_data)
            result = json.load(open_url(instance=request))
            if result:
                print_bold('MESSAGE', result.get('Ok').replace("<br />", " ").replace("\n", " ")
)
        except urllib2.HTTPError:
            print_bold('ERROR', 'Failed to rename server %s to %s' % (server.name, fqdn))

    def send_action(self, server, req_type, status, comment=''):
        """
        Send unformalized action to BOT-CRM.
        """
        request_data = urllib.urlencode({'oauth_token': self.oauth_token,
                                         'initiator': self.initiator,
                                         'osname': server.name,
                                         'source': 'ADMIN',
                                         'requesttype': req_type,
                                         'status': status,
                                         'comment': comment})
        try:
            request = open_url(url=self.bot_api, data='/api/scrm/import.php?%s'
                                                      % request_data)
        except urllib2.HTTPError:
            print_bold('ERROR', 'Failed to send action to BOT')

    def make_request(self, hostname, req_type, comment,
                     slot='', serial='', problem='', inv=''):
        """
        Main function for sending requests. Universal for all formalized
        requests. Requires "req_type" as argument.
        Also "slot", "problem" and "inv" for some requests.
        """
        print_bold('ACTION', '%s request' % req_type)
        if inv:
            host_ident = 'inv=%s' % inv
        else:
            host_ident = 'name=%s' % hostname
        operation = req_string(req_type, slot=slot,
                               problem=problem, serial=serial)
        if comment:
            comment = urllib2.quote(comment)
        request = ('/api/request.php?%s&initiator=%s&%s&email=yes&reboot=%s&'
                   'oauth_token=%s&boot=%s&problem=%s&replyto=%s&cc'
                   '=%s&needcall=%s&comment=%s' % (host_ident,
                                                   self.initiator, operation,
                                                   self.reboot,
                                                   self.oauth_token,
                                                   self.boot, problem,
                                                   self.email_replyto,
                                                   self.email_cc,
                                                   self.needcall,
                                                   comment))
        try:
            request_id = open_url(url=self.bot_api, data=request).readlines()[0]
        except urllib2.HTTPError:
            print_bold('ERROR', 'Request failed')
            sys.exit(1)

        request_status = open_url(url=self.bot_api,
                                  data='/api/status.php?id=%s&out=json'
                                       % request_id)
        print_bold('BOT', request_id)
        try:
            task_info = json.load(request_status)
            st_id = task_info['StNum']
            print_bold('STARTREK', st_id)
        except ValueError:
            print_bold('ERROR', 'Cannot get STARTREK task id')
            st_id = ''
        return request_id, st_id


# BASIC HOSTNAME OBJECT
class HostnameInfo(object):
    """
    Class for containing information about host.
    """
    def __init__(self, name, host_dc='', host_rack='',
                 host_unit='', host_queue='', inv='',
                 ipmi=False, kvm='', body_info='',
                 storage_info='', console=''):
        self.name = name
        self.inv = inv
        self.dc = host_dc
        self.queue = host_queue
        self.rack = host_rack
        self.unit = host_unit
        self.ipmi = ipmi
        self.kvm = kvm
        self.body_info = body_info
        self.storage_info = storage_info
        self.console = console

        print_bold('SERVER', self.name)

    def host_has_ipmi(self):
        """
        Checking for availability IPMI on host object.
        """
        ipmi_hostname = self.name + '.ipmi.yandex-team.ru'
        try:
            socket.getaddrinfo(ipmi_hostname, socket.AF_INET6)
            self.ipmi = True
        except socket.error:
            self.ipmi = False
        return self.ipmi


# OTHER REQUESTS
def unformalized_req(bot, server, theme, comment):
    """
    Send unformalized request to helpdc@yandex-team.ru.
    Requires "theme" and "comment" as arguments.
    """
    print_bold('ACTION', 'Other request')
    server = bot.get_dc(server)
    sender = bot.email_from
    reciever = 'helpdc@yandex-team.ru'
    message_text = """
    Добрый день.
    Сервер: %s
    Расположение: %s-%s/%s-%s

    Комментарий:
    %s

    Спасибо.
    Заявка сгенерирована модулем make_req.py
    """ % (server.name, server.dc, server.queue,
           server.rack, server.unit, comment)
    subject = '[%s] %s %s (%s)' % (server.dc, theme, server.name, server.inv)
    message = MIMEText(message_text, 'plain', 'utf-8')
    message['From'] = sender
    message['To'] = reciever
    message['Cc'] = bot.email_cc
    message['Subject'] = subject.decode('utf-8')
    try:
        request = smtplib.SMTP('localhost')
    except StandardError:
        request = smtplib.SMTP('omail.yandex.ru')

    try:
        request.sendmail(sender, (reciever, sender,), message.as_string())
        print_bold('MESSAGE', 'E-mail was sent')
        bot.send_action(server, 7, 'START', comment)
        bot.send_action(server, 7, 'SUCCESS', comment)
    except smtplib.SMTPException:
        print_bold('ERROR', 'Cannot send message')


def manage_downtime(bot, server, service, state_or_time, comment=''):
    """
    Add|show|remove downtime for host.
    Requires "service" and "state_or_time" as argument.
    """
    print_bold('ACTION', 'Downtime')

    juggler_server = find_juggler_server('search_instrum-juggler_master', 8998)
    juggler_api = '%s/api/downtimes' % juggler_server

    request_header = {'User-Agent': 'make_req.py'}
    if 'status' in state_or_time:
        request = ('%s/downtimes?do=1&object_name=%s' % (juggler_api,
                                                         server.name))
    elif 'remove' in state_or_time:
        request = ('%s/remove_downtime?do=1&object_name=%s&'
                   'service_name=%s' % (juggler_api, server.name, service))
    else:
        req_data = urllib.urlencode({'object_name': server.name,
                                     'service_name': service, 'do': '1',
                                     'end_time': '+%s' % state_or_time,
                                     'description': comment})
        request = '%s/set_downtime?%s' % (juggler_api, req_data)

        if not comment:
            comment = 'Downtime on %s' % state_or_time
        bot.send_action(server, 6, 'START', comment)
        bot.send_action(server, 6, 'SUCCESS', comment)

    req = urllib2.Request(url=request, headers=request_header)
    resp = open_url(instance=req)
    result = json.load(resp)
    if result:
        for service in result['HOST'][server.name]:
            result_by_service = result['HOST'][server.name][service]
            end_time = time_from_ts(result_by_service['end_time'])
            description = result_by_service['description']
            comment = 'With message: %s' % description if description else ''
            print_bold('MESSAGE', 'Service: %s | Downtimed till: %s %s'
                       % (service, end_time, comment))
    else:
        print_bold('MESSAGE', 'Downtime action complete')


def startrek_associate(bot, st_dest_id, st_source_id):
    """
    Associate task id with specify STARTREK id.
    Requires STARTREK source id and destination id as arguments.
    """
    request_header = {'Authorization': 'OAuth %s' % bot.oauth_token,
                      'Content-type': 'application/json',
                      'User-Agent': 'make_req.py'}
    request_data = json.dumps({'relationship': 'depends on',
                               'issue': st_source_id})
    request = urllib2.Request(url='http://st-api.yandex-team.ru/v2/issues/'
                                  '%s/links' % st_dest_id, data=request_data,
                              headers=request_header)
    resp = open_url(instance=request).readlines()[0]
    if len(resp) > 2:
        print_bold('MESSAGE', 'Request was associated with %s' % st_dest_id)
    else:
        print_bold('MESSAGE', 'Associating with %s failed' % st_dest_id)


# TOKENS
def req_oauth_token(cfg_dict):
    """
    Main function for requesting OAuth token.
    Requires config as argument.
    """
    print_bold('ACTION', 'Get OAuth token')
    token = get_oauth_token()
    cfg_dict['TOKENS']['oauth_token'] = token
    path = get_config_path()
    write_configuration(path, cfg_dict)
    print_bold('MESSAGE', 'Token writed to %s' % path)


def get_oauth_token():
    """
    Get OAuth token.
    Asks user for a password.
    """
    user = getpass.getuser()
    print 'User: %s' % user
    password = getpass.getpass()
    auth = urllib.urlencode({'grant_type': 'password', 'username': user,
                             'password': password,
                             'client_id': '662aa591ebb04a999d2a4faa316a097a',
                             'client_secret': '87224eb4ac514d2198e22b48166fc04c'
                             })
    request = urllib2.Request(url='https://oauth.yandex-team.ru/token',
                              data=auth)
    try:
        response = open_url(instance=request)
        oauth_token = json.load(response)
        return oauth_token['access_token']
    except urllib2.HTTPError:
        print_bold('ERROR', 'Invalid password')
        sys.exit(1)


# MISC
def remove_tags(data):
    """
    Parsing data for tags and clear data.
    """
    clean_data = []
    clean_out = re.compile(r'<.*?>')
    for line in data:
        clean_line = clean_out.sub(' ', line).strip()
        if len(clean_line) > 0:
            clean_data.append(clean_line)
    return clean_data


def open_url(**kwargs):
    """
    Open specified URL with data.
    """
    url = None
    try:
        url = kwargs.get('url', None)
        data = kwargs.get('data', None)
        instance = kwargs.get('instance', None)
        if url and data:
            result = urlopen("%s/%s" % (url, data))
        elif instance:
            url = instance.get_host()
            result = urlopen(instance)
        else:
            print_bold('ERROR', 'Cannot create HTTP request')
            sys.exit(1)
    except urllib2.URLError, err:
        print_bold('ERROR', 'Server %s returns error %s' % (url, err.code))
        sys.exit(1)

    return result


def host_fqdn(host):
    """
    Converting host shortname to FQDN.
    """
    fqdn = socket.getfqdn(host)
    return fqdn


def time_from_ts(timestamp):
    """
    Converting timestamp for readable format.
    """
    return (datetime.datetime.fromtimestamp(
        int(timestamp)).strftime('%d.%m.%Y %H:%M'))


def action_by_id(actionid):
    """
    Converting action number into a readable form.
    """
    action = ''
    if '2' in actionid:
        action = 'FORMAL'
    elif '3' in actionid:
        action = 'RAID RESYNC'
    elif '4' in actionid:
        action = 'RAID REPAIR'
    elif '5' in actionid:
        action = 'LAN REPAIR'
    elif '6' in actionid:
        action = 'DOWNTIME'
    elif '7' in actionid:
        action = 'OTHER'
    elif '8' in actionid:
        action = 'RESETUP'
    return action


def print_actions(actions):
    """
    Print json actions.
    """
    for action in actions:
        print_bold('', '%-10s %16s %18s %8s %12s %-15s'
                   % (action.get('RequestType', 'UNKNOWN'),
                       action['TimeCreate'],
                       action.get('StNum', 'NONE'),
                       action.get('Status', 'UNKNOWN'),
                       action['Initiator'],
                       action.get('ResultComment', '')))


def print_bold(bmsg, msg):
    """
    Print message if running from console.
    Requires 2 arguments: bold_message and message.
    """
    if type(bmsg) is unicode:
        bmsg = bmsg.encode('utf-8')
    if type(msg) is unicode:
        msg = msg.encode('utf-8')

    if __name__ == '__main__':
        if len(bmsg) > 0:
            out_string = '\033[1m%-9s\033[0m %s' % (bmsg, msg)
        else:
            out_string = msg
        print out_string


# CONFIGURATION
def get_configuration():
    """
    Main function for getting configuration.
    """
    cfg = {}
    path = get_config_path()
    config = ConfigParser.ConfigParser()
    if config.read(path) and len(config.sections()) > 0:
        for section in config.sections():
            cfg[section] = {}
            for option in config.options(section):
                cfg[section][option] = config.get(section, option)
    else:
        cfg = default_configuration()
        write_configuration(path, cfg)
    return cfg


def write_configuration(path, cfg):
    """
    Write configuration to specify path.
    """
    config = ConfigParser.ConfigParser()
    for section in cfg:
        config.add_section(section)
        for option in cfg[section]:
            config.set(section, option, cfg[section][option])
    with open(path, 'w') as config_file:
        config.write(config_file)


def default_configuration():
    """
    Return default configuration using user data.
    """
    user = pwd.getpwuid(os.geteuid()).pw_name
    cfg = {'REQUEST': {
           'initiator': user,
           'email_from': user + '@yandex-team.ru',
           'email_cc': user + '@yandex-team.ru'},
           'TOKENS': {}}
    return cfg


def generate_with_config(config, options=None):
    """
    Generate and return BotRequest instance using specified
    configuration and options.
    """
    reboot = False
    oauth_token = config['TOKENS'].get('oauth_token', '')
    if options:
        read_only = options.read_only
        force_mode = options.force_mode
        needcall = options.needcall
        reboot = options.reboot_request
    else:
        read_only = config['REQUEST'].get('read_only', False)
        force_mode = config['REQUEST'].get('force_mode', False)
        needcall = config['REQUEST'].get('needcall', False)
    return BotRequest(initiator=config['REQUEST']['initiator'],
                      email_cc=config['REQUEST']['email_cc'],
                      email_from=config['REQUEST']['email_from'],
                      read_only=read_only, force_mode=force_mode,
                      needcall=needcall, reboot=reboot,
                      oauth_token=oauth_token)


def get_config_path():
    """
    Define configuration path.
    """
    default_path = os.path.dirname(__file__)
    reserve_path = os.environ['HOME']
    if os.path.isfile('./make_req.conf'):
        path = './make_req.conf'
    else:
        path = reserve_path + '/.make_req.conf'
    return path


def find_juggler_server(cgroup, port):
    """
    Find available juggler server using conductor-api
    """
    conductor_api = 'https://c.yandex-team.ru/api-cached'
    server_list = urllib2.urlopen('%s/groups2hosts/%s' % (conductor_api, cgroup)).readlines()
    server_list.append('jmon-master.search.yandex.net')
    for server in server_list:
        server = server.strip()
        if check_server(server, port):
            return 'http://%s:%d' % (server, port)
    Message.error('No reachable juggler server found.')
    sys.exit(1)


def check_server(server, port):
    """
    Check availability of server on port
    """
    try:
        host_ip = socket.gethostbyname(server)
        conn = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        conn.settimeout(0.5)
        conn.connect((host_ip, port))
        conn.close()
        return True
    except:
        return False


# MANAGE REQUEST
def req_string(req_type, slot='', problem='', serial=''):
    """
    Make part of request string. Uses for method make_request
    of class  BotRequest.
    Result depends on the given parameters.
    """
    request_string = ''
    if 'HDD' in req_type:
        request_string = 'operation=hdd&slot=%s&serial=%s' % (slot, serial)
    elif 'KVM' in req_type:
        request_string = 'operation=kvm'
    elif 'MEMORY' in req_type:
        request_string = 'operation=memory&realmem=64&needmem=64'
    elif 'IPMI' in req_type:
        request_string = 'operation=check_ipmi&problem=%s' % problem
    elif 'LAN' in req_type:
        request_string = 'operation=check_eth&problem=%s' % problem
    elif 'PSU' in req_type:
        request_string = 'operation=psu'
    elif 'REBOOT' in req_type:
        request_string = 'operation=reboot'
    return request_string


def run_handler(bot, server, options):
    """
    Read options and run corresponding request.
    """
    bid = ''
    startrek = ''
    if options.kvm_request:
        bid, startrek = bot.req_kvm(server, comment=options.comment)
    elif options.reboot_request:
        bid, startrek = bot.req_reboot(server, comment=options.comment)
    elif options.memory_request:
        bid, startrek = bot.req_memory(server, comment=options.comment)
    elif options.lan_request:
        bid, startrek = bot.req_lan(server, problem=options.problem_name,
                                comment=options.comment)
    elif options.ipmi_request:
        bid, startrek = bot.req_ipmi(server, problem=options.problem_name,
                                 comment=options.comment)
    elif options.hdd_slot is not None:
        bid, startrek = bot.req_hdd(server, options.hdd_slot,
                                shelf_serial=options.shelf_disk_serial,
                                disk_serial=options.disk_serial,
                                comment=options.comment)
    elif options.psu_request:
        bid, startrek = bot.req_psu(server, comment=options.comment)
    elif options.info_request:
        bot.req_info(server)
    elif options.actions_request:
        bot.req_actions(server)
    elif options.server_fqdn:
        bot.rename_server(server, options.server_fqdn)
    elif options.theme and options.comment:
        unformalized_req(bot, server, options.theme, options.comment)

    if options.dt_time and options.dt_service:
        manage_downtime(bot, server,
                        options.dt_service,
                        options.dt_time,
                        comment=options.comment)
    if options.st_id and startrek:
        startrek_associate(bot, options.st_id, startrek)


# OPTIONS
def arg_parser(*args):
    """
    Specifying options and help function.
    """
    options = argparse.ArgumentParser(
        formatter_class=argparse.RawTextHelpFormatter,
        description="""
DESCRIPTION
  Sends requests for helpdesk and displays information about host.
                    """,
        epilog="""
EXAMPLES
    Sends request for reboot server ws10-000.yandex.ru:
        make_req.py -R ws10-000.yandex.ru

    Sends request for replace memory for ws10-000.yandex.ru:
        make_req.py -M -c CPU0DIMM1B ws10-000.yandex.ru

    Sends request for replace 1 from 0 disk for ws10-000.yandex.ru:
        make_req.py -H 1 ws10-000.yandex.ru

    Sends request for replace 10 from 0 disk in shelf for ws10-000.yandex.ru:
        make_req.py -H 10 -d 'HZRSGTEFS45' ws10-000.yandex.ru
    Where 'HZRSGTEFS45' serial number of any disk in target shelf.

    Sends request for reboot server ws10-000.yandex.ru and boot from lan:
        make_req.py -Rl ws10-000.yandex.ru

    Sends request for reboot server ws10-000.yandex.ru and call before
    performing:
        make_req.py -R --needcall ws10-000.yandex.ru

    Sends request for show already connected KVM for list of hosts without
    send a request:
        printf "ws10-000\\nws10-001\\nws10-002" | ${script_name} -Kr

    Sends request for KVM and reboot server ws10-000.yandex.ru:
        make_req.py -KR ws10-000.yandex.ru

    Adds downtime for host ws10-000.yandex.ru to 2 days:
        make_req.py -D 2days ws10-000.yandex.ru

    Removes downtime for host ws10-000.yandex.ru:
        make_req.py -D remove ws10-000.yandex.ru

    Sends request for make 1Gbit/s link instread of 100Mbit/s:
        make_req.py -L --problem 100mb ws10-000.yandex.ru

    Views information about host ws10-000.yandex.ru:
        make_req.py -S ws10-000.yandex.ru

    Views information about actions with host ws10-000.yandex.ru:
        make_req.py -A ws10-000.yandex.ru

ADDITIONAL
    For usage option -A, need to get role "responsible-admin" to link:
    https://idm.yandex-team.ru/?system=bot&type=role&role=responsible-admin
""")
    options.add_argument('host', type=str, nargs='?',
                         help='Specifies host.')
    options.add_argument('-K', '--kvm', dest='kvm_request', action='store_true',
                         help="""This option sends request to KVM for server
If specified host has ipmi, script shows link to IPMI-console
If KVM is already connected, script shows link to KVM-console.

""")
    options.add_argument('-R', '--reboot', dest='reboot_request',
                         action='store_true',
                         help="""Sends request to reboot a specific server.
Can use with option -H for hdd request.

""")
    options.add_argument('-I', '--ipmi', dest='ipmi_request',
                         action='store_true',
                         help="""Sends request to host with IPMI problems.
Problem can be specified using option -p.

""")
    options.add_argument('-P', '--psu', dest='psu_request',
                         action='store_true',
                         help="""Sends request to change power supply unit.

""")
    options.add_argument('-L', '--lan', dest='lan_request',
                         action='store_true',
                         help="""Sends request to host with LAN problems.
Problem can be specified using option -p.

""")
    options.add_argument('-M', '--memory', dest='memory_request',
                         action='store_true',
                         help="""Sends request to replace memory.
You can set bad memory module using option -c.

""")
    options.add_argument('-S', '--show-info', dest='info_request',
                         action='store_true',
                         help="""Shows information about host, its location
and storages.

""")
    options.add_argument('-A', '--actions', dest='actions_request',
                         action='store_true',
                         help="""Shows history of last actions with specified \
server. Requires "responsible-admin" role.

""")
    options.add_argument('-H', '--hdd', dest='hdd_slot',
                         help="""Sends request to replace a disk
Requires an argument <slotnum>
For disk changing in shelf need to specify serial number any
disk in shelf using option --shelf.

""")
    options.add_argument('-N', '--rename', dest='server_fqdn',
                         help="""Change server name to the specified fqdn.
""")
    options.add_argument('-D', '--downtime', dest='dt_time', type=str,
                         help="""Adds downtime for host. Downtime can be \
removed with option
"remove" and look on downtime status with option "status".
Also you can specify service (default UNREACHABLE) using option --dt-service

""")
    options.add_argument('-t', '--theme', type=str, default='',
                         help="""Sets theme on other request by email.

""")
    options.add_argument('-c', '--comment', type=str, default='',
                         help="""Adds comment as argument.

""")
    options.add_argument('-r', '--read', dest='read_only',
                         action='store_true',
                         help="""'read only' mode for option -K.
Shows already connected KVM or display message
that request is needed.

""")
    options.add_argument('-f', '--force', dest='force_mode',
                         action='store_true',
                         help="""force" mode for option -K, -R and -I.
This option ignores any checks and run command.

""")
    options.add_argument('--dt-service', dest='dt_service', type=str,
                         default='UNREACHABLE',
                         help="""Using only with option -D.
Specify service for downtime action.

""")
    options.add_argument('--shelf', dest='shelf_disk_serial', type=str,
                         help="""Using only with option -H.
Initiate SHELF HDD request. Requires serial number of one of drives,
connected to storage.

""")
    options.add_argument('--serial', dest='disk_serial', type=str,
                         default='', help="""Using only with option -H.
Specify serial number for change disk request.

""")
    options.add_argument('--problem', dest='problem_name',
                         type=str, default='',
                         help="""Clarifies problem. Used with -L or -I option.
Requires specifying argument. List of available values:
For LAN problems:
    down,
    100mb,
    switchport.
For IPMI problems:
    ipmidown,
    dns,
    nosignal,
    poweroff.

""")
    options.add_argument('--needcall', dest='needcall',
                         action='store_true',
                         help="""Identifies the need in a telephone
call before the action.

""")
    options.add_argument('--startrek', dest='st_id', type=str,
                         help="""Associate request STARTREK task id with specified \
STARTREK task id.
Requires oauth-token.

""")
    options.add_argument('--get-token', dest='oauth_token',
                         action='store_true',
                         help="""Get OAuth-token and write to config file""")

    if args:
        opts = options.parse_args(args)
    else:
        opts = options.parse_args()

    if opts.oauth_token:
        opts.host = ''
    elif not opts.host:
        opts.host = sys.stdin.readlines()
    else:
        opts.host = opts.host.split()
    return opts


def main(*args):
    config = get_configuration()
    opts = arg_parser(*args)
    if opts.oauth_token:
        req_oauth_token(config)
    else:
        bot = generate_with_config(config, options=opts)
        for host in opts.host:
            host_name = host_fqdn(host)
            host_obj = HostnameInfo(host_name)
            run_handler(bot, host_obj, opts)
            print_bold('', '')


if __name__ == '__main__':
    main()
