#!/bin/sh
#
# $Id$
#
# This script migrates directories passed as parameters from CVS to SVN
#

#-- Subroutines --------------------------------------------------------

err()
{
    local _exitval

    _exitval=$1
    shift

    echo 1>&2 "ERROR: $*"
    exit $_exitval
}

verbose()
{
    local _d

    if [ -n "$verbose" ]; then
        _d=$($date "+%F %T")
        echo 1>&2 "[${_d}] VERBOSE: $*"
    fi
}

usage()
{
    echo 1>&2 "Usage: ${0} [-v] [-u] [-U] [-N] dirs"
    echo 1>&2 "Options:"
    echo 1>&2 "  -v           verbose output"
    echo 1>&2 "  -u           run svn update if already converted"
    echo 1>&2 "  -N           checkout from svn without subdirectories"
    echo 1>&2 "  -U           user for svn commands"

    exit 2
}

get_options()
{
    local _opt

    while getopts "vuNU:" _opt; do
        case "$_opt" in
            v) verbose="yes" ;;
            u) update="yes" ;;
            N) svn_Nflag="-N" ;;
            U) export "SVN_SSH=$ssh -q -l $OPTARG" ;;
            *) usage ;;
        esac
    done

    return $(($OPTIND - 1))
}

#
# Copy permissions from file to file
# Returns non-zero in case of an error
#
copy_owner() {
    local _os _from _to _owner
    if [ ! -e "$1" ] || [ ! -e "$2" ]; then
        verbose "Either 'from' or 'to' does not exist"
        return 1
    fi
    _from=$1
    _to=$2

    _os=$(uname -s)
    case "$_os" in
        Linux)
            _owner=$($stat -t $_from | $awk '{printf($5":"$6)}')
        ;;
        FreeBSD)
            _owner=$(eval $($stat -s $_from); echo "$st_uid:$st_gid")
        ;;
        *)
            err 1 "Unknown OS"
            return 1
        ;;
    esac
    if [ -z "$_owner" ]; then
        echo "Can't determinate owner of $_from"
        return 1
    fi

    if [ $(id -u) -eq 0 ]; then
        $chown $_owner $_to
    fi

    return $?
}


#
# Prints path from CVS/Repository
# Returns 1 in case of error
#
get_repo_path_from_cvs() {
    local _dir _cvs_repo_path
    if [ ! -d "$1" ]; then
        verbose "Directory passed to get_repo_path_from_cvs($1) does not exist"
        return 1
    fi
    _dir="$1"
    _cvs_repo_path="$_dir/CVS/Repository"

    if [ ! -r "$_cvs_repo_path" ]; then
        verbose "Can't read: $_cvs_repo_path"
        return 1
    fi
    cat "$_cvs_repo_path"
    return $?
}

#
# Check that object exists in SVN
#
exists_in_svn() {
    local _dir _cvs_repo_path
    if [ ! -d "$1" ]; then
        verbose "Directory passed to exists_in_svn($1) does not exist"
        return 1
    fi
    _dir="$1"
    _cvs_repo_path="$(get_repo_path_from_cvs $_dir)"

    if [ -z "$_cvs_repo_path" ]; then
        verbose "Empty output from get_cvs_repo_path()"
        return 1
    fi

    $svn info $svn_repo/$_cvs_repo_path >/dev/null 2>&1
    return $?
}

#
# Moves parent folder's CVS dir to CVS.CVS to prevent human factor like `cvs up`
# on migrated folder
#
# $1 - path to dir
#
move_parent_cvs_dir() {
    local _dir _up_cvs_dir

    if [ ! -d "$1" ]; then
        verbose "Directory passed to move_parent_cvs_dir($1) does not exist"
        return 1
    fi
    _dir="$1"
    _up_cvs_dir=$($dirname $_dir)/CVS

    if [ ! -d $_up_cvs_dir ]; then
        return 1
    fi
    $mv $_up_cvs_dir $_up_cvs_dir.CVS
    return $?
}

#
# Checks out similar dir from SVN and puts it to $_dir.SVN folder
#
# $1 - path to dir
checkout_similar_from_svn() {
    local _path_in_repo _dir

    if [ ! -d "$1" ]; then
        verbose "Directory passed to checkout_similar_from_svn($1) does not exist"
        return 1
    fi
    _dir="$1"

    if [ ! -d "$_dir/CVS" ] || [ "${_dir%CVS*}" != "${_dir}" ]; then
        verbose "Directory $_dir is not a CVS dir (or already converted)"
        return 1
    fi

    _path_in_repo="$(get_repo_path_from_cvs $_dir)"
    if [ -z $_path_in_repo ]; then
        verbose "get_repo_path_from_cvs($_dir) failed"
        return 1
    fi

    $svn co $svn_Nflag $svn_repo/$_path_in_repo $_dir.SVN >/dev/null
    return $?
}

#
# Moves CVS to backup dir and moves .SVN folder in it's place
#
# $1 - path to dir
replace_cvs_with_svn() {
    local _dir
    if [ ! -d "$1" ]; then
        verbose "Directory passed to replace_cvs_with_svn($1) does not exist"
        return 1
    fi
    _dir="$1"

    if [ ! -d $_dir/CVS ] || [ ! -d $_dir.SVN/.svn ]; then
        verbose "Either $_dir is not CVS or $_dir.SVN is not SVN"
        return 1
    fi

    # XXX: Race here!
    $mv $_dir $_dir.CVS
    $mv $_dir.SVN $_dir

    # Copy owners from CVS
    # XXX: Copy permissions?
    for f in `$find $_dir -not -path \*/.svn\*`; do
        f="${f##$_dir/}"
        if [ -e "$_dir.CVS/$f" ] ; then
            copy_owner "$_dir.CVS/$f" "$_dir/$f"
        fi
    done

    # XXX:
    return 0
}

#
# Updates SVN in dir 
#
# $1 - path to dir
update_svn() {
    local _dir
    if [ ! -d "$1" ]; then
        verbose "Directory passed to replace_cvs_with_svn($1) does not exist"
        return 1
    fi
    _dir="$1"

    if [ ! -d $_dir/.svn ]; then
        verbose "Can't update. $_dir is not SVN"
        return 1
    fi
    $svn up $svn_Nflag $_dir >/dev/null
    return $?
}

#-- Variables ----------------------------------------------------------

: ${which:=$(which which)}
: ${which:=/usr/bin/which}
: ${mv:=$($which mv)}
: ${svn:=$($which svn)}
: ${ssh:=$($which ssh)}
: ${awk:=$($which awk)}
: ${date:=$($which date)}
: ${find:=$($which find)}
: ${stat:=$($which stat)}
: ${uname:=$($which uname)}
: ${chown:=$($which chown)}
: ${dirname:=$($which dirname)}
: ${svn_proto:="svn+ssh://"}
: ${svn_repo:="${svn_proto}arcadia.yandex.ru/arc/trunk"}
: ${svn_Nflag:=""}
: ${update:=""}
: ${verbose:=""}
: ${converted_dirs:=""}

#-- Main ---------------------------------------------------------------

get_options $*
shift $?

if [ -z "$*" ]; then
    usage
fi
dirs=$*

for dir in $dirs; do
    if [ "${_dir##*/}" = "CVS" ]; then
        continue
    fi
    if [ -d "$dir/.svn" ]; then
        verbose "$dir is already converted to SVN"
        if [ -n "$update" ]; then
            update_svn $dir
        fi
        continue
    fi
    if [ ! -d "$dir/CVS" ]; then
        verbose "$dir is not under CVS"
        continue
    fi
    move_parent_cvs_dir "$dir"
    if ! exists_in_svn $dir; then
        verbose "$dir does not exists in SVN!"
        continue
    fi
    if ! checkout_similar_from_svn "$dir"; then
        verbose "Failed to checkout: $dir"
        continue
    fi
    converted_dirs="$dir $converted_dirs"
done

for dir in $converted_dirs; do
    if exists_in_svn "$dir"; then
        replace_cvs_with_svn "$dir"
    fi
done
