#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
This module provide interface human <--> juggler server.
"""
import sys
import urllib2
import json
import os
import socket
from multiprocessing import Pool
import argparse


class HostInfo():
    """
    Class for collect information about host.
    """
    def __init__(self, name='No data', dc='', service='',
                 description='', status='', action='', ipmi=''):
        self.name = name
        self.dc = dc
        self.service = service
        self.description = description
        self.status = status
        self.action = action
        self.ipmi = ipmi
        self.output = ''

    def __str__(self):
        if self.dc and self.description:
            self.output = ('%-28s %-24s %-40s'
                           % (self.name,
                              col('blue', self.dc),
                              col('gray', self.description)))
        elif self.dc and self.action:
            self.output = ('%-28s %-24s %-12s %-7s %-18s %-15s'
                           % (self.name, col('blue', self.dc),
                              self.action['action'],
                              self.action['status'],
                              self.action['otrs_id'],
                              self.action['time_start']))
        elif self.description:
            self.output = ('%-28s %-40s' % (self.name,
                                            col('gray', self.description)))
        elif self.action:
            self.output = ('%-28s %-12s %-7s %-18s %-15s'
                           % (self.name, self.action['action'],
                              self.action['status'],
                              self.action['otrs_id'],
                              self.action['time_start']))
        elif self.dc:
            self.output = ('%-28s %-22s' % (self.name, col('blue', self.dc)))
        else:
            self.output = self.name
        return self.output


# MESSAGES AND COLORS
class MessageHandler(object):
    """
    Handling service messages.
    """
    @staticmethod
    def error(text):
        sys.stderr.write('| ' + col('red', 'ERROR')
                         + ' | ' + text + '\n')

    @staticmethod
    def msg(text):
        sys.stderr.write('| ' + col('green', 'MESSAGE')
                         + ' | ' + text + '\n')

    @staticmethod
    def warn(text):
        sys.stderr.write('| ' + col('yellow', 'WARNING')
                         + ' | ' + text + '\n')


def col(color, text):
    """
    Paints the specified text to the specified color.
    Requires 2 arguments: color and text.
    """
    end = '\033[0m'
    if 'red' in color:
        color = '\033[91m'
    elif 'gray' in color:
        color = '\033[90m'
    elif 'green' in color:
        color = '\033[92m'
    elif 'blue' in color:
        color = '\033[94m'
    elif 'white' in color:
        color = '\033[97m'
    elif 'yellow' in color:
        color = '\033[93m'
    return color + str(text) + end


# JUGGLER FUNCTIONS
def juggler_request(args):
    """
    Makes request to juggler server using specified parameters.
    """
    hostgroup = args.hostgroup
    service = args.service
    show_message = args.show_message
    out_dict = {}
    request = urllib2.Request(url='%s/api/events/complete_events?&do=1&'
                                  'host_name=%s&service_name=%s&all_flag=yes'
                                  % (juggler_server, hostgroup, service),
                              headers=request_header)
    try:
        openurl = urllib2.urlopen(request)
        json_out = json.load(openurl)
        result_host = json_out.get(hostgroup)
        result_descr = result_host.get(service).get('description')[0]
        result_body = result_descr.get('children')
        if result_body:
            for host in result_body:
                service = result_body[host].keys()[0]
                status = result_body[host][service]['']['status']
                if show_message:
                    description = result_body[host][service]['']['description']
                else:
                    description = ''
                out_dict[host] = HostInfo(name=host, service=service,
                                          status=status,
                                          description=description)
    except AttributeError:
        call.error('Hostgroup or service was not found')
        sys.exit(1)

    return out_dict


def juggler_list_checks(args):
    """
    Makes request to juggler server and returns list of services by hostgroup.
    """
    hostgroup = args.hostgroup
    out_list = []
    try:
        request = urllib2.Request(url='%s/api/checks/list_checks?do=1&host_name'
                                      '=%s' % (juggler_server, hostgroup),
                                  headers=request_header)
        result = urllib2.urlopen(request)
        json_result = json.load(result)
        for service in json_result[hostgroup]:
            out_list.append(service)
    except (KeyError, urllib2.HTTPError):
        call.error('Hostgroup was not found')
        sys.exit(1)
    return out_list


# HOST INFO FUNCTIONS
def get_dc(host):
    """
    Get server placement.
    """
    try:
        result = urllib2.urlopen('%s/api/osinfo.php?fqdn=%s\
&output=loc_segment4|loc_segment5|loc_segment6' % (bot_server, host))
        host_dc = json.load(result).get('os')[0]
        host_dc = '%s/%s-%s' % (host_dc.get('loc_segment4'),
                                host_dc.get('loc_segment5'),
                                host_dc.get('loc_segment6'))
    except AttributeError:
        return 'No data'
    return host_dc


def last_action(host):
    """
    Get last formalized BOT task for host.
    """
    action = ''
    try:
        result = urllib2.urlopen('%s/adm/requestshow.php?Limit=1&\
Initiator=&OsName=%s&token=%s&export=json' % (bot_server, host, user))
        task_data = json.load(result)[0]
        action = {'bot_id': task_data.get('Id'),
                  'action': task_data.get('RequestType'),
                  'status': task_data.get('Status'),
                  'time_start': task_data.get('TimeCreate'),
                  'time_end': task_data.get('TimeFinish'),
                  'otrs_id': task_data.get('OtrsNum')}
    except IndexError:
        pass
    return action


def ipmi_check(host_obj, args):
    """
    Check server for ipmi address.
    """
    if args.show_ipmi or args.show_kvm:
        ipmi_hostname = host_obj.name + '.ipmi.yandex-team.ru'
        try:
            socket.getaddrinfo(ipmi_hostname, socket.AF_INET6)
            host_obj.ipmi = True
        except socket.error:
            host_obj.ipmi = False
    return host_obj


def get_host_info(host_obj, args):
    """
    Management function for get additional information about hosts.
    """
    if args.show_dc:
        host_obj.dc = get_dc(host_obj.name)
    if args.show_action:
        host_obj.action = last_action(host_obj.name)
    return host_obj


# SERVICE FUNCTIONS
def threading(func, data_dict, args, workers=20):
    """
    Run specified function in few streams.
    """
    pool = Pool(processes=workers)
    result = [pool.apply_async(func, (host_obj, args))
              for host_obj in data_dict.itervalues()]
    for i in result:
        r = i.get()
        if r:
            data_dict[r.name] = r
    pool.close()
    pool.join()


def check_server(server, port):
    """
    Check availability of server on port
    """
    try:
        host_ip = socket.gethostbyname(server)
        conn = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        conn.settimeout(0.5)
        conn.connect((host_ip, port))
        conn.close()
        return True
    except:
        return False


def find_juggler_server(cgroup, port):
    """
    Find available juggler server using conductor-api
    """
    conductor_api = 'https://c.yandex-team.ru/api-cached'
    server_list = urllib2.urlopen('%s/groups2hosts/%s' % (conductor_api, cgroup)).readlines()
    for server in server_list:
        server = server.strip()
        if check_server(server, port):
            return 'http://%s:%d' % (server, port)
    call.error('No reachable juggler server found.')
    sys.exit(1)


def arg_manager(*args):
    options = argparse.ArgumentParser(
        version='%(prog)s 1.10',
        formatter_class=argparse.RawTextHelpFormatter,
        description="""
description:
 Show hosts with a given hostgroup and service.""",
        epilog="""
examples:
 Shows list of unreachable hosts in "msk_search" group:
            mon_check.py msk_search UNREACHABLE
 Or simply:
            mon_check.py -U msk_search

 Shows list of unreachable host in "msk_search" group
 with IPMI:
            mon_check.py -Ui msk_search
 Same as:
            mon_check.py -i msk_search UNREACHABLE

 Shows information about location unreachable hosts in
 "msk_search" group:
            mon_check.py -Ud msk_search

 Shows information about last action and location of
 unreachable hosts in "msk_search" group:
            mon_check.py -Uda msk_search

 Shows list of unreachable hosts in "msk_search" group
 with message from juggler:
            mon_check.py -Um msk_search""")

    options.add_argument('hostgroup', type=str, help='Specifies hostgtoup.')
    options.add_argument('service', type=str, help='Specifies service.',
                         nargs="?")
    options.add_argument('-U', dest='service_opt', action='store_const',
                         const='UNREACHABLE',
                         help='Store "UNREACHABLE" as service.')
    options.add_argument('-M', dest='service_opt', action='store_const',
                         const='META', help='Store "META" as service.')
    options.add_argument('-S', dest='service_opt', action='store_const',
                         const='ssh', help='Store "ssh" as service.')
    options.add_argument('-d', '--dc', dest='show_dc', action='store_true',
                         help='Shows server placement.')
    options.add_argument('-a', '--action', dest='show_action',
                         action='store_true',
                         help='Shows server last action using bot-api.')
    options.add_argument('-m', '--message', dest='show_message',
                         action='store_true',
                         help='Shows message about state of hosts.')
    options.add_argument('-i', '--ipmi', dest='show_ipmi',
                         action='store_true',
                         help='Shows hosts with IPMI only.')
    options.add_argument('-k', '--kvm', dest='show_kvm',
                         action='store_true',
                         help='Shows hosts with KVM only.')
    options.add_argument('-w', '--warn', dest='show_warn',
                         action='store_true',
                         help='Shows hosts with status "WARNING".')
    options.add_argument('-c', '--crit', dest='show_crit',
                         action='store_true',
                         help='Shows hosts with status "CRITICAL".')

    if args:
        opts = options.parse_args(args)
    else:
        opts = options.parse_args()

    if not opts.service and opts.service_opt:
        opts.service = opts.service_opt

    return opts


def main(*args):
    # VARIABLES
    global juggler_server, bot_server, user, call, request_header
    call = MessageHandler
    user = os.environ['LOGNAME']
    juggler_server = find_juggler_server('search_instrum-juggler_master', 8998)
    request_header = {'User-Agent': 'make_req.py'}
    bot_server = 'http://bot.yandex-team.ru'

    # OPTIONS
    opts = arg_manager(*args)

    # BODY
    mon_data = {}
    if not opts.service:
        services = juggler_list_checks(opts)
        if __name__ == '__main__':
            call.msg('List of available services for %s:'
                     % opts.hostgroup)
            print(' '.join(services))
        else:
            return services
        sys.exit(0)

    juggler_data = juggler_request(opts)

    if opts.show_ipmi or opts.show_kvm:
        threading(ipmi_check, juggler_data, opts, 30)

    for host in juggler_data:
        if not (opts.show_crit and not juggler_data[host].status == 'CRIT'
                or opts.show_warn and not juggler_data[host].status == 'WARN'
                or opts.show_ipmi and not juggler_data[host].ipmi
                or opts.show_kvm and juggler_data[host].ipmi):
            mon_data[host] = juggler_data[host]

    threading(get_host_info, mon_data, opts, 40)

    # FINAL PRINT
    if __name__ == '__main__':
        for host in sorted(mon_data):
            print(mon_data[host])
    else:
        return mon_data

if __name__ == '__main__':
    main()
