#!/usr/bin/env sh

#This script is designed to work with juggler client for monitoring hosts.
#Author: Aleksandr Kurnosov <saku@yandex-team.ru>

##LOCAL FUNCTIONS
###################

juggler_request()
{
#send request to juggler server with the specified arguments
	_host_arg_="${1}"
	_service_arg_="${2}"
	IFS="
	"
	for i in $(juggler_filter exec ssh ${_juggler_server} jctl list events \
		    -host ${_host_arg_} -service ${_service_arg_} \
		    -verbose -expand -all); do
		_host_="$(echo ${i} | awk '{print $1}')"

		if [ -z "${_host_}" ];then
			continue
		fi

		if [ -n "${_ipmi_flag}" ]; then
			_ipmi_kvm_="$(ipmi_filter show_dc ${_host_} short_line,short_dc)"
			if [ "${_ipmi_kvm_}" = "" ]; then
					continue
			fi
		elif [ -n "${_no_ipmi_flag}" ]; then
			_ipmi_kvm_="$(ipmi_filter show_dc ${_host_} short_line,short_dc)"
			if [ "${_ipmi_kvm_}" != "" ]; then
				continue
			fi
		fi

		if [ -n "${_message_flag}" ]; then
			_message_=$(echo ${i} | awk '{ sub($1." ",""); print $0 }'| \
				    sed 's/%/%%/g')
		fi

		if [ -n "${_show_dc}" ]; then
			_dc_=$(show_dc ${_host_} dc)
		fi

		if [ -n "${_last_action_flag}" ]; then
			_last_action_="$(show_last_action ${_host_})"
			print_output
		elif [ -n "${_message_warn_only}" ] && $(echo "${_message_}" |\
		    grep -q WARN); then
			print_output
		elif [ -n "${_message_crit_only}" ] && $(echo "${_message_}" |\
		    grep -q CRIT); then
			print_output
		elif [ -z "${_message_warn_only}" ] && [ -z "${_message_crit_only}" ];
		then
			print_output
		fi
	done
}

print_output()
{
	if [ -z "${_plain_text}" ]; then
		f_b='\033[1m'
		f_n='\033[0m'
		f_ul='\033[4m'
		f_f='\033[7m'
	fi
	if [ -n "${_host_}" ] && [ -n "${_dc_}" ] && [ -n "${_message_}" ]; then
		printf "${f_b}${_host_}${f_n}\t${f_ul}${_dc_}${f_n}\t${_message_}\n"
	elif [ -n "${_host_}" ] && [ -n "${_dc_}" ] && \
		 [ -n "${_last_action_flag}" ]; then
		printf "${f_b}${_host_}${f_n}\t${f_ul}${_dc_}${f_n}\t${_last_action_}\n"
	elif [ -n "${_host_}" ] && [ -n "${_dc_}" ]; then
		printf "${f_b}${_host_}${f_n}\t${f_ul}${_dc_}${f_n}\n"
	elif [ -n "${_host_}" ] && [ -n "${_message_}" ]; then
		printf "${f_b}${_host_}${f_n}\t${_message_}\n"
	elif [ -n "${_host_}" ] && [ -n "${_last_action_flag}" ]; then
		printf "${f_b}${_host_}${f_n}\t${_last_action_}\n"
	elif [ -n "${_host_}" ]; then
		printf "${_host_}\n"
	else
		echo "Nothing to display"
	fi
}

juggler_filter()
{
#filters out the query and displays only hosts. First 8 strings contains
#common information
	if [ -n "${_message_flag}" ]; then
		$* | awk '/;/{if (NR > 8)\
		    {split($5,a," : "); split($0,b," : "); print a[1],b[2]}}'
	else
		$* | awk '/;/{if (NR > 8)\
		    {split($5,a,":"); print a[1]}}'
	fi
}

ipmi_filter()
{
#filers ipmi hosts using issue golem-api
	$* | egrep -v '(eto[1-3]|iva[1-4]|myt[1-6]|ugr[1-4])' |\
	     perl -pe 's/\033\[\d*(;\d*)*m//g; s/[ ,	].*$//g'
}
#look at these parameters and option arguments

arg_check()
{
	_variable_="${1}"
	_opt_="${2}"
	_arg_="${3}"
	if [ -z "${_opt_}" ] && [ -z "${_arg_}" ]; then
		echo "Required parameter '${_variable_}' is not set"
		exit 1
	elif [ -z "${_opt_}" ] && [ -n "${_arg_}" ]; then
		export echo ${_variable_}="${_arg_}"
	else
		export echo ${_variable_}="${_opt_}"
	fi
}

show_last_action()
{
	local _hostname_="${1}"
	curl -s -k "https://bot.yandex-team.ru/adm/requestshow.php?Limit=1&\
Initiator=&OsName=${_hostname_}&token=${initiator}&export=file"| \
awk 'NR>=2 {gsub(/".$/, "\"; ");gsub(/; /,";\n"); gsub(/\"/,""); printf $0}'|\
awk -F";" '{printf "%-10s %16s %18s %7s %12s %-15s",$2,$5,$22,$21,$4,$6}'
}

show_dc()
{
#obtain information about the location of host
	_hostname_="${1}"
	_column_arg_="${2}"
	curl -k -f -Ss --connect-timeout ${_timeout} "http://${_golem_server}\
/api/host_query.sbml"\
	    -d "hostname_pat=${_hostname_}&columns=${_column_arg_}"

}

check_golem_avalibility()
{
	curl -s -k --connect-timeout ${_timeout} -o /dev/null \
		"http://${_golem_server}" && golem_available="True"
	if [ -z "${golem_available}" ]; then
		echo "${_golem_server} is not available now
you can use script without options -d and -i"
		exit 1
	fi
}

help()
{
	less <<EndOfHelp
NAME
	$script_name  - show hosts with a given hostgroup with the specified
	service

SYNTAX
	${script_name} [-U][-S][-M][-d][-i][-k][-m][-a][-w][-c][-p] <hostgroup>
	<service>
	${script_name} -h

DESCRIPTION
	Show hosts with a given hostgroup and service. Script uses juggler-client
	utility an remote server over ssh. For using this script need account on
	monitoring server. For greater efficiency recommended use rsa-key and
	ssh-agent. Also script can display information about location of hosts
	and can show only hosts with IPMI.

OPTIONS
	-h	Set hostgroup as argument
	-s	Set service as argument. If the service is defined by other
		options, appointed last-mentioned parameter.
	-d	Shows information about location of hosts.
	-m 	Shows message about state of hosts.
	-w  Shows only hosts with CRIT state. Uses only with option -m.
	-c 	Shows only hosta with WARN state. Uses only with option -m.
	-a 	Shows last action with host.
	-i	Shows hosts with ipmi only.
	-k 	Shows hosts without ipmi.
	-p	Shows output as plain text.

The following options are added for work with basic services:
	-U	Set "UNREACHABLE" as service.
	-S	Set "ssh" as service.
	-M	Set "META" as service.

EXAMPLES
	Shows list of unreachable hosts in "msk_search" group:
		${script_name} msk_search UNREACHABLE
	Or simply:
		${script_name} -U msk_search

	Shows list of unreachable host in "msk_search" group with IPMI only:
		${script_name} -Ui msk_search
	Same as:
		${script_name} -i msk_search UNREACHABLE

	Shows information about location unreachable hosts in "msk_search" group:
		${script_name} -Ud msk_search

	Shows information about last action and location of unreachable hosts in
	"msk_search" group:
		${script_name} -Uda msk_search

	Shows list of unreachable hosts in "msk_search" group with message from
	juggler:
		${script_name} -Um msk_search


Oct 19 2012

EndOfHelp
}

script_name="$(basename ${0})"

##PARSE OPTIONS
#################

while getopts "hUSMadtmipkwc" opt; do
	case $opt in
	h)
		help
		exit 0
		;;
	U)
		_service="UNREACHABLE"
		;;
	S)
		_service="ssh"
		;;
	M)
		_service="META"
		;;
	d)
		_show_dc="True"
		;;
	i)
		_ipmi_flag="True"
		;;
	k)
		_no_ipmi_flag="True"
		;;
	m)
		_message_flag="True"
		;;
	w)	if [ "${_message_flag}" = "True" ]; then
			_message_warn_only="True"
		fi
		;;
	c)	if [ "${_message_flag}" = "True" ]; then
			_message_crit_only="True"
		fi
		;;
	a)
		_last_action_flag="True"
		;;
	p)
		_plain_text="True"
		;;

	\?)
		exit 1
		;;
	esac
done
shift $(($OPTIND -1))

##LOCAL VARIABLES
###################

_juggler_server=jnob00.yandex.ru
_golem_server=ro.admin.yandex-team.ru
host=""
service=""
_default_hostarg="${1}"
_default_servicearg="${2}"
_timeout=2
initiator="$(whoami)"
##BODY
########

if [ "$#" -lt "1" ] && [ -z "${_host}" ] && [ -z "${_service}" ]; then
	cat <<EndOfUsage
Requires at least 1 parameter.
Usage:
${script_name} [-U][-S][-M][-d][-i][-k][-m][-a][-w][-c][-p] <hostgroup> <service>

Use -h for additional help.
EndOfUsage
	exit 1
fi

check_golem_avalibility
arg_check host "${_host}" "${_default_hostarg}"
arg_check service "${_service}" "${_default_servicearg}"

if [ -n "${golem_available}" ]; then
		juggler_request "${host}" "${service}"
fi

exit 0
