#!/usr/bin/env python
#
# $Id: agave2.py 968122 2013-02-18 11:28:10Z zont $
#

import collections
import ctypes as C
import logging
import logging.handlers
import signal
import socket
import threading
import time
import traceback
import Queue
import subprocess

#include <net/if.h>
IF_NAMESIZE = 16
IFNAMSIZ = IF_NAMESIZE
LINK_STATE_UP = 2
#include <net/if_mib.h>
IFDATA_GENERAL = 1
IFMIB_IFDATA = 2
NETLINK_GENERIC = 0
#include <net/if_types.h>
IFT_ETHER = 0x6
#include <sys/devicestat.h>
DEVSTAT_READ = 0x01
DEVSTAT_WRITE = 0x02
DEVSTAT_N_TRANS_FLAGS = 4
DEVSTAT_NAME_LEN = 16
#include <sys/mount.h>
MFSNAMELEN = 16
MNAMELEN = 88
MNT_NOWAIT = 2
#include <sys/proc.h>
SIDL = 1
SRUN = 2
SSLEEP = 3
SSTOP = 4
SZOMB = 5
SWAIT = 6
SLOCK = 7
#include <sys/resource.h>
CP_USER = 0
CP_NICE = 1
CP_SYS = 2
CP_INTR = 3
CP_IDLE = 4
CPUSTATES = 5
#include <sys/sysctl.h>
CTL_NET = 4
KERN_PROC_PROC = 8
#include <sys/socket.h>
AF_LINK = 18
PF_LINK = AF_LINK
#include <fcntl.h>
O_RDONLY = 0x0000
#include <paths.h>
_PATH_DEVNULL = "/dev/null"

ABS = 0
REL = 1

hostname = socket.gethostname()
master = ''
step = '60'

LOGFILENAME = '/var/log/agave/checks/system.log'
LOGFORMAT = "%(levelname)s:%(asctime)s: %(message)s"
handler = logging.handlers.RotatingFileHandler(LOGFILENAME, maxBytes=1024*1024, backupCount=3)
handler.setFormatter(logging.Formatter(LOGFORMAT))
log = logging.getLogger('agave2')
log.addHandler(handler)
log.setLevel(logging.WARN)

libc = C.CDLL("libc.so")
libkvm = C.CDLL("libkvm.so")
libdevstat = C.CDLL("libdevstat.so")

libkvm.kvm_openfiles.restype = C.POINTER(C.c_void_p)
kd = libkvm.kvm_openfiles(None, _PATH_DEVNULL, None, O_RDONLY, None)
if kd == None:
    raise ValueError("kvm_openfiles()")

class bintime(C.Structure):
    _fields_ = [("sec", C.c_int64), # time_t
                ("frac", C.c_uint64)]

class devstat(C.Structure):
    pass

devstat._fields_ = [("sequence0", C.c_uint),
                ("allocated", C.c_int),
                ("start_count", C.c_uint),
                ("end_count", C.c_uint),
                ("busy_from", bintime),
                ("dev_links", C.POINTER(devstat)), # XXX STAILQ_ENTRY
                ("device_number", C.c_uint32),
                ("device_name", C.c_char * DEVSTAT_NAME_LEN),
                ("unit_number", C.c_int),
                ("bytes", C.c_uint64 * DEVSTAT_N_TRANS_FLAGS),
                ("operations", C.c_uint64 * DEVSTAT_N_TRANS_FLAGS),
                ("duration", bintime * DEVSTAT_N_TRANS_FLAGS),
                ("busy_time", bintime),
                ("creation_time", bintime),
                ("block_size", C.c_uint32),
                ("tag_types", C.c_uint64 * 3),
                ("flags", C.c_int),
                ("device_type", C.c_int),
                ("priority", C.c_int),
                ("id", C.POINTER(C.c_void_p)),
                ("sequence1", C.c_uint)]

class devinfo(C.Structure):
    _fields_ = [("devices", C.POINTER(devstat)),
                ("mem_ptr", C.POINTER(C.c_int8)),
                ("generation", C.c_long),
                ("numdevs", C.c_int)]

class statinfo(C.Structure):
    _fields_ = [("cp_time", C.c_long * CPUSTATES),
                ("tk_nin", C.c_long),
                ("tk_nout", C.c_long),
                ("dinfo", C.POINTER(devinfo)),
                ("snap_time", C.c_double)] # XXX long double

class kvm_swap(C.Structure):
    _fields_ = [("ksw_devname", C.c_char * 32),
                ("ksw_used", C.c_int),
                ("ksw_total", C.c_int),
                ("ksw_flags", C.c_int),
                ("ksw_reserved1", C.c_int),
                ("ksw_reserved2", C.c_int)]

class timeval(C.Structure):
    _fields_ = [("tv_sec", C.c_int64), # time_t
                ("tv_usec", C.c_long)] # suseconds_t

class if_data(C.Structure):
    _fields_ = [("ifi_type", C.c_ubyte),
                ("ifi_physical", C.c_ubyte),
                ("ifi_addrlen", C.c_ubyte),
                ("ifi_hdrlen", C.c_ubyte),
                ("ifi_link_state", C.c_ubyte),
                ("ifi_spare_char1", C.c_ubyte),
                ("ifi_spare_char2", C.c_ubyte),
                ("ifi_datalen", C.c_ubyte),
                ("ifi_mtu", C.c_ulong),
                ("ifi_metric", C.c_ulong),
                ("ifi_baudrate", C.c_ulong),
                ("ifi_ipackets", C.c_ulong),
                ("ifi_ierrors", C.c_ulong),
                ("ifi_opackets", C.c_ulong),
                ("ifi_oerrors", C.c_ulong),
                ("ifi_collisions", C.c_ulong),
                ("ifi_ibytes", C.c_ulong),
                ("ifi_obytes", C.c_ulong),
                ("ifi_imcasts", C.c_ulong),
                ("ifi_omcasts", C.c_ulong),
                ("ifi_iqdrops", C.c_ulong),
                ("ifi_noproto", C.c_ulong),
                ("ifi_hwassist", C.c_ulong),
                ("ifi_epoch", C.c_int64), # time_t
                ("ifi_lastchange", timeval)]

class ifmibdata(C.Structure):
    _fields_ = [("ifmd_name", C.c_char * IFNAMSIZ),
                ("ifmd_pcount", C.c_int),
                ("ifmd_flags", C.c_int),
                ("ifmd_snd_len", C.c_int),
                ("ifmd_snd_maxlen", C.c_int),
                ("ifmd_snd_drops", C.c_int),
                ("ifmd_filler", C.c_int * 4),
                ("ifmd_data", if_data)]

class kinfo_proc(C.Structure):
    _fields_ = [("spare1", C.c_byte * 388),
                ("ki_stat", C.c_byte),
                ("spare2", C.c_byte * 699)]

class statfs(C.Structure):
    _fields_ = [("f_version", C.c_uint32),
                ("f_type", C.c_uint32),
                ("f_flags", C.c_uint64),
                ("f_bsize", C.c_uint64),
                ("f_iosize", C.c_uint64),
                ("f_blocks", C.c_uint64),
                ("f_bfree", C.c_uint64),
                ("f_bavail", C.c_int64),
                ("f_files", C.c_uint64),
                ("f_ffree", C.c_int64),
                ("f_syncwrites", C.c_uint64),
                ("f_asyncwrites", C.c_uint64),
                ("f_syncreads", C.c_uint64),
                ("f_asyncreads", C.c_uint64),
                ("f_spare", C.c_uint64 * 10),
                ("f_namemax", C.c_uint32),
                ("f_owner", C.c_uint32), # uid_t
                ("f_fsid", C.c_int32 * 2), # fsid_t
                ("f_charspare", C.c_char * 80),
                ("f_fstypename", C.c_char * MFSNAMELEN),
                ("f_mntfromname", C.c_char * MNAMELEN),
                ("f_mntonname", C.c_char * MNAMELEN)]

class tcpstat(C.Structure):
    _fields_ = [("tcps_connattempt", C.c_ulong),
                ("tcps_accepts", C.c_ulong),
                ("tcps_connects", C.c_ulong),
                ("tcps_drops", C.c_ulong),
                ("tcps_conndrops", C.c_ulong),
                ("tcps_minmssdrops", C.c_ulong),
                ("tcps_closed", C.c_ulong),
                ("tcps_segstimed", C.c_ulong),
                ("tcps_rttupdated", C.c_ulong),
                ("tcps_delack", C.c_ulong),
                ("tcps_timeoutdrop", C.c_ulong),
                ("tcps_rexmttimeo", C.c_ulong),
                ("tcps_persisttimeo", C.c_ulong),
                ("tcps_keeptimeo", C.c_ulong),
                ("tcps_keepprobe", C.c_ulong),
                ("tcps_keepdrops", C.c_ulong),
                ("tcps_sndtotal", C.c_ulong),
                ("tcps_sndpack", C.c_ulong),
                ("tcps_sndbyte", C.c_ulong),
                ("tcps_sndrexmitpack", C.c_ulong),
                ("tcps_sndrexmitbyte", C.c_ulong),
                ("tcps_sndrexmitbad", C.c_ulong),
                ("tcps_sndacks", C.c_ulong),
                ("tcps_sndprobe", C.c_ulong),
                ("tcps_sndurg", C.c_ulong),
                ("tcps_sndwinup", C.c_ulong),
                ("tcps_sndctrl", C.c_ulong),
                ("tcps_rcvtotal", C.c_ulong),
                ("tcps_rcvpack", C.c_ulong),
                ("tcps_rcvbyte", C.c_ulong),
                ("tcps_rcvbadsum", C.c_ulong),
                ("tcps_rcvbadoff", C.c_ulong),
                ("tcps_rcvmemdrop", C.c_ulong),
                ("tcps_rcvshort", C.c_ulong),
                ("tcps_rcvduppack", C.c_ulong),
                ("tcps_rcvdupbyte", C.c_ulong),
                ("tcps_rcvpartduppack", C.c_ulong),
                ("tcps_rcvpartdupbyte", C.c_ulong),
                ("tcps_rcvoopack", C.c_ulong),
                ("tcps_rcvoobyte", C.c_ulong),
                ("tcps_rcvpackafterwin", C.c_ulong),
                ("tcps_rcvbyteafterwin", C.c_ulong),
                ("tcps_rcvafterclose", C.c_ulong),
                ("tcps_rcvwinprobe", C.c_ulong),
                ("tcps_rcvdupack", C.c_ulong),
                ("tcps_rcvacktoomuch", C.c_ulong),
                ("tcps_rcvackpack", C.c_ulong),
                ("tcps_rcvackbyte", C.c_ulong),
                ("tcps_rcvwinupd", C.c_ulong),
                ("tcps_pawsdrop", C.c_ulong),
                ("tcps_predack", C.c_ulong),
                ("tcps_preddat", C.c_ulong),
                ("tcps_pcbcachemiss", C.c_ulong),
                ("tcps_cachedrtt", C.c_ulong),
                ("tcps_cachedrttvar", C.c_ulong),
                ("tcps_cachedssthresh", C.c_ulong),
                ("tcps_usedrtt", C.c_ulong),
                ("tcps_usedrttvar", C.c_ulong),
                ("tcps_usedssthresh", C.c_ulong),
                ("tcps_persistdrop", C.c_ulong),
                ("tcps_badsyn", C.c_ulong),
                ("tcps_mturesent", C.c_ulong),
                ("tcps_listendrop", C.c_ulong),
                ("tcps_badrst", C.c_ulong),
                ("tcps_sc_added", C.c_ulong),
                ("tcps_sc_retransmitted", C.c_ulong),
                ("tcps_sc_dupsyn", C.c_ulong),
                ("tcps_sc_dropped", C.c_ulong),
                ("tcps_sc_completed", C.c_ulong),
                ("tcps_sc_bucketoverflow", C.c_ulong),
                ("tcps_sc_cacheoverflow", C.c_ulong),
                ("tcps_sc_reset", C.c_ulong),
                ("tcps_sc_stale", C.c_ulong),
                ("tcps_sc_aborted", C.c_ulong),
                ("tcps_sc_badack", C.c_ulong),
                ("tcps_sc_unreach", C.c_ulong),
                ("tcps_sc_zonefail", C.c_ulong),
                ("tcps_sc_sendcookie", C.c_ulong),
                ("tcps_sc_recvcookie", C.c_ulong),
                ("tcps_hc_added", C.c_ulong),
                ("tcps_hc_bucketoverflow", C.c_ulong),
                ("tcps_finwait2_drops", C.c_ulong),
                ("tcps_sack_recovery_episode", C.c_ulong),
                ("tcps_sack_rexmits", C.c_ulong),
                ("tcps_sack_rexmit_bytes", C.c_ulong),
                ("tcps_sack_rcv_blocks", C.c_ulong),
                ("tcps_sack_send_blocks", C.c_ulong),
                ("tcps_sack_sboverflow", C.c_ulong),
                ("tcps_ecn_ce", C.c_ulong),
                ("tcps_ecn_ect0", C.c_ulong),
                ("tcps_ecn_ect1", C.c_ulong),
                ("tcps_ecn_shs", C.c_ulong),
                ("tcps_ecn_rcwnd", C.c_ulong),
                ("tcps_sig_rcvgoodsig", C.c_ulong),
                ("tcps_sig_rcvbadsig", C.c_ulong),
                ("tcps_sig_err_buildsig", C.c_ulong),
                ("tcps_sig_err_sigopt", C.c_ulong),
                ("tcps_sig_err_nosigopt", C.c_ulong),
                ("_pad", C.c_ulong * 12)]


class Counters(object):
    def __init__(self):
        self.values = {}
        self.cp_time = {}
        self.cp_time['total'] = 0

    def add_task(self, type, module, task):
        if module not in self.values:
            self.values[module] = collections.deque()
        if type == ABS or type == REL:
            task['_type_'] = type
        else:
            raise ValueError("Unknown type: %s" % type)
        task['_timestamp_'] = time.time()
        self.values[module].append(task)

    def get_task(self, module):
        try:
            length = len(self.values[module])
        except:
            return None

        if length == 0:
            return None

        task = self.values[module][0]
        type = task['_type_']
        if type == ABS:
            task1 = self.values[module].popleft()
            task = {}
            task['timestamp'] = int(task1['_timestamp_'])
            task['module'] = module
            task['values'] = {}
            for key in task1:
                if key == '_type_' or key == '_timestamp_':
                    continue
                task['values'][key] = task1[key]
            return task
        elif type == REL:
            if length > 1:
                task1 = self.values[module].popleft()
                task2 = self.values[module][0]
                tsdiff = task2['_timestamp_'] - task1['_timestamp_']
                if tsdiff <= 0:
                    raise ValueError("Negative timestamps difference")
                task = {}
                task['timestamp'] = int(task2['_timestamp_'])
                task['module'] = module
                task['values'] = {}
                for key in task1:
                    if key == '_type_' or key == '_timestamp_':
                        continue
                    task['values'][key] = (task2[key] - task1[key]) / tsdiff
                return task
            else:
                return None
        else:
            raise ValueError("Unknown type: %s" % type)

    def modules(self):
        for module in self.values:
            yield module

    def tasks(self):
        for module in self.modules():
            task = self.get_task(module)
            if task != None:
                yield task

    def cpu(self):
        task = {}
        names = {}
        names[CP_USER] = "user"
        names[CP_NICE] = "nice"
        names[CP_SYS]  = "sys"
        names[CP_INTR] = "intr"
        names[CP_IDLE] = "idle"

        cp_time = (C.c_long * CPUSTATES)()
        size = C.c_size_t()
        size.value = C.sizeof(C.c_long * CPUSTATES)
        name = "kern.cp_time"
        rc = libc.sysctlbyname(name, C.byref(cp_time), C.byref(size), None, 0)
        if rc == -1:
            log.error("sysctlbyname(%s)", name)
            return

        time = 0.0
        for i in range(CPUSTATES):
            time += cp_time[i]
        if self.cp_time['total']:
            dtime = time - self.cp_time['total']

        for i in range(CPUSTATES):
            log.debug("%s\t%s" % (names[i], cp_time[i]))
            if self.cp_time['total']:
                task[names[i]] = "%.3f" % (100. * (cp_time[i] - self.cp_time[names[i]]) / dtime)
            self.cp_time[names[i]] = cp_time[i]
        if self.cp_time['total']:
            self.add_task(ABS, "cpu", task)
        self.cp_time['total'] = time

    def iostat(self):
        task_io = {}
        task_busy = {}

        stats = statinfo()
        dinfo = devinfo()
        C.memset(C.byref(stats), 0, C.sizeof(stats))
        C.memset(C.byref(dinfo), 0, C.sizeof(dinfo))
        stats.dinfo = C.pointer(dinfo)
        rc = libdevstat.devstat_getdevs(None, C.byref(stats))
        if rc == -1:
            log.error("devstat_getdevs()")
            return

        disks_io = {}
        disks_busy = {}
        for i in range(dinfo.numdevs): # XXX extra loops here
            dev = dinfo.devices[i]
            if dev.device_name == "pass": # XXX
                continue
            disks_io[dev.device_name + "_read"] = 0
            disks_io[dev.device_name + "_write"] = 0
            disks_busy[dev.device_name + "_busy"] = 0
            task_io[dev.device_name + "_read"] = 0
            task_io[dev.device_name + "_write"] = 0
            task_busy[dev.device_name + "_busy"] = 0
        for i in range(dinfo.numdevs):
            dev = dinfo.devices[i]
            if dev.device_name == "pass": # XXX
                continue
            disks_io[dev.device_name + "_read"] += 1
            disks_io[dev.device_name + "_write"] += 1
            disks_busy[dev.device_name + "_busy"] += 1
            task_io[dev.device_name + "_read"] += dev.bytes[DEVSTAT_READ]
            task_io[dev.device_name + "_write"] += dev.bytes[DEVSTAT_WRITE]
            task_busy[dev.device_name + "_busy"] += dev.busy_time.sec * 100
            log.debug("%s%d:\tread: %lu\twrite: %lu\tbusy: %lu" % (dev.device_name, \
                                                                       dev.unit_number, \
                                                                       dev.bytes[DEVSTAT_READ], \
                                                                       dev.bytes[DEVSTAT_WRITE], \
                                                                       dev.busy_time.sec))
        for key in disks_io:
            task_io[key] = task_io[key] / disks_io[key]
        for key in disks_busy:
            task_busy[key] = task_busy[key] / disks_busy[key]
        self.add_task(REL, "disk_io", task_io)
        self.add_task(REL, "disk_busy", task_busy)

    def loadavg(self):
        task = {}
        names = {}
        names[0] = "1_min"
        names[1] = "5_min"
        names[2] = "15_min"

        loadavg = (C.c_double * len(names))()
        rc = libc.getloadavg(C.byref(loadavg), len(names))
        if rc == -1:
            log.error("getloadavg()")
            return

        for i in names:
            log.debug("%s\t%s" % (names[i], loadavg[i]))
            task[names[i]] = "%.3f" % loadavg[i]
        self.add_task(ABS, "loadavg", task)

    def memory(self):
        task = {}
        names = {}
        names[0] = "cache"
        names[1] = "inactive"
        names[2] = "active"
        names[3] = "wire"
        names[4] = "free"

        pagesize = libc.getpagesize()
        if pagesize == -1:
            log.error("getpagesize()")
            return

        for i in names:
            value = C.c_uint()
            size = C.c_size_t()
            size.value = C.sizeof(C.c_uint)
            name = "vm.stats.vm.v_" + names[i] + "_count"
            rc = libc.sysctlbyname(name, C.byref(value), C.byref(size), None, 0)
            if rc == -1:
                log.error("sysctlbyname(%s)", name)
            log.debug("%s\t%s" % (names[i], value.value * pagesize))
            task[names[i]] = value.value * pagesize

        value = kvm_swap()
        size = C.c_size_t()
        size.value = C.sizeof(kvm_swap)
        rc = libkvm.kvm_getswapinfo(kd, C.byref(value), 1, 0)
        if rc == -1:
            log.error("kvm_getswapinfo()")
        task["swap_used"] = value.ksw_used * pagesize
        task["swap_free"] = (value.ksw_total - value.ksw_used) * pagesize

        self.add_task(ABS, "memory", task)

    def netstat(self):
        task_pps = {}
        task_bytes = {}
        task_errors = {}

        value = C.c_uint()
        size = C.c_size_t()
        size.value = C.sizeof(C.c_uint)
        name = "net.link.generic.system.ifcount"
        rc = libc.sysctlbyname(name, C.byref(value), C.byref(size), None, 0)
        if rc == -1:
            log.error("sysctlbyname(%s)", name)
            return

        for i in range(1, value.value + 1):
            name = (C.c_int * 6)()
            name[0] = CTL_NET
            name[1] = PF_LINK
            name[2] = NETLINK_GENERIC
            name[3] = IFMIB_IFDATA
            name[4] = i
            name[5] = IFDATA_GENERAL
            value = ifmibdata()
            size = C.c_size_t()
            size.value = C.sizeof(ifmibdata)
            rc = libc.sysctl(name, 6, C.byref(value), C.byref(size), None, 0)
            if rc == -1:
                log.error("sysctl(net.link.generic)")
                continue
            ifdata = value.ifmd_data
            if ifdata.ifi_type != IFT_ETHER or ifdata.ifi_link_state != LINK_STATE_UP:
                continue
            task_pps['in_packets'] = ifdata.ifi_ipackets
            task_pps['out_packets'] = ifdata.ifi_opackets
            task_bytes['in_bytes'] = ifdata.ifi_ibytes
            task_bytes['out_bytes'] = ifdata.ifi_obytes
            task_errors['in_errors'] = ifdata.ifi_ierrors
            task_errors['out_errors'] = ifdata.ifi_oerrors
            task_errors['in_drops'] = ifdata.ifi_iqdrops
            break # XXX
        self.add_task(REL, "netstat_pps", task_pps)
        self.add_task(REL, "netstat_bytes", task_bytes)
        self.add_task(REL, "netstat_errors", task_errors)

    def openfiles(self):
        task = {}

        value = C.c_int()
        size = C.c_size_t()
        size.value = C.sizeof(C.c_int)
        name = "kern.openfiles"
        rc = libc.sysctlbyname(name, C.byref(value), C.byref(size), None, 0)
        if rc == -1:
            log.error("sysctlbyname(%s)", name)
        task["openfiles"] = value.value

        value = C.c_int()
        size = C.c_size_t()
        size.value = C.sizeof(C.c_int)
        name = "kern.ipc.numopensockets"
        rc = libc.sysctlbyname(name, C.byref(value), C.byref(size), None, 0)
        if rc == -1:
            log.error("sysctlbyname(%s)", name)
        task["opensockets"] = value.value

        self.add_task(ABS, "openfiles", task)

    def processes(self):
        task = {}
        names = {}
        names[SIDL] = "starting"
        names[SRUN] = "running"
        names[SSLEEP] = "sleeping"
        names[SSTOP] = "stopped"
        names[SZOMB] = "zombie"
        names[SWAIT] = "waiting"
        names[SLOCK] = "lock"

        value = C.c_uint()
        size = C.c_size_t()
        size.value = C.sizeof(C.c_uint)
        name = "debug.sizeof.kinfo_proc"
        rc = libc.sysctlbyname(name, C.byref(value), C.byref(size), None, 0)
        if rc == -1:
            log.error("sysctlbyname(%s)", name)
            return

        if value.value != C.sizeof(kinfo_proc):
            log.error("Size of struct kinfo_proc missmatched")
            return

        cnt = C.c_int()
        libkvm.kvm_getprocs.restype = C.POINTER(kinfo_proc)
        kinfop = libkvm.kvm_getprocs(kd, KERN_PROC_PROC, 0, C.byref(cnt))
        if kinfop == None:
            log.error("kvm_getprocs()")
            return

        kinfo = kinfo_proc()
        kinfo.values = C.cast(kinfop, C.POINTER(kinfo_proc))
        proc = {}
        for state in range(1, 8): # XXX
            proc[state] = 0
        for i in range(cnt.value):
            state = kinfo.values[i].ki_stat
            proc[state] += 1
        for state in proc:
            task[names[state]] = proc[state]

        self.add_task(ABS, "processes", task)

    def statfs(self):
        task = {}

        supported_fs = ["ufs", "zfs"]
        mntbufp = C.c_void_p()
        rc = libc.getmntinfo(C.byref(mntbufp), MNT_NOWAIT)
        if rc == 0:
            log.error("getmntinfo()")
            return

        mntbuf = statfs()
        mntbuf.values = C.cast(mntbufp, C.POINTER(statfs))
        for i in range(rc):
            fs = mntbuf.values[i]
            if not fs.f_fstypename in supported_fs:
                continue
            size = float(fs.f_blocks * fs.f_bsize)
            # Skip FS with size less than 16Mb
            if size < (16 * 1024 * 1024):
                continue
            if fs.f_mntonname == '/':
                fsname = 'root'
            else:
                fsname = fs.f_mntonname.lstrip('/').replace('/', '_')
            used = (fs.f_blocks - fs.f_bavail) * fs.f_bsize
            task[fsname] = "%.3f" % (used/size * 100)
            log.debug("%s\t%s" % (fsname, task[fsname]))

        self.add_task(ABS, "statfs", task)

    def tcpstat(self):
        task = {}

        value = tcpstat()
        size = C.c_size_t()
        size.value = C.sizeof(tcpstat)
        name = "net.inet.tcp.stats"
        rc = libc.sysctlbyname(name, C.byref(value), C.byref(size), None, 0)
        if rc == -1:
            log.error("sysctlbyname(%s)", name)
            return

        task['connections_initiated'] = value.tcps_connattempt
        task['connections_accepted'] = value.tcps_accepts
        task['connections_dropped'] = value.tcps_drops + value.tcps_conndrops
        task['listen_queue_overflows'] = value.tcps_listendrop
        task['packets_retransmitted'] = value.tcps_sndrexmitpack
        task['synack_retransmitted'] = value.tcps_sc_retransmitted

        self.add_task(REL, "tcpstat", task)

    def update(self):
        self.cpu()
        self.iostat()
        self.loadavg()
        self.memory()
        self.netstat()
        self.openfiles()
        self.processes()
        self.statfs()
        self.tcpstat()

def producer(q):
    c = Counters()
    time0 = time.time()
    n = 0
    while True:
        tasks = []
        c.update()
        for task in c.tasks():
            agavesend_cmd = [
                'agavesend',
                '--fieldset',
                task['module'],
                '--step',
                step,
                '--values',
            ]
            for key in task['values']:
                agavesend_cmd.append("%s:%s" % (key, task['values'][key]))

            subprocess.call(agavesend_cmd)

            tasks.append(task)
        q.put(tasks)
        time1 = time.time()
        n += 1
        sleep = int(step) * n - (time1 - time0)
        if sleep > 0:
            time.sleep(sleep)

def wrapper(func, args):
    while True:
        try:
            func(args)
        except:
            log.critical(traceback.format_exc())
            time.sleep(float(step))

def main():
    global master, step

    q = Queue.Queue()
    p = threading.Thread(target=wrapper, kwargs={'func': producer, 'args': q})
    p.daemon = True
    p.start()
    try:
        signal.pause()
    except:
        exit(0)

if __name__ == '__main__':
    main()
