#!/bin/sh
#
# Script for data collection for MySQL graphs.
#
# $Id$

me=${0##*/}	# strip path
me=${me%.*}	# strip extension
BASE=$HOME/agents
PATH=/bin:/usr/bin:/usr/local/bin


#-- Subroutines --------------------------------------------------------

err()
{       
	local _exitval

	_exitval=$1
	shift

	echo 1>&2 "ERROR (${me}): $*"
	exit $_exitval
}

warn()
{       
	echo 1>&2 "WARNING (${me}): $*"
}

check_depends()
{
	if ! which -s $mysql_cmd; then
		err 1 "${mysql_cmd} command not found !"
	fi
}


#-- Variables ----------------------------------------------------------

config="${BASE}/etc/${me}.conf"

mysql_cmd="mysql"
mysql_def_file="${HOME}/.my.cnf"

instances="master"

sql_get_global_status="SHOW GLOBAL STATUS"
sql_get_engines="SHOW ENGINES"

#-- Main ---------------------------------------------------------------

if [ -r "${config}" ]; then
	. ${config}
fi

check_depends

for instance in $instances; do
	eval mysql_grp_sfx=\${mysql_grp_sfx_${instance}:-\${instance}}

	# Check support of InnoDB
	innodb_support=`$mysql_cmd --defaults-file="${mysql_def_file}" \
		--defaults-group-suffix="${mysql_grp_sfx}" -BAe \
		"${sql_get_engines}" | \
		awk '$1 == InnoDB {print $2}'`
	if ! $mysql_cmd --defaults-file="${mysql_def_file}" \
		--defaults-group-suffix="${mysql_grp_sfx}" -BAe \
		"${sql_get_global_status}" | \
		awk -v instance="${instance}" \
		    -v innodb_support="${innodb_support}" '
			function output( _type, _arr) {
				_list = ""
				_header = sprintf("%s_%s", instance, _type)
				for (_param in _arr) {
					printf "%s:%d\n", _header, _arr[_param]
					if (_list) _list = sprintf("%s/%s", 
						_list, _param)
					else _list = _param
				}
				printf "%s:%s\n", _header, _list
			}

			BEGIN { input_is_valid = 0 }

			$1 == "Uptime" { input_is_valid = 1 }

			$1 ~ /^Connections$/ {
				connections[$1] = $2
			}
			$1 ~ /^(Com_(select|delete|insert|update|replace)|Questions)$/ {
				questions[$1] = $2
			}
			$1 ~ /^(Created_tmp_(file|(disk_)?table)s)$/ {
				created_tmp[$1] = $2
			}
			$1 ~ /^Innodb_buffer_pool_reads$/ {
				ibp_reads[$1] = $2
			}
			$1 ~ /^Innodb_row_lock_time$/ {
				ir_lock_time[$1] = $2
			}
			$1 ~ /^Innodb_data_(read|write|fsync)s$/ {
				id_ops[$1] = $2
			}
			$1 ~ /^Innodb_pages_(created|read|written)$/ {
				ip_ops[$1] = $2
			}
			$1 ~ /^Innodb_rows_(deleted|inserted|read|updated)$/ {
				ir_ops[$1] = $2
			}
			$1 ~ /^Slow_queries$/ {
				slow_queries[$1] = $2
			}

			END{
				if (! input_is_valid) exit 1

				output("questions", questions)
				output("connections", connections)
				output("created_tmp", created_tmp)
				if ( innodb_support == "YES" ||
				     innodb_support == "DEFAULT" ) {
					output("innodb_bp_reads", ibp_reads)
					output("innodb_row_lock_time", ir_lock_time)
					output("innodb_data_ops", id_ops)
					output("innodb_pages_ops", ip_ops)
					output("innodb_rows_ops", ir_ops)
				}
				output("slow_queries", slow_queries)
			}
		'
	then
		warn "Can't get global status for instance \"${instance}\" !"
	fi
done

