#!/bin/sh -e
#
# $Id$
#
# Script for checking status of software RAIDs and sending tickets to
# replace hard drives.

PATH=/sbin:/bin:/usr/sbin:/usr/bin:/usr/local/sbin:/usr/local/bin
export PATH

#-- Subroutines --------------------------------------------------------

err()
{
	local _exitval _d

	_exitval=$1
	shift

	if [ -s "${err_file}" ]; then
		cat $err_file || true
	fi

	_d=$(date "+%F %T")
	echo 1>&2 "[${_d}] ERROR: $*"
	exit $_exitval
}

verbose()
{
	local _d

	if [ -n "$verbose" ]; then
		_d=$(date "+%F %T")
		echo "[${_d}] VERBOSE: $*"
	fi
}

usage()
{
	echo 1>&2 "Usage: ${thiscmd} [-f | -i] [-dv]"
	echo 1>&2 "Option is:"
	echo 1>&2 "  -d    dry-run mode"
	echo 1>&2 "  -f    force to send ticket in spite of hard drive state"
	echo 1>&2 "  -i    inserting a new hard drive after replacing"
	echo 1>&2 "  -v    verbose mode"
	exit 1
}

get_options()
{
	local _opt

	while getopts "dfiv" _opt; do
		case "${_opt}" in
			d) dry_run="echo" ;;
			f) force="yes" ;;
			i) mode="insert" ;;
			v) verbose="yes" ;;
			*) usage ;;
		esac
	done

	shift $(($OPTIND - 1))

	if [ $# -ne 0 ]; then
		usage
	fi

	if [ "${mode}" = "insert" -a -n "${force}" ]; then
		usage
	fi
}

handle_options()
{
	: ${mode:=${default_mode}}
}

get_drv_list()
{
	# This subroutine sets following global variables:
	# * drv_list
	# * num_drv

	drv_re="(ad|ada|da)[0-9]+"

	eval $(sysctl -n kern.disks | awk -v drv_re="${drv_re}" '
		BEGIN{ RS = " "; drv_list = del = ""; }
		$1 ~ drv_re {
			drvs[$1] = 1
		}
		END{
			for (drv in drvs) {
				num_drv++
				drv_list = sprintf("%s%s%s", drv_list, del, drv)
				del = " "
			}
			printf "drv_list=\"%s\";\nnum_drv=\"%d\";\n", \
				drv_list, num_drv
		}
	' || echo num_drv=0)

	if [ $num_drv -lt 1 ]; then
		err 1 "It's necessary at least 1 drives to check !"
	fi

	verbose "Detected drives are: ${drv_list} (${num_drv})."
}

add_to_cleanup()
{
	cleanup_files="${cleanup_files}${cleanup_files:+ }${_label_f}"
}

get_scheme()
{
	# This subroutine sets following global variables:
	# * scheme

	scheme=$(echo $hostname_s | \
		sed -E -e 's,^([0-9]+-){3}[0-9]+$,spider,' -e 's/[0-9-]+$//')

	if [ -z "${scheme}" ]; then
		err 1 "Can't get scheme !"
	fi

	if ! echo "${scheme_list}" | grep -qw "${scheme}"; then
		err 1 "There is no support for this host !"
	fi

	verbose "Scheme is: ${scheme}."
}

get_mirrors_status()
{
	# This subroutine sets following global variables:
	# * list_mirrors
	# * list_bad_mirrors
	# * mirror_<name>_num
	# * mirror_<name>_list

	eval $(gmirror status -s | awk -v type="mirror" '
		function join(_list, _el, _del) {
			if (_list) _list = sprintf("%s%s%s", _list, _del, _el)
			else _list = _el
			return _list
		}
		$1 ~ type {
			name = $1
			sub(type"/", "",  name)
			state[name] = $2
			if ($2 == "DEGRADED" && $NF ~ /[0-9]+%/) \
				rebuild[name] = 1
			num_provs[name]++
			prov = $3
			prov_list[name] = join(prov_list[name], prov, " ")
		}
		END{
			for (name in state) {
				printf "%s_%s_num=\"%s\";\n", type, \
					name, num_provs[name]
				printf "%s_%s_list=\"%s\";\n", type, \
					name, prov_list[name]
				list = join(list, name, " ")
				if (state[name] != "COMPLETE" && \
					! rebuild[name]) \
					list_bad = join(list_bad, name, " ")
			}
			printf "list_%ss=\"%s\";\n", type, list
			printf "list_bad_%ss=\"%s\";\n", type, list_bad
		}
	')

	if [ -z "${list_mirrors}" ]; then
		err 1 "Can't get list of mirrors !"
	fi

	verbose "Detected mirrors are: ${list_mirrors}"
	if [ -n "${list_bad_mirrors}" ]; then
		verbose "  including bad ones: ${list_bad_mirrors}."
	else
		verbose "  (without any bad ones)."
	fi

	drv_in_mirr_list=$(gmirror status -s | \
		awk -v drv_re="${drv_re}" '$3 ~ drv_re {print $3}' | \
		grep -Eo "${drv_re}" | sort -u | tr '\n' ' ')

	if [ -z "${drv_in_mirr_list}" ]; then
		err 1 "Can't get list of drives which are used in mirrors !"
	fi

	verbose "Drives which are used in mirrors: ${drv_in_mirr_list}."
}

name2class()
{
	echo "$1" | sed -Ee 's/[0-9]+/X/' || true
}

check_sata_drv()
{
	# This subroutine sets following global variables:
	# * ticket_err_str
	# * ticket_drv_model
	# * ticket_drv_sn
	# * ticket_drv_num
	# * ticket_drv_is_sata
	# * drv_sata_channel (for ad-drives)

	local _drv

	_drv="$1"

	eval $(smartctl -a /dev/${_drv} | awk '
		BEGIN{ err_str = del = "" }
		$2 ~ /(Reallocated_Sector_Ct|Reallocated_Event_Count|Current_Pending_Sector)/ && $10 > 0 {
			err_str = sprintf("%s%s%s(%d)", err_str, del, $2, $10)
			del = "; "
		}
		/^Device Model:/ {
			model = $0
			sub(/^.*: +/, "", model)
		}
		/Serial Number:/ {
			sn = $0
			sub(/^.*: +/, "", sn)
		}
		END{
			printf "ticket_err_str=\"%s\";\n", err_str
			printf "ticket_drv_model=\"%s\";\n", model
			printf "ticket_drv_sn=\"%s\";\n", sn
		}
	')

	if [ -n "${ticket_err_str}" -o -n "${force}" ]; then
		ticket_drv_num=$(echo "${_drv}" | grep -oE "[0-9]+")
		ticket_drv_is_sata="YES"
		case "${_drv}" in
			ad[0-9]*)
				eval $(atacontrol list | awk -v drv="${_drv}" '
					/^ATA channel/ {
						ch = $3
						sub(/:/, "", ch)
					}
					$0 ~ drv {
						printf "drv_sata_channel=\"ata%d\";\n", ch
						exit
					}
					')
				;;
		esac
		return 1
	fi
}

prov2drv()
{
	echo $1 | sed -Ee 's/^('"${drv_re}"').*/\1/'
}

check_drv()
{
	local _drv _full_drv_path

	_drv="$1"

	case "${_drv}" in
		ad[0-9]*|ada[0-9]*) _check_cmd="check_sata_drv $_drv" ;;
		*) err 1 "Unsupported drive type (${_drv}) to check !" ;;
	esac

	if ! $_check_cmd; then
		verbose "Some errors are detected on ${_drv}:" \
			"${ticket_err_str}."
		return 1
	fi
}

forget_mirror()
{
	local _mirror

	_mirror="$1"

	if ! $dry_run gmirror forget $geom_opts $_mirror; then
		err 1 "Can't forget mirror ${_mirror} !"
	fi

	verbose "Mirror ${_mirror} has been forgotten successfully."
}

free_drv()
{
	local _drv _prov _name _class_name _real_num _real_list _ctl_let
	local _detach_cmd

	_drv="$1"

	for _name in $list_mirrors; do
		_class_name=$(name2class $_name)
		eval _real_num=\${mirror_${_name}_num}
		eval _real_list=\${mirror_${_name}_list}
		eval _ctl_let=\${${scheme}_mirror_${_class_name}_let}
		if ! echo "$_real_list" | grep -q $_drv; then
			continue
		fi
		if [ $_real_num -lt 2 ]; then
			err 1 "There is broken mirror (${_name}) on" \
				"broken drive (${_drv}) !"
		fi

		_prov="${_drv}${_ctl_let}"
		if ! $dry_run gmirror remove $geom_opts $_name ${_prov}; then
			err 1 "Can't remove ${_prov} from mirror ${_name} !"
		fi
		verbose "Provider ${_prov} has been removed from" \
			"mirror ${_name}."
	done

	case "${_drv}" in
		ad[0-9]*) _detach_cmd="atacontrol detach ${drv_sata_channel}" ;;
		ada[0-9]*) _detach_cmd="" ;;
		*) err 1 "Unsupported drive type (${_drv}) to detach !" ;;
	esac

	if ! $dry_run $_detach_cmd; then
		err 1 "Can't detach ${_drv} (${_detach_cmd}) !"
	fi
	verbose "The drive ${_drv} has been detached."
}

get_dc()
{
	local _dc

	_dc=$(fetch -q -T 10 -o - $golem_api_url 2>/dev/null)

	echo ${_dc:="Unknown_DC"}
}

send_ticket()
{
	ticket_num=$(curl -k -f -sS "http://bot.yandex-team.ru/api/request.php?name=${hostname}\
initiator=search&operation=hdd&slot=${ticket_drv_num}&email=yes&cc=${email_cc}")
	if [ -z $ticket_num ]; then
		ticket_subj=$(echo "${ticket_subj_tpl}" | \
			sed -e "s/<DC>/$(get_dc)/")
		ticket_subj_encoded=$(python -c "print('=?${ticket_encoding}?B?%s?=' \
			% '${ticket_subj}'.encode('base64').strip())")
	
		if ! sendmail -t <<EOF_TICKET
From: ${ticket_from}
Subject: ${ticket_subj_encoded}
To: ${ticket_to}
Cc: ${ticket_cc}
Content-Type: text/plain; charset=${ticket_encoding}
Content-Transfer-Encoding: 8bit
MIME-Version: 1.0
User-Agent: ${ticket_user_agent:-$0}

Hi;

   ${ticket_drv_num} (  )   ${hostname_s} 
:
${ticket_drv_model}
S/N: ${ticket_drv_sn}

${ticket_drv_is_sata:+    SATA-2. }   .

.
${ticket_err_str:+
:
${ticket_err_str}
}
-- 
${ticket_signature}
EOF_TICKET
		then
			err 1 "Can't send ticket !"
		fi
	fi
	verbose "Ticket has been sent."
}

handle_bad_mirrors()
{
	local _name_bad _name _ctl_num _ctl_let _real_num _real_list
	local _prov_used _prov_num _list_prov_used _re_prov_used _class_ctl_num
	local _class_real_num _miss_num _miss_prov _miss_provs _handled_drvs

	_handled_drvs=0

	for _name_bad in $list_bad_mirrors; do
		_class_name_bad=$(name2class $_name_bad)

		eval _ctl_num=\${${scheme}_mirror_${_class_name_bad}_num}
		eval _ctl_let=\${${scheme}_mirror_${_class_name_bad}_let}
		eval _real_num=\${mirror_${_name_bad}_num}
		eval _real_list=\${mirror_${_name_bad}_list}

		if [ "${_real_num}" -ge "${_ctl_num}" ]; then
			# We may have different hosts within one scheme (e.g.
			# with 2 or 4 drives in root). This is heuristic rule
			# to detect such hosts and handle it automatically.
			#
			# Example:
			#   We set smth_mirror_root_num=2, but here we
			#   detected that there are 3 drives in root, so
			#   probably we should have 4 drives in this root.
			if [ $((${_real_num} % 2)) -eq 0 ]; then
				_ctl_num=$_real_num
			else
				_ctl_num=$((${_real_num} + 1))
			fi
			verbose "Use ${_ctl_num} as a control number for" \
				"drives in ${_class_name_bad} (we have" \
				"${_real_num} drives in the mirror)."
		fi

		_list_prov_used=""
		_class_ctl_num=0
		_class_real_num=0
		for _name in $list_mirrors; do
			_class_name=$(name2class $_name)
			if [ "${_class_name}" != "${_class_name_bad}" ]; then
				continue
			fi
			eval _prov_used=\${mirror_${_name}_list}
			eval _prov_num=\${mirror_${_name}_num}
			_list_prov_used="${_list_prov_used}${_list_prov_used:+ }${_prov_used}"
			_class_ctl_num=$(($_class_ctl_num + $_ctl_num))
			_class_real_num=$(($_class_real_num + $_prov_num))
		done

		_miss_num=$(($_class_ctl_num - $_class_real_num))
		_re_prov_used=$(echo "(${_list_prov_used})" | \
			sed -Ee 's/ +/|/g')
		_miss_provs=$(echo "${drv_list}" | tr ' ' '\n' | sort | \
			sed -Ee 's/$/'${_ctl_let}'/' | \
			grep -Ev "${_re_prov_used}" | head -${_miss_num})

		verbose "Missing providers are:${_miss_provs:+ }${_miss_provs}."

		for _miss_prov in $_miss_provs; do
			if [ $_handled_drvs -ge $max_drvs_for_handle ]; then
				verbose "\$max_drvs_for_handle" \
					"(${max_drvs_for_handle}) has been" \
					"reached."
				break 2
			fi
			_miss_drv=$(prov2drv $_miss_prov)
			forget_mirror $_name_bad
			if ! check_drv $_miss_drv; then
				# This drive is really bad
				free_drv $_miss_drv
				send_ticket
			else
				verbose "The drive ${_miss_drv} is OK."
				if ! $dry_run gmirror insert ${geom_opts} \
					$_name_bad $_miss_prov
				then
					err 1 "Can't insert normal provider" \
						"${_miss_prov} to mirror" \
						"${_name_bad} !"
				fi
				verbose "Provider ${_miss_prov} has been" \
					"inserted into ${_name_bad}."
			fi
			_handled_drvs=$(($_handled_drvs + 1))
		done
	done
}

init_sata_ad_drv()
{
	# This subroutine sets following global variables:
	# * new_drv

	local _ch _free_chs_list

	eval $(atacontrol list | awk -v drv_re="${drv_re}" '
		/^ATA channel/ {
			ch = $3
			sub(/:/, "", ch)
			free_chs[ch] = 1
		}
		$0 ~ drv_re {
			delete free_chs[ch]
		}
		END{
			free_chs_list = del = ""
			for (ch in free_chs) {
				free_chs_list = sprintf("%s%sata%s", \
					free_chs_list, del, ch)
				del = " "
			}
			printf "_free_chs_list=\"%s\";\n", free_chs_list
		}
	' || _free_chs_list="")

	for  _ch in $_free_chs_list; do
		# We might forget to detach a channel so we do a control detach
		$dry_run atacontrol detach $_ch 2>/dev/null || true

		eval $($dry_run atacontrol attach $_ch | awk -v drv_re="${drv_re}" '
			$2 ~ drv_re {
				printf "new_drv=\"%s\";\n", $2;
				exit
			}
		' || new_drv="")

		if [ -n "${new_drv}" ]; then
			break
		fi
	done

	if [ -z "${new_drv}" ]; then
		return 1
	fi
}

init_sata_ada_drv()
{
	# This subroutine sets following global variables:
	# * new_drv

	local _drv_in_mirr_re

	_drv_in_mirr_re="^($(echo ${drv_in_mirr_list} | sed -e 's/ /|/g'))$"
	new_drv=$(echo $drv_list | tr ' ' '\n' | grep -vE "${_drv_in_mirr_re}")

	if [ -z "${new_drv}" -o "${new_drv}" != "${new_drv%% *}" ]; then
		return 1
	fi
}

label_drv()
{
	local _drv _smpl_drv _label_f

	_drv="$1"

	if ! dd if=/dev/zero of=/dev/${_drv} bs=1m count=1 2>/dev/null; then
		err 1 "Can't clean drive ${_drv} !"
	fi

	_smpl_drv=$(echo ${drv_list} | tr ' ' '\n' | grep -vm1 "${new_drv}")
	_label_f=$(mktemp -t ${thiscmd}) || \
		err 1 "Can't create temporary file !"
	add_to_cleanup $_label_f

	if ! $dry_run bsdlabel $_smpl_drv > $_label_f; then
		err 1 "Can't get sample of bsdlabel !"
	fi

	if ! $dry_run bsdlabel -B -R $_drv $_label_f; then
		err 1 "Can't label drive ${_drv} !"
	fi
	verbose "The drive ${_drv} has been labeled like ${_smpl_drv}."
}

init_drv()
{
	case "${drv_list}" in
		ad[0-9]*) _init_cmd="init_sata_ad_drv" ;;
		ada[0-9]*) _init_cmd="init_sata_ada_drv" ;;
		*) err 1 "Unsupported drive type to init !" ;;
	esac

	if ! $_init_cmd; then
		err 1 "Can't find new drive (new_drv: ${new_drv}) !"
	fi
	verbose "New drive ${new_drv} has been initialized."

	if ! check_drv $new_drv; then
		err 1 "New drive ${new_drv} is bad (${ticket_err_str}) !"
	fi

	label_drv $new_drv
}

insert_new_drv()
{
	local _name _class_name _prov

	init_drv

	for _name in $list_mirrors; do
		_class_name=$(name2class $_name)

		eval _ctl_num=\${${scheme}_mirror_${_class_name}_num}
		eval _ctl_let=\${${scheme}_mirror_${_class_name}_let}
		eval _real_num=\${mirror_${_name}_num}
		eval _real_list=\${mirror_${_name}_list}

		if [ "${_real_num}" -eq "${_ctl_num}" ]; then
			# This mirror is OK
			continue
		fi

		_prov="${new_drv}${_ctl_let}"
		if ! $dry_run gmirror insert ${geom_opts} $_name ${_prov}
		then
			err 1 "Can't insert ${_prov} into ${_name} !"
		fi
		verbose "Provider ${_prov} has been inserted into ${_name}."
	done

}

cleanup()
{
	if [ -n "${cleanup_files}" ]; then
		rm -f ${cleanup_files} 2>/dev/null || true
	fi
}


#-- Variables ----------------------------------------------------------

thiscmd=$(basename $0)

hostname=$(hostname)
hostname_s=$(hostname -s)

default_mode="check"

drv_re="(ad|ada|da)[0-9]+"

max_drvs_for_handle=1

geom_opts=""

ticket_from="SW Raid <search-maintenance@yandex-team.ru>"
ticket_subj_tpl="[<DC>]     ${hostname_s}"
ticket_to="helpdc@yandex-team.ru"
ticket_cc="search-maintenance@yandex-team.ru, saku@yandex-team.ru"
ticket_user_agent="${thiscmd} $(echo '$Revision$' | \
	awk '{printf "%s", $2}')"
ticket_signature="SW Raid at ${hostname_s}"
ticket_encoding="koi8-r"

golem_api_url="https://golem.yandex-team.ru/api/host_query.sbml?hostname=${hostname}&columns=dc&encoding=${ticket_encoding}"

scheme_list="orange primus realspam spider walrus"

# Orange scheme
orange_mirror_root_num="4"
orange_mirror_root_let="a"
orange_mirror_swap_num="4"
orange_mirror_swap_let="b"
orange_mirror_var_num="4"
orange_mirror_var_let="d"
orange_mirror_optmirrX_num="2"
orange_mirror_optmirrX_let="f"
orange_mirror_plcmirrX_num="2"
orange_mirror_plcmirrX_let="e"

# Primus scheme
primus_mirror_root_num="2"
primus_mirror_root_let="a"
primus_mirror_swap_num="2"
primus_mirror_swap_let="b"
primus_mirror_var_num="2"
primus_mirror_var_let="d"
primus_mirror_opt_num="2"
primus_mirror_opt_let="e"

# Realspam scheme
realspam_mirror_root_num="4"
realspam_mirror_root_let="a"
realspam_mirror_swap_num="2"
realspam_mirror_swap_let="b"
realspam_mirror_var_num="4"
realspam_mirror_var_let="d"
realspam_mirror_optmirrX_num="2"
realspam_mirror_optmirrX_let="f"
realspam_mirror_placemirrX_num="2"
realspam_mirror_placemirrX_let="e"

# Spider scheme
spider_mirror_root_num="4"
spider_mirror_root_let="a"
spider_mirror_swap_num="2"
spider_mirror_swap_let="b"
spider_mirror_swapX_num="2"
spider_mirror_swapX_let="b"
spider_mirror_var_num="4"
spider_mirror_var_let="d"
spider_mirror_optmirrX_num="2"
spider_mirror_optmirrX_let="f"
spider_mirror_placemirrX_num="2"
spider_mirror_placemirrX_let="e"

# Walrus scheme
walrus_mirror_root_num="2"
walrus_mirror_root_let="a"
walrus_mirror_swap_num="2"
walrus_mirror_swap_let="b"
walrus_mirror_var_num="2"
walrus_mirror_var_let="d"
walrus_mirror_opt_num="2"
walrus_mirror_opt_let="e"
walrus_mirror_optmirrX_num="2"
walrus_mirror_optmirrX_let="e"
walrus_mirror_test_num="2"
walrus_mirror_test_let="f"
walrus_mirror_testmirrX_num="2"
walrus_mirror_testmirrX_let="f"


#-- Main ---------------------------------------------------------------

get_options $@
handle_options

verbose "Mode is: ${mode}."

trap cleanup EXIT

get_drv_list
get_scheme
get_mirrors_status

case "${mode}" in
	insert) insert_new_drv ;;
	check) handle_bad_mirrors ;;
	*) err 1 "Unsupported mode: ${mode}."
esac

