#!/bin/sh
#
# $Id$
#
# Script for monitoring WMCLoader's processes
#

me=${0##*/}     # strip path
me=${me%.*}     # strip extension
BASE=$HOME/agents
TMP=$BASE/tmp
PATH=/bin:/sbin:/usr/bin:/usr/sbin


#-- Subroutines --------------------------------------------------------

die()
{
	echo "PASSIVE-CHECK:$me;$1;$2"
	exit 0
}

inc_configs()
{
	if [ -r "${wmc_cfg}" ]; then
		. $wmc_cfg
	else
		die 2 "WMC config $wmc_cfg not found !"
	fi

	if [ -r "${cmd_cfg}" ]; then
		. ${cmd_cfg}
	fi
}

add_to_crit_msg()
{
	crit_msg="${crit_msg}${crit_msg:+ }$*"
}

check_corba_mode()
{
	local _seg _seg_pidfile _seg_pid _seg_ns_name _failed_segs
	local _loader_pids _loader_program _foreign_loaders

	if [ ! -x "${pinger_cmd}" ]; then
		die 2 "Pinger is not found."
	fi

	_loader_program=$(cat ${WMCLOADER_HOME}/program.name 2>/dev/null)

	for _seg in $loader_segments; do
		_seg_pidfile="${WMCLOADER_HOME}/${_seg}.${_loader_program}.pid"
		_seg_pid=$(cat $_seg_pidfile 2>/dev/null)
		_seg_ns_name="${WMC_NS_PATH}/${WMCLOADER_NS_PREFIX}${_seg}${WMC_NS_SUFFIX}"

		# Check segment in process list
		if ! ps -o command= -p $_seg_pid 2>/dev/null | \
			grep -q "${_loader_program}"
		then
			_failed_segs="${_failed_segs}${_failed_segs:+, }${_seg}"
			continue
		fi

		_loader_pids="${_loader_pids}${_loader_pids:+|}${_seg_pid}"

		# Check segment in CORBA
		if ! $pinger_cmd -ORBInitRef NameService=$NAME_SERVICE \
			--servant-ns-name=$_seg_ns_name >/dev/null 2>&1
		then
			_failed_segs="${_failed_segs}${_failed_segs:+, }${_seg}"
		fi
	done

	if [ -n "${_failed_segs}"  ]; then
		add_to_crit_msg "Failed segments: ${_failed_segs}."
	fi

	_foreign_loaders=$(ps -U $WMC_USER -o pid= -o command= 2>/dev/null | \
		awk 'BEGIN{ n = 0; d = "" }
		$1 !~ /^('${_loader_pids}')$/ && $2 ~ /'${_loader_program}'/ {
			if (n++ > 0) d = ","
			printf "%s%d", d, $1
		}')

	if [ -n "${_foreign_loaders}" ]; then
		add_to_crit_msg "Foreign segments' PIDs: ${_foreign_loaders}."
	fi
}

check_http_mode()
{
	local _seg _failed_segs

	if [ ! -x "${pinger_http_cmd}" ]; then
		die 2 "HTTP Pinger is not found."
	fi

	for _seg in $loader_segments; do
		# Check segment by HTTP
		if ! $pinger_http_cmd --loader-name=$WMCLOADER_HTTP_LINK \
			--segment=$_seg >/dev/null 2>&1
		then
			_failed_segs="${_failed_segs}${_failed_segs:+, }${_seg}"
		fi
	done

	if [ -n "${_failed_segs}"  ]; then
		add_to_crit_msg "Failed segments: ${_failed_segs}."
	fi
}

check_linkhelper()
{
	local _linkhelper_pidfile _linkhelper_pid

	_linkhelper_pidfile="${WMCLOADER_HOME}/${linkhelper_program}.pid"
	_linkhelper_pid=$(cat $_linkhelper_pidfile 2>/dev/null)

	if ps -o command= -p $_linkhelper_pid 2>/dev/null | \
		grep -q "${linkhelper_program}"
	then
		if [ -z "${loader_segments}" ]; then
			add_to_crit_msg "${linkhelper_program} is not stopped."
		fi
	else
		if [ -n "${loader_segments}" ]; then
			add_to_crit_msg "${linkhelper_program} is not running."
		fi
	fi
}

report()
{
	if [ -n "${crit_msg}" ]; then
		die 2 "${crit_msg}"
	elif [ -z "${loader_segments}" ]; then
		die 0 "Not used."
	else
		die 0 "Ok"
	fi	
}


#-- Variables ----------------------------------------------------------

cmd_cfg="${BASE}/etc/${me}.conf"
wmc_cfg="/place/wmconsole/wmconsole.cfg"

pinger_cmd="${BASE}/pinger"
pinger_http_cmd="${BASE}/pinger_http"

linkhelper_program="WMCLinkHelper"

scheme="all"	# (corba|http|all)


#-- Main ---------------------------------------------------------------

inc_configs

loader_segments=$(${HTARCLOOKUP} -r 2>/dev/null)

crit_msg=""

case "${scheme}" in
	corba|http)	check_${scheme}_mode ;;
	all)		check_corba_mode; check_http_mode ;;
	*)		die 2 "Unknown scheme !" ;;
esac

check_linkhelper

report

