#!/bin/sh -e
#
# Script for automatic creation of MySQL slaves for WMC
#
# $Id$


PATH=/bin:/usr/bin:/usr/local/bin
export PATH

#-- Subroutines --------------------------------------------------------

err()
{
	local _exitval

	_exitval=$1
	shift

	echo 1>&2 "ERROR: $*"
	exit $_exitval
}

log()
{
	local _date

	_date=$(date "+%F %H:%M:%S")
	echo "[${_date}] $*"
}

usage()
{
	echo 1>&2 "Usage: ${thiscmd} -h <src_host> -s <src_db> [option]..."
	echo 1>&2 "Options are:"
	echo 1>&2 "  -p <port>           master port (default:" \
					"${default_src_port})"
	echo 1>&2 "  -i <dst_instance>   slave instance (default:" \
					"${default_dst_instance})"
	echo 1>&2 "  -d <dst_db>         slave database"
	echo 1>&2 "  -c                  drop database on slave" \
					"before copying"
	echo 1>&2 "  -a                  auto start for slave after" \
					"copying"
	exit 1
}

get_opts()
{
	local _opt

	while getopts "h:p:s:i:d:ca" _opt; do
		case "$_opt" in
			h) src_host="$OPTARG" ;;
			p) src_port="$OPTARG" ;;
			s) src_db="$OPTARG" ;;
			i) dst_instance="$OPTARG" ;;
			d) dst_db="$OPTARG" ;;
			c) clean=1 ;;
			a) auto_ch_m=1 ;;
			*) usage ;;
		esac
	done

	shift $(($OPTIND - 1))

	if [ $# -ne 0 ]; then
		usage
	fi
}

handle_opts()
{
	if [ -z "${src_host}" ]; then
		err 1 "Source host not defined !"
	fi

	: ${src_port:="$default_src_port"}

	if [ -z "${src_db}" ]; then
		err 1 "Source db not defined !"
	fi

	: ${dst_instance:="$default_dst_instance"}

	mysql_rc_script="${local_startup}/${dst_instance}"
	if [ ! -x $mysql_rc_script ]; then
		err 1 "Executable RC script for instance" \
			"${dst_instance} (${mysql_rc_script})" \
			"not found !"
	fi

	mysql_datadir=$(${mysql_rc_script} get_datadir)
	if [ ! -d $mysql_datadir ]; then
		err 1 "Datadir for instance ${dst_instance}" \
			"(${mysql_datadir}) not found !"
	fi

	mysql_socket=$(${mysql_rc_script} get_socket)
	if [ ! -S $mysql_socket ]; then
		err 1 "Socket for instance ${dst_instance}" \
			"(${mysql_socket}) not found !"
	fi

	mysql_defaults_file=$(${mysql_rc_script} get_defaults_file)
	if [ ! -r $mysql_defaults_file ]; then
		err 1 "Defaults file for instance ${dst_instance}" \
			"(${mysql_defaults_file}) not found !"
	fi

	: ${mysqldump_defaults_file:="$mysql_defaults_file"}

	: ${dst_db:="$src_db"}

	mysql_common_opts="--defaults-file=${mysql_defaults_file} \
		--socket=${mysql_socket}"

	mysql_run="mysql ${mysql_common_opts} ${mysql_run_opts}"
	mysql_run_input="mysql ${mysql_common_opts} ${dst_db}"

	mysqldump_logname="${dst_instance}_cp_$(date +%s)_err.log"
	mysqldump_log="${mysqldump_logdir}/${mysqldump_logname}"

	mysqldump_common_opts="--defaults-file=${mysqldump_defaults_file} \
		--defaults-group-suffix=${mysqldump_def_grp_sfx}\
		-h ${src_host} -P ${src_port} --log-error=${mysqldump_log}"

	mysqldump_run="mysqldump ${mysqldump_common_opts} \
		${mysqldump_run_opts} ${src_db}"
}

get_master_info()
{
	local _mode _pre_vars _post_vars _vars
	local _master_info _master_info_bak _master_var _master_val

	if [ -z "${auto_ch_m}" ]; then
		return
	fi

	case "$1" in
		pre|post) _mode="$1" ;;
		*) err 1 "Unknown mode for get_master_info !"
	esac

	_master_info="${mysql_datadir}/${master_info_file}"
	_master_info_bak="${_master_info}.bak"

	if [ ! -f "${_master_info}" ]; then
		err 1 "Master file ${_master_info} is not found !"
	fi

	eval $(awk -v mode="${_mode}" '
		# See for details:
		# http://dev.mysql.com/doc/refman/5.1/en/slave-logs-status.html

		NR == 2 && mode == "post" {
			param[NR] = "master_log_file"
			val[NR] = $1
		}
		NR == 3 && mode == "post" {
			param[NR] = "master_log_pos"
			val[NR] = $1
		}
		NR == 4 && mode == "pre" {
			param[NR] = "master_host"
			val[NR] = $1
		}
		NR == 5 && mode == "pre" {
			param[NR] = "master_user"
			val[NR] = $1
		}
		NR == 6 && mode == "pre" {
			param[NR] = "master_password"
			val[NR] = $1
		}
		NR == 7 && mode == "pre" {
			param[NR] = "master_port"
			val[NR] = $1
		}

		END{
			for (n in param) {
				printf "%s=\"%s\";\n", param[n], val[n]
			}
		}
		' $_master_info)

	_pre_vars="master_host master_port master_user master_password"
	_post_vars="master_log_file master_log_pos"

	eval _vars=\$_${_mode}_vars

	for _master_var in $_vars
	do
		eval _master_val="\$${_master_var}"
		if [ -z "${_master_val}" ]; then
			err 1 "Can't get value of \${${_master_var}} !"
		fi
		log "Mode \"${_mode}\" -> $(echo ${_master_var} | \
			tr '[:lower:]' '[:upper:]') = ${_master_val}."
	done

	if [ "${_mode}" = "pre" ]; then
		if [ "${master_host}" != "${src_host}" -o \
			"${master_port}" != "${src_port}" ]
		then
			err 1 "Values from ${_master_info} are different from" \
				"${src_host}:${src_port}. Perhaps processing" \
				"of ${_master_info} is not correct !"
		fi
		if ! cp -a $_master_info $_master_info_bak; then
			err 1 "Can't backup ${_master_info} to" \
				"${_master_info_bak} !"
		fi
	fi
}

reset_slave()
{
	if ! $mysql_run "STOP SLAVE; RESET SLAVE;"; then
		err 1 "Can't stop & reset slave !"
	fi
	log "Slave is stopped."
}

clean_old_db()
{
	if [ -z "${clean}" ]; then
		return
	fi

	if ! $mysql_run "DROP DATABASE ${dst_db}"; then
		err 1 "Can't drop old database ${dst_db} !"
	fi
	log "Database ${dst_db} is droped."
}

install_db()
{
	if ! $mysql_run "SHOW DATABASES LIKE '${dst_db}'" 2>/dev/null | \
		grep -qw "${dst_db}"
	then
		if ! $mysql_run "CREATE DATABASE ${dst_db}"; then
			err 1 "Can't create destination database" \
				"${dst_db} !"
		fi
		log "Database ${dst_db} is created."
	fi
		
}

cp_db()
{
	log "Begin copy mysql://${src_host}:${src_port}/${src_db} to" \
		"local ${dst_db}."

	if ! $mysqldump_run | $mysql_run_input; then
		err 1 "Can't copy ${src_db} from" \
			"${src_host}:${src_port} to local ${dst_db} !"
	fi

	if [ -s $mysqldump_log ]; then
		err 1 "See $mysqldump_log for errors !"
	else
		rm -f $mysqldump_log 2>/dev/null
	fi

	log "End copy mysql://${src_host}:${src_port}/${src_db} to" \
		"local ${dst_db}."
}

change_master()
{
	if ! $mysql_run "CHANGE MASTER TO MASTER_HOST='${master_host}',
		MASTER_PORT=${master_port},
		MASTER_USER='${master_user}',
		MASTER_PASSWORD='${master_password}',
		MASTER_LOG_FILE='${master_log_file}',
		MASTER_LOG_POS=${master_log_pos};"
	then
		err 1 "Can't change master !"
	fi
	log "New slave is prepared."
}

start_slave()
{
	if ! $mysql_run "START SLAVE"; then
		err 1 "Can't start slave !"
	fi
	log "Slave is started."
}


#-- Variables ----------------------------------------------------------

thiscmd=$(basename $0)

local_startup="/usr/local/etc/rc.d"

default_src_port=3306
default_dst_instance="mysql"

mysql_run_opts="-NBAe"
mysqldump_run_opts="--single-transaction --master-data --quick"

mysqldump_def_grp_sfx="_btw_group"
mysqldump_logdir="/var/tmp"

master_info_file="master.info"


#-- Main ---------------------------------------------------------------

get_opts $@
handle_opts

get_master_info pre
reset_slave

clean_old_db
install_db
cp_db

get_master_info post
change_master
start_slave

